package ru.yandex.ace.ventura.proxy.list;

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.ace.ventura.proxy.common.AceVenturaContact;
import ru.yandex.ace.ventura.proxy.common.AceVenturaEmail;
import ru.yandex.ace.ventura.proxy.common.SharedScope;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.search.rules.pure.SearchRule;

public class ListContactsShareAndPersonalRule<T extends ListContactsContext>
    implements SearchRule<T, List<AceVenturaContact>>
{
    private final SearchRule<T, List<AceVenturaContact>> personal;
    private final SearchRule<T, List<List<AceVenturaContact>>> shared;

    public ListContactsShareAndPersonalRule(
        final SearchRule<T, List<AceVenturaContact>> personal,
        final SearchRule<T, List<List<AceVenturaContact>>> shared)
    {
        this.personal = personal;
        this.shared = shared;
    }

    @Override
    public void execute(
        final T input,
        final FutureCallback<? super List<AceVenturaContact>> callback)
        throws HttpException
    {
        if (input.shareScope() == SharedScope.EXCLUDE) {
            personal.execute(input, callback);
        } else if (input.shareScope() == SharedScope.INCLUDE) {
            personal.execute(input, new PersonalCallback(input, callback));
        } else if (input.shareScope() == SharedScope.ONLY) {
            shared.execute(
                input,
                new MergeCallback(input, Collections.emptyList(), callback));
        } else {
            callback.failed(new Exception("Unsupported shared scope"));
        }
    }

    private final class PersonalCallback
        extends AbstractFilterFutureCallback<List<AceVenturaContact>, List<AceVenturaContact>>
    {
        private final T input;

        public PersonalCallback(
            final T input,
            final FutureCallback<? super List<AceVenturaContact>> callback)
        {
            super(callback);
            this.input = input;
        }

        @Override
        public void completed(final List<AceVenturaContact> contacts) {
            input.logger().info(
                "Personal, Found " + contacts.size() + " requested "
                    + input.totalLength() + " scope " + input.shareScope());

            if (contacts.size() < input.totalLength()
                && input.shareScope() == SharedScope.INCLUDE)
            {
                input.logger().info("Not enough, looking in shared");
                try {
                    shared.execute(input, new MergeCallback(input, contacts, callback));
                } catch (HttpException e) {
                    callback.failed(e);
                }
            } else {
                callback.completed(contacts);
            }
        }
    }

    private class MergeCallback
        extends AbstractFilterFutureCallback<List<List<AceVenturaContact>>, List<AceVenturaContact>>
    {
        private final T input;
        private final List<AceVenturaContact> personal;

        public MergeCallback(
            final T input,
            final List<AceVenturaContact> personal,
            final FutureCallback<? super List<AceVenturaContact>> callback)
        {
            super(callback);

            this.personal = personal;
            this.input = input;
        }

        private void add(
            final LinkedHashMap<Set<String>, AceVenturaContact> set,
            final List<AceVenturaContact> collection)
        {
            for (AceVenturaContact contact: collection) {
                Set<String> emailsKeys = new LinkedHashSet<>(contact.emails().size());
                for (AceVenturaEmail email: contact.emails()) {
                    emailsKeys.add(email.email());
                }

                AceVenturaContact current = set.get(emailsKeys);
                if (current != null) {
                    continue;
                }

                set.put(emailsKeys, contact);
            }
        }

        @Override
        public void completed(
            final List<List<AceVenturaContact>> shared)
        {
            LinkedHashMap<Set<String>, AceVenturaContact> emails
                = new LinkedHashMap<>(input.totalLength());

            int sharedTotal = 0;
            add(emails, personal);
            for (List<AceVenturaContact> list: shared) {
                add(emails, list);
                sharedTotal += list.size();
            }

            input.logger().info("Shared found " + sharedTotal);

            List<AceVenturaContact> result = new ArrayList<>(emails.values());

            callback.completed(result);
        }
    }
}
