package ru.yandex.ace.ventura.proxy.search;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.ace.ventura.AceVenturaFields;
import ru.yandex.ace.ventura.AceVenturaRecordType;
import ru.yandex.ace.ventura.proxy.AceVenturaProxy;
import ru.yandex.ace.ventura.proxy.common.AceVenturaContact;
import ru.yandex.ace.ventura.proxy.common.AceVenturaEmail;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.request.util.SearchRequestText;
import ru.yandex.search.rules.pure.SearchRule;
import ru.yandex.search.rules.pure.providers.RequestProvider;

public class AceVenturaSearchRule<T extends RequestProvider & AceVenturaSearchContextProvider>
    implements SearchRule<T, List<AceVenturaContact>>
{
    private static final String OR = " OR ";
    private static final String AND = " AND ";

    private static final String LEFT_JOIN_DP =
        "left_join(" + AceVenturaFields.EMAIL_CID.stored()
            + ',' + AceVenturaFields.CID.prefixed()
            // get/out fields
            + ",," + AceVenturaFields.VCARD.stored()
            + ',' + AceVenturaFields.EN_NAMES.stored()
            + ',' + AceVenturaFields.LIST_ID.stored() + ')';

    private final AceVenturaProxy proxy;

    public AceVenturaSearchRule(final AceVenturaProxy proxy) {
        this.proxy = proxy;
    }

    public AceVenturaProxy proxy() {
        return proxy;
    }

    @Override
    public void execute(
        final T input,
        final FutureCallback<? super List<AceVenturaContact>> callback)
        throws HttpException
    {
        AceVenturaSearchContext context = input.searchContext();

        StringBuilder textSb = new StringBuilder();

        if (input.request().contains("@")
            && context.searchBy().contains(AceVenturaFields.EMAIL.prefixed()))
        {
            textSb.append(AceVenturaFields.RECORD_TYPE.prefixed());
            textSb.append(':');
            textSb.append(AceVenturaRecordType.EMAIL.fieldValue());
            textSb.append(AND);
            textSb.append(AceVenturaFields.EMAIL.prefixed());
            textSb.append(':');
            textSb.append(SearchRequestText.fullEscape(input.request(), true));
            textSb.append('*');
        } else {
            textSb.append(AceVenturaFields.RECORD_TYPE.prefixed());
            textSb.append(":(");
            textSb.append(AceVenturaRecordType.EMAIL.fieldValue());
            textSb.append(OR);
            textSb.append(AceVenturaRecordType.CONTACT.fieldValue());
            textSb.append(")");
            SearchRequestText request = SearchRequestText.parseSuggest(
                input.request(),
                context.locale());

            if (request.hasWords()) {
                textSb.append(AND);
                textSb.append('(');
                request.fieldsQuery(textSb, context.searchBy());
                textSb.append(')');
            }
        }

        if (context.tagId() != null) {
            textSb.append(AND);
            textSb.append(AceVenturaFields.TAGS.prefixed());
            textSb.append(':');
            textSb.append(context.tagId());
        }

        if (!context.listIds().isEmpty()) {
            textSb.append(AND);
            textSb.append(AceVenturaFields.LIST_ID.prefixed());
            textSb.append(":(");
            for (Long listId: context.listIds()) {
                textSb.append(listId);
                textSb.append(" ");
            }

            textSb.setLength(textSb.length() - 1);
            textSb.append(')');
        }

        QueryConstructor query =
            new QueryConstructor(
                "/search-ace-search?IO_PRIO=0&json-type=dollar");

        query.append("service", context.user().service());
        query.append("text", new String(textSb));
        query.append("get", "*");
        query.append("dp", LEFT_JOIN_DP);
        query.append("length", context.length() * 2);
        query.append("prefix", context.user().prefix().toStringFast());
        query.append("lowercase-expanded-terms", "true");

        if (context.sortBy().size() == 1) {
            query.append("sort", context.sortBy().iterator().next());
        }
        // without this option we will loose vcard in response
        // for av_record_type: contact
        query.append("dp-read-out-fields", "true");
        query.copyIfPresent(context.session().params(), "debug");

        long failover = proxy.config().search().failoverDelay();
        boolean localityShuffle = proxy.config().search().localityShuffle();

        if (failover > 0) {
            context.proxy().sequentialRequest(
                    context.session(),
                    context,
                    new BasicAsyncRequestProducerGenerator(query.toString()),
                    failover,
                    localityShuffle,
                    JsonAsyncTypesafeDomConsumerFactory.OK,
                    context.contextGenerator(),
                    new SearchCallback(callback, context));
        } else {
            context.proxy().parallelRequest(
                    context.session(),
                    context,
                    new BasicAsyncRequestProducerGenerator(query.toString()),
                    JsonAsyncTypesafeDomConsumerFactory.OK,
                    context.contextGenerator(),
                    new SearchCallback(callback, context));
        }
    }


    private static final class SearchCallback
        extends AbstractFilterFutureCallback<JsonObject, List<AceVenturaContact>>
    {
        private final AceVenturaSearchContext context;

        private SearchCallback(
            final FutureCallback<? super List<AceVenturaContact>> callback,
            final AceVenturaSearchContext context)
        {
            super(callback);
            this.context = context;
        }

        @Override
        public void completed(final JsonObject response) {
            try {
                JsonMap map = response.asMap();
                JsonList hits = map.getList("hitsArray");
                if (context.debug()) {
                    context.logger().info(
                        "LuceneResponse "
                            + JsonType.NORMAL.toString(response));
                }

                Map<Long, AceVenturaContact> contactMap = new LinkedHashMap<>();
                for (JsonObject item: hits) {
                    JsonMap itemMap = item.asMap();

                    AceVenturaRecordType recordType = itemMap.getEnum(
                        AceVenturaRecordType.class,
                        AceVenturaFields.RECORD_TYPE.stored());
                    AceVenturaContact contact;
                    if (recordType == AceVenturaRecordType.CONTACT) {
                        Long cid =
                            itemMap.getLong(AceVenturaFields.CID.stored());

                        contact = AceVenturaContact.fromLuceneResponse(
                            context.session().logger(),
                            context.prefix(),
                            context.english(),
                            context.fields(),
                            itemMap);
                        AceVenturaContact curContact = contactMap.get(cid);
                        if (curContact != null) {
                            contact.addAllEmails(curContact.emails());
                        }

                        contactMap.put(cid, contact);
                    } else {
                        Long cid =
                            itemMap.getLong(AceVenturaFields.EMAIL_CID.stored());

                        contact = contactMap.get(cid);
                        if (contact == null) {
                            contact = AceVenturaContact.fromLuceneResponse(
                                context.session().logger(),
                                context.prefix(),
                                context.english(),
                                context.fields(),
                                itemMap);
                            contactMap.put(cid, contact);
                        }

                        AceVenturaEmail email =
                            AceVenturaEmail.fromLuceneResponse(itemMap);
                        contact.addEmail(email);
                    }
                }

                callback.completed(new ArrayList<>(contactMap.values()));
            } catch (JsonException je) {
                failed(je);
            }
        }
    }
}
