package ru.yandex.ace.ventura.proxy.search;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.Locale;
import java.util.Set;

import ru.yandex.ace.ventura.AceVenturaFields;
import ru.yandex.ace.ventura.proxy.AbstractAceVenturaProxyContext;
import ru.yandex.ace.ventura.proxy.AceVenturaProxy;
import ru.yandex.ace.ventura.proxy.AceVenturaProxyParams;
import ru.yandex.ace.ventura.proxy.fetch.SearchTransform;
import ru.yandex.function.GenericFunction;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.search.rules.pure.providers.RequestProvider;

public class BasicAceVenturaSearchContext
    extends AbstractAceVenturaProxyContext
    implements AceVenturaSearchContext,
    RequestProvider,
    AceVenturaSearchContextProvider
{
    private static final int DEFAULT_LIMIT = 30;
    private static final CollectionParser<String, Set<String>, Exception>
        SET_PARSER =
        new CollectionParser<>(NonEmptyValidator.TRIMMED, LinkedHashSet::new);
    private static final CollectionParser<String, Set<String>, Exception>
        FIELDS_SET_PARSER =
        new CollectionParser<>(new FieldsParser(), LinkedHashSet::new);
    private static final Set<String> DEFAULT_SEARCH_BY_WITHOUT_AT =
        Collections.unmodifiableSet(
            new LinkedHashSet<>(
                Arrays.asList(
                    AceVenturaFields.NAMES.prefixed(),
                    AceVenturaFields.EN_NAMES.prefixed(),
                    AceVenturaFields.LOGIN_LETTER.prefixed(),
                    AceVenturaFields.DOMAIN_NT.prefixed())));

    private static final Set<String> DEFAULT_SEARCH_BY_WITH_AT =
        Collections.unmodifiableSet(
            new LinkedHashSet<>(
                Arrays.asList(
                    AceVenturaFields.NAMES.prefixed(),
                    AceVenturaFields.EN_NAMES.prefixed(),
                    AceVenturaFields.EMAIL.prefixed())));

    private final int length;
    private final int offset;
    private final String request;

    private final Set<SearchTransform> transforms;
    private final Set<String> fields;
    private final Set<String> searchBy;
    private final Set<String> sortBy;
    private final Long tagId;
    private volatile boolean forceEnglish = false;

    public BasicAceVenturaSearchContext(
        final AceVenturaProxy proxy,
        final ProxySession session)
        throws BadRequestException
    {
        super(proxy, session);

        CgiParams params = session.params();

        String request = params.getString(AceVenturaProxyParams.QUERY, null);
        if (request == null) {
            request = params.getString(AceVenturaProxyParams.REQUEST, "");
        }

        this.request = request;

        fields = params.getAll(
            AceVenturaProxyParams.GET_FIELDS,
            null,
            SET_PARSER);

        Set<String> searchBy = params.get(
            AceVenturaProxyParams.SEARCH_BY,
            null,
            FIELDS_SET_PARSER);
        boolean hasAt = request.contains("@");
        if (searchBy == null) {
            if (hasAt) {
                searchBy = new LinkedHashSet<>(DEFAULT_SEARCH_BY_WITH_AT);
            } else {
                searchBy = new LinkedHashSet<>(DEFAULT_SEARCH_BY_WITHOUT_AT);
            }
        }

        transforms = params.getAll(
            AceVenturaProxyParams.TRANSFORM,
            Collections.emptySet(),
            SearchTransform.SET_PARSER);

        if (transforms.contains(SearchTransform.HUMANNAMES)
            && searchBy.contains(AceVenturaFields.NAMES.prefixed()))
        {
            searchBy.add(AceVenturaFields.NAMES_ALIAS.prefixed());
        }

        this.searchBy = searchBy;

        tagId =
            params.getLong(AceVenturaProxyParams.TAG_FILTER, null);

        sortBy = params.getAll(
            AceVenturaProxyParams.SORT_FIELDS,
            Collections.emptySet(),
            SET_PARSER);

        Integer limit = params.getInt(AceVenturaProxyParams.PAGE_SIZE, null);
        if (limit == null) {
            limit = params.getInt(AceVenturaProxyParams.LIMIT, DEFAULT_LIMIT);
        }

        this.length = limit;

        Integer offset = params.getInt(AceVenturaProxyParams.OFFSET, null);
        if (offset == null) {
            offset =
                length * params.getInt(AceVenturaProxyParams.PAGE, 0);
        }

        this.offset = offset;
    }

    @Override
    public BasicAceVenturaSearchContext searchContext() {
        return this;
    }

    @Override
    public int length() {
        return length;
    }

    @Override
    public String request() {
        return request;
    }

    @Override
    public int offset() {
        return offset;
    }

    @Override
    public Set<SearchTransform> transforms() {
        return transforms;
    }

    @Override
    public Set<String> fields() {
        return fields;
    }

    @Override
    public Set<String> searchBy() {
        return searchBy;
    }

    @Override
    public Set<String> sortBy() {
        return sortBy;
    }

    @Override
    public Long tagId() {
        return tagId;
    }

    @Override
    public Long failOverDelay() {
        return 300L;
    }

    @Override
    public Collection<Long> listIds() {
        return Collections.emptySet();
    }

    private static class FieldsParser
        implements GenericFunction<String, String, Exception>
    {
        @Override
        public String apply(final String s) throws Exception {
            switch (s.trim().toLowerCase(Locale.ENGLISH)) {
                case "names":
                    return AceVenturaFields.NAMES.prefixed();
                case "telephone_numbers":
                    return AceVenturaFields.PHONES.prefixed();
                case "emails":
                    return AceVenturaFields.EMAIL.prefixed();
                default:
                    throw new IllegalArgumentException("Unknown field " + s);
            }
        }
    }

    public void forceEnglish() {
        forceEnglish = true;
    }

    @Override
    public boolean english() {
        if (forceEnglish) {
            return true;
        }

        return corp() && Locale.ENGLISH.equals(locale());
    }
}

