package ru.yandex.ace.ventura.proxy.suggest;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.ace.ventura.AceVenturaFields;
import ru.yandex.ace.ventura.AceVenturaRecordType;
import ru.yandex.ace.ventura.proxy.AceVenturaProxy;
import ru.yandex.ace.ventura.proxy.common.AceVenturaContact;
import ru.yandex.ace.ventura.proxy.common.AceVenturaEmail;
import ru.yandex.ace.ventura.proxy.common.SharedScope;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.request.util.SearchRequestText;
import ru.yandex.search.rules.pure.SearchRule;
import ru.yandex.search.rules.pure.TranslitRule;
import ru.yandex.search.rules.pure.providers.RequestProvider;

public class AceVenturaEmailSuggestRule<T extends RequestProvider & AceVenturaSuggestContextProvider>
    implements SearchRule<T, List<AceVenturaContact>>
{
    protected static final String GET_FIELDS =
        AceVenturaFields.TAGS.stored() + ',' +
            AceVenturaFields.REVISION.stored() + ',' +
            AceVenturaFields.VCARD.stored() + ',' +
            AceVenturaFields.LIST_ID.stored() + ',' +
            AceVenturaFields.CID.stored() + ',' +
            AceVenturaFields.EMAIL_CID.stored() + ',' +
            AceVenturaFields.EMAIL.stored() + ',' +
            AceVenturaFields.EMAIL_ID.stored() + ',' +
            AceVenturaFields.EN_NAMES.stored() + ',' +
            AceVenturaFields.LAST_USAGE.stored();

    protected static final int EXCLUDE_MAX_SIZE = 15;

    protected static final String LEFT_JOIN_DP =
        "left_join(" + AceVenturaFields.EMAIL_CID.stored()
            + ',' + AceVenturaFields.CID.prefixed()
            // get/out fields
            + ",," + AceVenturaFields.VCARD.stored()
            + ',' + AceVenturaFields.EN_NAMES.stored()
            + ',' + AceVenturaFields.TAGS.stored()
            + ',' + AceVenturaFields.REVISION.stored()
            + ',' + AceVenturaFields.LIST_ID.stored() + ')';
    protected static final String OR = " OR ";
    protected static final String AND = " AND ";
    protected static final Collection<String> SEARCH_FIELDS =
        Arrays.asList(
            AceVenturaFields.EMAIL.prefixed(),
            AceVenturaFields.LOGIN_LETTER.prefixed(),
            AceVenturaFields.LOGINS_LETTER.prefixed(),
            AceVenturaFields.DOMAIN_NT.prefixed(),
            AceVenturaFields.NAMES.prefixed(),
            AceVenturaFields.NAMES_ALIAS.prefixed());
    protected static final Collection<String> EN_SEARCH_FIELDS =
        Arrays.asList(
            AceVenturaFields.EMAIL.prefixed(),
            AceVenturaFields.LOGIN_LETTER.prefixed(),
            AceVenturaFields.LOGINS_LETTER.prefixed(),
            AceVenturaFields.DOMAIN_NT.prefixed(),
            AceVenturaFields.NAMES.prefixed(),
            AceVenturaFields.EN_NAMES.prefixed(),
            AceVenturaFields.NAMES_ALIAS.prefixed());

    private final AceVenturaProxy proxy;

    protected AceVenturaEmailSuggestRule(final AceVenturaProxy proxy) {
        this.proxy = proxy;
    }

    public AceVenturaProxy proxy() {
        return proxy;
    }

    protected Collection<String> searchFields(final AceVenturaSuggestContext context) {
        if (context.english()) {
            return EN_SEARCH_FIELDS;
        }

        return SEARCH_FIELDS;
    }

    public StringBuilder prepareQuery(
        final AceVenturaSuggestContext context,
        final Collection<String> requests,
        final String original)
    {
        // expecting non zero size requests
        StringBuilder queryText = new StringBuilder();

        queryText.append(AceVenturaFields.RECORD_TYPE.prefixed());
        queryText.append(':');
        queryText.append(AceVenturaRecordType.EMAIL.fieldValue());

        if (!context.listIdFilter().isEmpty()) {
            queryText.append(AND);
            queryText.append(AceVenturaFields.LIST_ID.field());
            queryText.append(":(");
            for (Long listId: context.listIdFilter()) {
                queryText.append(listId);
                queryText.append(" ");
            }

            queryText.setLength(queryText.length() - 1);
            queryText.append(')');
        }

        if (original.contains("@")) {
            queryText.append(AND);
            queryText.append(AceVenturaFields.EMAIL.field());
            queryText.append(':');
            if (requests.size() == 1) {
                String request = requests.iterator().next();
                queryText.append(
                    SearchRequestText.fullEscape(
                        request,
                        true));
                queryText.append('*');
            } else {
                queryText.append('(');
                for (String request: requests) {
                    queryText.append(
                        SearchRequestText.fullEscape(
                            request,
                            true));
                    queryText.append('*');
                    queryText.append(OR);
                }

                queryText.setLength(queryText.length() - OR.length());
                queryText.append(')');
            }
        } else {
            queryText.append(AND);
            boolean added = false;
            queryText.append('(');
            for (String request: requests) {
                SearchRequestText requestText = SearchRequestText.parseSuggest(
                    request,
                    context.locale());

                if (requestText.hasWords()) {
                    queryText.append('(');
                    requestText.fieldsQuery(queryText, searchFields(context));
                    queryText.append(')');
                    queryText.append(OR);
                    added = true;
                }
            }
            if (!added) {
                queryText.setLength(queryText.length() - AND.length() - 1);
            } else {
                queryText.setLength(queryText.length() - OR.length());
                queryText.append(')');
            }
        }

        if (context.onlyIfhasPhone()) {
            queryText.append(AND);
            queryText.append(AceVenturaFields.HAS_PHONES.field());
            queryText.append(":true");
        }

        if (context.excludeEmails().size() > 0
            && context.excludeEmails().size() < EXCLUDE_MAX_SIZE)
        {
            queryText.append(" AND NOT ");
            queryText.append(AceVenturaFields.EMAIL.field());
            queryText.append(":(");
            for (String exclude: context.excludeEmails()) {
                queryText.append(
                    SearchRequestText.fullEscape(exclude, false));
                queryText.append(OR);
            }
            queryText.setLength(queryText.length() - OR.length());
            queryText.append(')');
        }

        return queryText;
    }

    protected QueryConstructor createBaseQuery(
        final T contextProvider)
        throws HttpException
    {
        AceVenturaSuggestContext context = contextProvider.searchContext();
        QueryConstructor query =
            new QueryConstructor(
                "/search-ace-suggest?IO_PRIO=0&json-type=dollar");

        query.append("service", context.user().service());
        query.append("dp", LEFT_JOIN_DP);
        query.append("get", context.getFields());
        query.append("length", 2 * context.length());
        query.append("prefix", context.user().prefix().toStringFast());
        query.append("lowercase-expanded-terms", "true");
        query.append("replace-ee-expanded-terms", "true");

        if (context.sortField() == null) {
            query.append("collector", "passthru(" + 2 * context.length() + ')');
        } else {
            query.append("sort", context.sortField().stored());
        }

        return query;
    }

    @Override
    public void execute(
        final T input,
        final FutureCallback<? super List<AceVenturaContact>> callback)
        throws HttpException
    {
        AceVenturaSuggestContext context = input.searchContext();

        if (context.shareScope() == SharedScope.ONLY) {
            callback.completed(Collections.emptyList());
            return;
        }

        String originalQuery = prepareQuery(context,
            Collections.singleton(input.request()),
            input.request()).toString();

        QueryConstructor query = createBaseQuery(input);

        context.session().logger().info(
            "Contact suggest rule executed " + query.toString() + " "
                + context.getClass().getSimpleName());

        boolean translit = input.request().length() > 1
            && context.session().params().getBoolean("translit", true);
        boolean waitTranslit = context.session().params().getBoolean("wait-translit", false);

        Aggregator aggregator =
            new Aggregator(context, callback, translit && waitTranslit);

        int sbPos = query.sb().length();
        query.append("text", originalQuery);

        long failoverDelay = proxy.config().suggest().failoverDelay();
        boolean localityShuffle = proxy.config().suggest().localityShuffle();
        if (input.request().length() == 0) {
            failoverDelay = proxy.config().suggest().zeroFailoverDelay();
            localityShuffle = proxy.config().suggest().zeroLocalityShuffle();
        }

        if (failoverDelay > 0) {
            context.proxy().sequentialRequest(
                    context.session(),
                    context,
                    new BasicAsyncRequestProducerGenerator(query.toString()),
                    failoverDelay,
                    localityShuffle,
                    JsonAsyncTypesafeDomConsumerFactory.OK,
                    context.contextGenerator(),
                    new EmailSuggestCallback(aggregator, context, false));
        } else {
            context.proxy().parallelRequest(
                    context.session(),
                    context,
                    new BasicAsyncRequestProducerGenerator(query.toString()),
                    JsonAsyncTypesafeDomConsumerFactory.OK,
                    context.contextGenerator(),
                    new EmailSuggestCallback(aggregator, context, false));
        }

        if (translit) {
            TranslitRule.Context context1 =
                new TranslitRule.Context(input.request());
            Set<String> requests = new LinkedHashSet<>(
                TranslitRule.TABLES.size() << 1);
            for (TranslitRule.Table table: TranslitRule.TABLES) {
                requests.add(table.translate(context1));
            }

            requests.remove(input.request());
            if (requests.size() == 0) {
                aggregator.completeTranslit(Collections.emptyList());
            } else {
                String translitRequest =
                    prepareQuery(context, requests, input.request()).toString();

                query.sb().setLength(sbPos);
                query.append("text", translitRequest);
                context.proxy().sequentialRequest(
                        context.session(),
                        context,
                        new BasicAsyncRequestProducerGenerator(query.toString()),
                        proxy.config().suggest().translitFailOverDelay(),
                        proxy.config().suggest().translitLocalityShuffle(),
                        JsonAsyncTypesafeDomConsumerFactory.OK,
                        context.contextGenerator(),
                        new EmailSuggestCallback(aggregator, context, true));
            }
        } else {
            aggregator.completeTranslit(Collections.emptyList());
        }
    }

    private static final class Aggregator {
        private final AceVenturaSuggestContext context;
        private final FutureCallback<? super List<AceVenturaContact>> callback;
        private final boolean alwaysWaitTranslit;
        private List<AceVenturaContact> translit = null;
        private List<AceVenturaContact> original = null;
        private boolean done = false;

        public Aggregator(
            final AceVenturaSuggestContext context,
            final FutureCallback<? super List<AceVenturaContact>> callback,
            final boolean alwaysWaitTranslit)
        {
            this.context = context;
            this.callback = callback;
            this.alwaysWaitTranslit = alwaysWaitTranslit;
        }

        List<AceVenturaContact> deduplicate(
            final List<AceVenturaContact> first,
            final List<AceVenturaContact> second)
        {
            Map<Long, AceVenturaContact> map =
                new LinkedHashMap<>(first.size() + second.size());
            for (AceVenturaContact contact: first) {
                map.put(contact.contactId(), contact);
            }

            for (AceVenturaContact contact: second) {
                map.put(contact.contactId(), contact);
            }

            return new ArrayList<>(map.values());
        }

        public void complete() {
            List<AceVenturaContact> result = null;
            synchronized (this) {
                if (done) {
                    return;
                }

                if (this.original != null) {
                    if (this.original.size() <= 0) {
                        if (this.translit == null) {
                            // waiting for translit
                            return;
                        } else {
                            done = true;
                            result = translit;
                        }
                    } else {
                        if (this.translit != null) {
                            result = deduplicate(original, translit);
                        } else if (!alwaysWaitTranslit) {
                            result = original;
                        } else {
                            return;
                        }

                        done = true;
                    }
                }
            }

            if (result != null) {
                callback.completed(result);
            }
        }

        public void completeTranslit(final List<AceVenturaContact> result) {
            synchronized (this) {
                if (result != null) {
                    this.translit = result;
                } else {
                    this.translit = Collections.emptyList();
                }
            }

            complete();
        }

        public void completeOriginal(final List<AceVenturaContact> result) {
            synchronized (this) {
                if (result != null) {
                    this.original = result;
                } else {
                    this.original = Collections.emptyList();
                }
            }

            complete();
        }

        public void failTranslit(final Exception e) {
            this.context.logger().log(
                Level.WARNING,
                "Translit failed " + e.getMessage());
            boolean failing = false;
            synchronized (this) {
                this.translit = Collections.emptyList();
                if (!done && original != null && original.isEmpty()) {
                    failing = true;
                    done = true;
                }
            }

            if (failing) {
                callback.failed(e);
            } else {
                complete();
            }
        }

        public void failOriginal(final Exception e) {
            this.context.logger().log(
                Level.WARNING,
                "Original failed "
                    + e.getMessage());

            boolean failing = false;
            synchronized (this) {
                this.original = Collections.emptyList();
                if (!done && translit != null && translit.isEmpty()) {
                    failing = true;
                    done = true;
                }
            }

            if (failing) {
                callback.failed(e);
            } else {
                complete();
            }
        }
    }


    private static final class EmailSuggestCallback
        implements FutureCallback<JsonObject>
    {
        private final AceVenturaSuggestContext context;
        private final Aggregator aggregator;
        private final long fetchTime;
        private final boolean translit;

        private EmailSuggestCallback(
            final Aggregator aggregator,
            final AceVenturaSuggestContext context,
            final boolean translit)
        {
            this.context = context;
            this.fetchTime = System.currentTimeMillis();
            this.translit = translit;
            this.aggregator = aggregator;
        }

        @Override
        public void failed(final Exception ex) {
            if (translit) {
                aggregator.failTranslit(ex);
            } else {
                aggregator.failOriginal(ex);
            }
        }

        @Override
        public void cancelled() {
            if (translit) {
                aggregator.completeTranslit(Collections.emptyList());
            } else {
                aggregator.completeOriginal(Collections.emptyList());
            }
        }

        @Override
        public void completed(final JsonObject response) {
            context.logger().info(
                "Email suggest finished in "
                    + (System.currentTimeMillis() - fetchTime) + "ms");
            try {
                JsonMap map = response.asMap();

                JsonList hits = map.getList("hitsArray");

                if (context.debug()) {
                    context.logger().info(
                        "EmailSuggest lucene response "
                            + JsonType.HUMAN_READABLE.toString(hits));
                }

                Map<Long, AceVenturaContact> resMap = new LinkedHashMap<>();
                for (JsonObject item: hits) {
                    JsonMap emailRecord = item.asMap();
                    String emailStr =
                        emailRecord.getString(AceVenturaFields.EMAIL.stored());
                    if (emailStr == null) {
                        continue;
                    }

                    if (context.excludeEmails().contains(
                        emailStr.toLowerCase(Locale.ENGLISH)))
                    {
                        continue;
                    }

                    Long cid =
                        emailRecord.getLong(AceVenturaFields.EMAIL_CID.stored());
                    AceVenturaEmail email =
                        AceVenturaEmail.fromLuceneResponse(emailRecord);

                    AceVenturaContact contact = resMap.get(cid);
                    if (contact == null) {
                        contact =
                            AceVenturaContact.fromLuceneResponse(
                                context.session().logger(),
                                context.prefix(),
                                context.english(),
                                item.asMap());
                        resMap.put(cid, contact);
                    }

                    contact.addEmail(email);
                }

                List<AceVenturaContact> result =
                    new ArrayList<>(resMap.values());
                if (translit) {
                    aggregator.completeTranslit(result);
                } else {
                    aggregator.completeOriginal(result);
                }
            } catch (JsonException je) {
                if (translit) {
                    aggregator.failTranslit(je);
                } else {
                    aggregator.failOriginal(je);
                }
            }
        }
    }
}
