package ru.yandex.ace.ventura.proxy.suggest;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.ace.ventura.AceVenturaFields;
import ru.yandex.ace.ventura.AceVenturaRecordType;
import ru.yandex.ace.ventura.proxy.AceVenturaProxy;
import ru.yandex.ace.ventura.proxy.common.AceVenturaResultTag;
import ru.yandex.ace.ventura.proxy.suggest.tags.TagsSuggestCallback;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.request.util.SearchRequestText;
import ru.yandex.search.rules.pure.SearchRule;
import ru.yandex.search.rules.pure.TranslitRule;
import ru.yandex.search.rules.pure.providers.RequestProvider;

public class AceVenturaTagsSuggestRule<
    T extends RequestProvider & AceVenturaSuggestContextProvider>
    implements SearchRule<T, List<AceVenturaResultTag>>
{
    public static final String LEFT_JOIN_DP =
        "left_join(" + AceVenturaFields.EMAIL_CID.stored()
            + ',' + AceVenturaFields.CID.prefixed()
            // get/out fields
            + ",," + AceVenturaFields.VCARD.stored()
            + ',' + AceVenturaFields.REVISION.stored()
            + ',' + AceVenturaFields.LIST_ID.stored() + ')';

    private static final int CONTACTS_IN_GROUP_MAX = 70;
    private static final int FULL_TRANSLIT_LENGTH_THRSH = 3;
    private static final String OR = " OR ";
    private static final String AND = " AND ";

    private static final Collection<String> SEARCH_FIELDS =
        Collections.unmodifiableList(
            Arrays.asList(
                AceVenturaFields.TAG_NAME_LETTER.prefixed(),
                AceVenturaFields.TAG_NAME.prefixed()));

    private static final Collection<String> TRANSLIT_SEARCH_FIELDS =
        Collections.singleton(AceVenturaFields.TAG_NAME.prefixed());
    private final AceVenturaProxy proxy;

    public AceVenturaTagsSuggestRule(final AceVenturaProxy proxy) {
        this.proxy = proxy;
    }

    public AceVenturaProxy proxy() {
        return proxy;
    }

    @Override
    public void execute(
        final T input,
        final FutureCallback<? super List<AceVenturaResultTag>> callback)
        throws HttpException
    {
        AceVenturaSuggestContext context = input.searchContext();

        SearchRequestText request = SearchRequestText.parseSuggest(
            input.request(),
            context.locale());

        StringBuilder textSb = new StringBuilder();

        textSb.append('(');

        if (request.hasWords()) {
            request.fieldsQuery(textSb, SEARCH_FIELDS);
        } else {
            context.session().logger().warning(
                "Empty request, skipping search in tags");
            callback.completed(Collections.emptyList());
            return;
        }

        textSb.append(')');

        boolean translit = input.request().length() > 1
            && context.session().params().getBoolean("translit", true);

        if (translit) {
            TranslitRule.Context context1 =
                new TranslitRule.Context(input.request());
            Set<String> translitRequests = new LinkedHashSet<>(
                TranslitRule.TABLES.size() << 1);
            for (TranslitRule.Table table: TranslitRule.TABLES) {
                translitRequests.add(table.translate(context1));
            }
            translitRequests.remove(input.request());

            if (translitRequests.size() > 0) {
                for (String translitRequest: translitRequests) {
                    request = new SearchRequestText(translitRequest);
                    Collection<String> searchFields = TRANSLIT_SEARCH_FIELDS;
                    if (translitRequest.length() >= FULL_TRANSLIT_LENGTH_THRSH) {
                        searchFields = SEARCH_FIELDS;
                    }
                    if (request.hasWords()) {
                        textSb.append(OR);
                        textSb.append('(');
                        request.fieldsQuery(textSb, searchFields);
                        textSb.append(')');
                    }
                }
            }
        }
        //request.negationsQuery(textSb, SEARCH_FIELDS);
        QueryConstructor query =
            new QueryConstructor(
                "/search-ace-suggest?IO_PRIO=0&json-type=dollar");

        query.append("service", context.user().service());
        query.append("text", new String(textSb));
        query.append("length", context.length());
        query.append("get",
            AceVenturaFields.TAG_ID.stored() + ','
                + AceVenturaFields.TAG_NAME.stored());
        query.append("prefix", context.user().prefix().toStringFast());

        context.session().logger().info(
            "Tags fetching tags ids " + query.toString());
        context.proxy().sequentialRequest(
            context.session(),
            context,
            new BasicAsyncRequestProducerGenerator(query.toString()),
            null,
            false,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            context.contextGenerator(),
            new TagIdsCallback(context, callback));
    }

    private static class TagIdsCallback
        extends AbstractFilterFutureCallback<JsonObject, List<AceVenturaResultTag>>
    {
        private final AceVenturaSuggestContext context;

        public TagIdsCallback(
            final AceVenturaSuggestContext context,
            final FutureCallback<? super List<AceVenturaResultTag>> callback)
        {
            super(callback);

            this.context = context;
        }

        @Override
        public void completed(final JsonObject response) {
            QueryConstructor query =
                new QueryConstructor(
                    "/search-ace-suggest-tags?IO_PRIO=0&json-type=dollar");

            Set<Integer> tagIds;
            try {
                JsonMap map = response.asMap();
                //long total = map.getLong("hitsCount");
                JsonList hits = map.getList("hitsArray");
                tagIds = new LinkedHashSet<>(hits.size());

                StringBuilder textSb = new StringBuilder();
                textSb.append(AceVenturaFields.RECORD_TYPE.prefixed());
                textSb.append(':');
                textSb.append(AceVenturaRecordType.EMAIL.fieldValue());
                textSb.append(AND);
                textSb.append(AceVenturaFields.TAGS.prefixed());
                textSb.append(":(");
                Map<Integer, String> tagMap = new LinkedHashMap<>(hits.size());
                for (JsonObject jo: hits) {
                    JsonMap tagObj = jo.asMap();
                    int tagId =
                        tagObj.getInt(AceVenturaFields.TAG_ID.stored());
                    String tagName =
                        tagObj.getString(AceVenturaFields.TAG_NAME.stored());

                    tagMap.put(tagId, tagName);

                    if (tagIds.add(tagId)) {
                        textSb.append(tagId);
                        textSb.append(OR);
                    }
                }

                if (tagIds.size() <= 0) {
                    callback.completed(Collections.emptyList());
                    return;
                }

                context.session().logger().info("Found tags: " + tagIds);

                textSb.setLength(textSb.length() - 4);
                textSb.append(')');

                query.append("service", context.user().service());
                query.append("text", textSb.toString());
                query.append(
                    "length",
                    tagIds.size() * Math.max(CONTACTS_IN_GROUP_MAX, context.length()));
                query.append("dp", LEFT_JOIN_DP);
                query.append("get", "*");
                query.append(
                    "prefix",
                    context.user().prefix().toStringFast());

                context.session().logger().info(
                    "Tags suggest rule executed " + query.toString());
                context.proxy().parallelRequest(
                    context.session(),
                    context,
                    new BasicAsyncRequestProducerGenerator(query.toString()),
                    JsonAsyncTypesafeDomConsumerFactory.OK,
                    context.contextGenerator(),
                    new TagsSuggestCallback(callback, context, tagMap));
            } catch (JsonException | BadRequestException e) {
                failed(e);
                return;
            }

            if (tagIds.size() == 0) {
                callback.completed(Collections.emptyList());
                return;
            }
        }
    }

}
