package ru.yandex.ace.ventura.proxy.suggest;

import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Locale;
import java.util.Set;
import java.util.function.Predicate;
import java.util.regex.Pattern;

import ru.yandex.ace.ventura.AceVenturaFields;
import ru.yandex.ace.ventura.proxy.AbstractAceVenturaProxyContext;
import ru.yandex.ace.ventura.proxy.AceVenturaProxy;
import ru.yandex.ace.ventura.proxy.AceVenturaProxyParams;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.search.rules.pure.TranslitRule;
import ru.yandex.search.rules.pure.providers.RequestProvider;

public class BasicAceVenturaSuggestContext
    extends AbstractAceVenturaProxyContext
    implements AceVenturaSuggestContext,
    RequestProvider,
    AceVenturaSuggestContextProvider
{
    public static final Pattern LONG_QUERY_SPLITTER = Pattern.compile("\\s+");
    private static final Predicate<String> FILTER_EXCLUDES = (s) -> !s.contains("@");
    private static final int MAX_QUERY_LENGTH = 40;
    private static final int MAX_TANSLIT_LENGTH = 20;
    private static final int MIN_TANSLIT_LENGTH = 1;

    private static final int DEFAULT_LIMIT = 30;
    private static final CollectionParser<String, Set<String>, Exception>
        SET_PARSER = new CollectionParser<>(String::trim, LinkedHashSet::new);

    private final int length;
    private final String request;

    private final Set<String> excludeEmails;
    private final Set<String> fields;
    private final Set<String> translits;
    private final List<Long> listIdFilter = Collections.emptyList();
    private final boolean onlyIfhasPhone;
    private volatile boolean forceEnglish = false;

    public BasicAceVenturaSuggestContext(
        final AceVenturaProxy proxy,
        final ProxySession session)
        throws BadRequestException
    {
        super(proxy, session);

        CgiParams params = session.params();

        fields = params.getAll(
            AceVenturaProxyParams.GET_FIELDS,
            Collections.emptySet(),
            SET_PARSER);

        Set<String> excludeEmails = new LinkedHashSet<>();
        excludeEmails.addAll(
            params.get("to", Collections.emptySet(), SET_PARSER));
        excludeEmails.addAll(
            params.get("cc", Collections.emptySet(), SET_PARSER));
        excludeEmails.addAll(
            params.get("bcc", Collections.emptySet(), SET_PARSER));

        if (!excludeEmails.isEmpty()) {
            excludeEmails.removeIf(FILTER_EXCLUDES);
        }

        this.excludeEmails = Collections.unmodifiableSet(excludeEmails);

        onlyIfhasPhone = params.getBoolean(
            AceVenturaProxyParams.HAS_TELEPHONE_NUMBER,
            false);
        String query = params.getString(AceVenturaProxyParams.QUERY, "");
        if (query.length() > MAX_QUERY_LENGTH) {
            query = query.substring(query.length() - MAX_QUERY_LENGTH);
            query = query.trim();
            String[] split = LONG_QUERY_SPLITTER.split(query);
            if (split.length > 0) {
                query = split[split.length - 1];
            }
        } else {
            query = query.trim();
            int atIndex = query.lastIndexOf('@');
            if (atIndex >= 0) {
                String[] split = LONG_QUERY_SPLITTER.split(query);
                if (split.length > 1) {
                    for (int i = split.length - 1; i >= 0; i--) {
                        atIndex = split[i].lastIndexOf('@');
                        if (atIndex >= 0) {
                            query = split[i];
                            break;
                        }
                    }
                }
            }
        }

        request = query;
        length = params.getInt(AceVenturaProxyParams.LIMIT, DEFAULT_LIMIT);

        boolean translit =
            request.length() > MIN_TANSLIT_LENGTH
                && request.length() < MAX_TANSLIT_LENGTH
                && params.getBoolean("translit", true);
        if (translit) {
            TranslitRule.Context context1 =
                new TranslitRule.Context(request);
            Set<String> translitRequests = new LinkedHashSet<>(
                TranslitRule.TABLES.size() << 1);
            for (TranslitRule.Table table: TranslitRule.TABLES) {
                translitRequests.add(table.translate(context1));
            }
            translitRequests.remove(request);
            translits = translitRequests;
        } else {
            translits = Collections.emptySet();
        }
    }

    @Override
    public Set<String> translits() {
        return translits;
    }

    @Override
    public BasicAceVenturaSuggestContext searchContext() {
        return this;
    }

    @Override
    public int length() {
        return length;
    }

    @Override
    public String request() {
        return request;
    }

    @Override
    public Set<String> excludeEmails() {
        return excludeEmails;
    }

    @Override
    public Set<String> fields() {
        return fields;
    }

    @Override
    public boolean onlyIfhasPhone() {
        return onlyIfhasPhone;
    }

    @Override
    public List<Long> listIdFilter() {
        return listIdFilter;
    }

    @Override
    public AceVenturaFields sortField() {
        return AceVenturaFields.LAST_USAGE;
    }

    public void forceEnglish() {
        forceEnglish = true;
    }

    @Override
    public boolean english() {
        if (forceEnglish) {
            return true;
        }

        return corp() && Locale.ENGLISH.equals(locale());
    }
}
