package ru.yandex.ace.ventura.proxy.suggest;

import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.ace.ventura.proxy.common.AceVenturaContact;
import ru.yandex.ace.ventura.proxy.common.AceVenturaEmail;
import ru.yandex.ace.ventura.proxy.common.AceVenturaResultTag;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.ErrorSuppressingFutureCallback;
import ru.yandex.http.util.FilterFutureCallback;
import ru.yandex.http.util.RequestErrorType;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.rules.pure.SearchRule;

public class MailSearchFallbackRule
    implements SearchRule<AceVenturaSuggestContext, Map.Entry<Collection<AceVenturaContact>, List<AceVenturaResultTag>>>
{
    private static final int MINIMUM_REQUEST_LENGTH = 4;
    private static final int MAXIMUM_REQUEST_LENGTH = 20;
    private final SearchRule<AceVenturaSuggestContext, Map.Entry<Collection<AceVenturaContact>, List<AceVenturaResultTag>>> next;

    public MailSearchFallbackRule(
        final SearchRule<AceVenturaSuggestContext, Map.Entry<Collection<AceVenturaContact>, List<AceVenturaResultTag>>> next)
    {
        this.next = next;
    }

    @Override
    public void execute(
        final AceVenturaSuggestContext input,
        final FutureCallback<? super Map.Entry<Collection<AceVenturaContact>, List<AceVenturaResultTag>>> callback)
        throws HttpException
    {
        next.execute(input, new Callback(input, callback));
    }

    private static class Callback
        extends FilterFutureCallback<Map.Entry<Collection<AceVenturaContact>, List<AceVenturaResultTag>>>
    {
        private final AceVenturaSuggestContext context;

        public Callback(
            final AceVenturaSuggestContext context,
            final FutureCallback<? super Map.Entry<Collection<AceVenturaContact>, List<AceVenturaResultTag>>> callback)
        {
            super(callback);
            this.context = context;
        }

        @Override
        public void completed(final Map.Entry<Collection<AceVenturaContact>, List<AceVenturaResultTag>> result) {
            if (!result.getValue().isEmpty()
                || !result.getKey().isEmpty()
                || context.request().length() < MINIMUM_REQUEST_LENGTH
                || context.request().length() > MAXIMUM_REQUEST_LENGTH)
            {
                callback.completed(result);
                return;
            }

            AsyncClient client =
                context.proxy().mailSearchProxyClient()
                    .adjust(context.session().context());

            QueryConstructor qc = new QueryConstructor("/api/async/mail/suggest/contact?");
            try {
                qc.append("request", context.request());
                qc.append("timeout", context.proxy().config().mailSearch().timeout());
                qc.append("uid", context.prefix().uid());
                qc.append("aceventura", "1");
            } catch (BadRequestException bre) {
                completed(result);
                return;
            }

            client.execute(
                context.proxy().mailSearchConfig().host(),
                new BasicAsyncRequestProducerGenerator(qc.toString()),
                JsonAsyncTypesafeDomConsumerFactory.OK,
                context.session().listener().createContextGeneratorFor(client),
                new MailSearchCallback(
                    new ErrorSuppressingFutureCallback<>(callback, (e) -> RequestErrorType.HTTP, result),
                    context));
        }
    }


    private static class MailSearchCallback
        extends AbstractFilterFutureCallback<JsonObject, Map.Entry<Collection<AceVenturaContact>, List<AceVenturaResultTag>>>
    {
        private final AceVenturaSuggestContext context;

        public MailSearchCallback(
            final FutureCallback<? super Map.Entry<Collection<AceVenturaContact>,
                List<AceVenturaResultTag>>> callback,
            final AceVenturaSuggestContext context)
        {
            super(callback);

            this.context = context;
        }

        @Override
        public void completed(final JsonObject root) {
            List<AceVenturaContact> contacts;
            try {
                JsonList contactsList = root.asMap().getList("contacts");
                contacts = new ArrayList<>(contactsList.size());
                for (JsonObject contactObj: contactsList) {
                    JsonMap contactMap = contactObj.asMap();
                    String email = contactMap.getString("email");

                    AceVenturaContact contact =
                        new AceVenturaContact(
                            context.prefix(),
                            -1L,
                            -1L,
                            "0",
                            Collections.emptySet(),
                            new JsonMap(BasicContainerFactory.INSTANCE));

                    contact.addEmail(
                        new AceVenturaEmail(
                            -1L,
                            Collections.emptySet(),
                            email,
                            0L));
                    contacts.add(contact);
                }
            } catch (JsonException je) {
                failed(je);
                return;
            }

            context.logger().info("Mail search fallback found " + contacts.size());
            callback.completed(new AbstractMap.SimpleEntry<>(contacts, Collections.emptyList()));
        }
    }
}
