package ru.yandex.ace.ventura.proxy.suggest;

import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.collections4.trie.PatriciaTrie;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.ace.ventura.proxy.common.AceVenturaContact;
import ru.yandex.ace.ventura.proxy.common.AceVenturaEmail;
import ru.yandex.ace.ventura.proxy.common.AceVenturaResultTag;
import ru.yandex.http.util.FilterFutureCallback;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonMap;

public class PopularDomainsFallbackSuggestCallback
    extends FilterFutureCallback<Map.Entry<Collection<AceVenturaContact>, List<AceVenturaResultTag>>>
{
    private static final PatriciaTrie<Domain> DOMAINS;

    static {
        Set<String> domainsSet =
            new LinkedHashSet<>(
                Arrays.asList(
                    "yandex.ru",
                    "yahoo.com",
                    "gmail.com",
                    "mail.ru",
                    "rambler.ru",
                    "icloud.com",
                    "qip.ru",
                    "bk.ru",
                    "inbox.ru"
                ));

        DOMAINS = new PatriciaTrie<>();
        int weight = 0;
        for (String domain: domainsSet) {
            DOMAINS.put(domain, new Domain(domain, weight));
            weight += 1;
        }
    }

    private final AceVenturaSuggestContext context;

    public PopularDomainsFallbackSuggestCallback(
        final AceVenturaSuggestContext context,
        final FutureCallback<? super Map.Entry<Collection<AceVenturaContact>, List<AceVenturaResultTag>>> callback)
    {
        super(callback);

        this.context = context;
    }

    @Override
    public void completed(final Map.Entry<Collection<AceVenturaContact>, List<AceVenturaResultTag>> result) {
        if (!result.getValue().isEmpty() || !result.getKey().isEmpty()) {
            callback.completed(result);
            return;
        }

        int atIndex = context.request().lastIndexOf('@');
        if (atIndex > 0) {
            String preDomainPart = context.request().substring(0, atIndex + 1);
            String domainPart = context.request().substring(atIndex + 1);
            Map<String, Domain> domainsMap = DOMAINS.prefixMap(domainPart);
            List<Domain> domains = new ArrayList<>(domainsMap.values());
            Collections.sort(domains);

            List<AceVenturaContact> contacts = new ArrayList<>();
            for (Domain domain: domains) {
                AceVenturaContact contact =
                    new AceVenturaContact(
                        context.prefix(),
                        -1L,
                        -1L,
                        "0",
                        Collections.emptySet(),
                        new JsonMap(BasicContainerFactory.INSTANCE));

                contact.addEmail(
                    new AceVenturaEmail(
                        -1L,
                        Collections.emptySet(),
                        preDomainPart + domain.name(), 0L));

                contacts.add(contact);
            }

            callback.completed(new AbstractMap.SimpleEntry<>(contacts, Collections.emptyList()));
        } else {
            callback.completed(result);
        }
    }

    private static final class Domain implements Comparable<Domain> {
        private final String name;
        private final int weight;

        public Domain(final String name, final int weight) {
            this.name = name;
            this.weight = weight;
        }

        public String name() {
            return name;
        }

        public long weight() {
            return weight;
        }

        @Override
        public int compareTo(Domain o) {
            return Integer.compare(weight, o.weight);
        }
    }
}
