package ru.yandex.ace.ventura.proxy.suggest;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.http.HttpException;

import ru.yandex.ace.ventura.proxy.AceVenturaProxy;
import ru.yandex.ace.ventura.proxy.DeduplicateByEmailRule;
import ru.yandex.ace.ventura.proxy.common.AceVenturaContact;
import ru.yandex.ace.ventura.proxy.common.AceVenturaResultTag;
import ru.yandex.ace.ventura.proxy.suggest.corp.AceVenturaCorpSuggestRule;
import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.search.rules.pure.SearchRule;
import ru.yandex.stater.CountAggregatorFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;

public class SuggestHandler implements ProxyRequestHandler {
    private static final Set<Long> EXPERIMENT_USERS;

    static {
        Set<Long> users = new LinkedHashSet<>();
        try {
            List<String> lines =
                Files.readAllLines(
                    Paths.get(
                        new File("experiment_corp_users.txt").toURI()));
            for (String line: lines) {
                if (!line.isEmpty()) {
                    try {
                        String[] split = line.trim().split("\\s+");
                        users.add(Long.parseLong(split[0].trim()));
                        System.err.println("Experiment user parsed " + split[0]);
                    } catch (NumberFormatException nfe) {
                        System.err.println("Failed to parse uid " + line);
                    }
                }
            }
        } catch (IOException ioe) {
            System.err.println("Failed to load experiments users file");
            ioe.printStackTrace(System.err);
        }

        EXPERIMENT_USERS = Collections.unmodifiableSet(users);
    }

    private final AceVenturaProxy proxy;

    private final SearchRule<AceVenturaSuggestContext,
        Collection<AceVenturaContact>> contactsRule;

    private final SearchRule<AceVenturaSuggestContext,
        Map.Entry<Collection<AceVenturaContact>, List<AceVenturaResultTag>>> suggestRule;

    private final AceVenturaCorpSuggestRule<AceVenturaSuggestContext> corpRule;

    private final TimeFrameQueue<Integer> emptyResponses;
    private final TimeFrameQueue<Integer> corpEmptyResponses;
    private final TimeFrameQueue<Integer> emptyResponseAfterAt;
    private final TimeFrameQueue<Integer> corpEmptyResponseAfterAt;

    public SuggestHandler(final AceVenturaProxy proxy) {
        this.proxy = proxy;

        emptyResponses = new TimeFrameQueue<>(proxy.config().metricsTimeFrame());
        proxy.registerStater(
            new PassiveStaterAdapter<>(
                emptyResponses,
                new NamedStatsAggregatorFactory<>(
                    "suggest-empty-responses_ammm",
                    CountAggregatorFactory.INSTANCE)));

        corpEmptyResponses = new TimeFrameQueue<>(proxy.config().metricsTimeFrame());
        proxy.registerStater(
            new PassiveStaterAdapter<>(
                corpEmptyResponses,
                new NamedStatsAggregatorFactory<>(
                    "suggest-corp-empty-responses_ammm",
                    CountAggregatorFactory.INSTANCE)));

        emptyResponseAfterAt = new TimeFrameQueue<>(proxy.config().metricsTimeFrame());
        proxy.registerStater(
            new PassiveStaterAdapter<>(
                emptyResponseAfterAt,
                new NamedStatsAggregatorFactory<>(
                    "suggest-empty-responses-with-at_ammm",
                    CountAggregatorFactory.INSTANCE)));

        corpEmptyResponseAfterAt = new TimeFrameQueue<>(proxy.config().metricsTimeFrame());
        proxy.registerStater(
            new PassiveStaterAdapter<>(
                corpEmptyResponseAfterAt,
                new NamedStatsAggregatorFactory<>(
                    "suggest-corp-empty-responses-with-at_ammm",
                    CountAggregatorFactory.INSTANCE)));

        contactsRule = new DeduplicateByEmailRule<>(
            Arrays.asList(
                new AceVenturaEmailSuggestRule<>(proxy),
                new SharedSuggestRule<>(
                    proxy,
                    new AceVenturaEmailSuggestRule<>(proxy))));

        this.corpRule = new AceVenturaCorpSuggestRule<>(proxy);
        this.suggestRule =
            new MailSearchFallbackRule(
                new PopularDomainsFallbackSuggestRule(
                    new TagsAndContactsSuggestRule(contactsRule, new AceVenturaTagsSuggestRule<>(proxy))));
    }

    public TimeFrameQueue<Integer> emptyResponses() {
        return emptyResponses;
    }

    public TimeFrameQueue<Integer> emptyResponseAfterAt() {
        return emptyResponseAfterAt;
    }

    public TimeFrameQueue<Integer> corpEmptyResponses() {
        return corpEmptyResponses;
    }

    public TimeFrameQueue<Integer> corpEmptyResponseAfterAt() {
        return corpEmptyResponseAfterAt;
    }

    @Override
    public void handle(final ProxySession session)
        throws HttpException, IOException
    {
        BasicAceVenturaSuggestContext suggestContext =
            new BasicAceVenturaSuggestContext(proxy, session);

        SuggestResultPrinter printer =
            new SuggestResultPrinter(this, session, suggestContext);

        //suggestContext.prefix().uid() == 1120000000092594L ||
        if (suggestContext.prefix().uid() == 1120000000038012L ||
            suggestContext.prefix().uid() == 1120000000040290L)
        {
            suggestContext.forceEnglish();
        }

        if (suggestContext.corp() && corpExperiment(session.params(), suggestContext)) {
            suggestContext.logger().warning("User in corp experiment");
            suggestContext.forceEnglish();
            corpRule.execute(suggestContext, printer);
        } else {
            suggestRule.execute(suggestContext, printer);
        }
    }

    protected boolean corpExperiment(
        final CgiParams params,
        final BasicAceVenturaSuggestContext context)
        throws HttpException
    {
        Boolean experiment = params.getBoolean("experiment", null);
        if (experiment == null) {
            return EXPERIMENT_USERS.contains(context.prefix().uid());
        }

        return experiment;
    }
}
