package ru.yandex.ace.ventura.proxy.suggest.corp;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.http.HttpException;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.ace.ventura.AceVenturaFields;
import ru.yandex.ace.ventura.AceVenturaPrefix;
import ru.yandex.ace.ventura.AceVenturaRecordType;
import ru.yandex.ace.ventura.UserType;
import ru.yandex.ace.ventura.proxy.AceVenturaProxy;
import ru.yandex.ace.ventura.proxy.DeduplicateByEmailCallback;
import ru.yandex.ace.ventura.proxy.common.AceVenturaContact;
import ru.yandex.ace.ventura.proxy.common.AceVenturaResultTag;
import ru.yandex.ace.ventura.proxy.suggest.AceVenturaSuggestContext;
import ru.yandex.ace.ventura.proxy.suggest.AceVenturaSuggestContextProvider;
import ru.yandex.ace.ventura.proxy.suggest.AceVenturaTagsSuggestRule;
import ru.yandex.ace.ventura.proxy.suggest.tags.TagsSuggestCallback;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.DoubleFutureCallback;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.request.util.SearchRequestText;
import ru.yandex.search.rules.pure.SearchRule;
import ru.yandex.search.rules.pure.providers.RequestProvider;

public class AceVenturaCorpSuggestRule<T extends RequestProvider & AceVenturaSuggestContextProvider>
    implements SearchRule<T, Map.Entry<Collection<AceVenturaContact>, List<AceVenturaResultTag>>>
{
    private static final ScoreContactsComparator COMPARATOR = new ScoreContactsComparator(new CorpScorer());

    private static final String FIRST_STEP_INFO_TEXT =
        AceVenturaFields.RECORD_TYPE.prefixed() + ":("
            + AceVenturaRecordType.SHARED.fieldValue() + " OR "
            + AceVenturaRecordType.CORP_STAFF.fieldValue() + ")";

    private static final String FIRST_STEP_GET_FIELDS =
        AceVenturaFields.SHARED_OWNER_UTYPE.stored() + ','
            + AceVenturaFields.SHARED_OWNER_UID.stored() + ','
            + AceVenturaFields.SHARED_LIST_ID.stored() + ','
            + AceVenturaFields.TAG_ID.stored() + ','
            + AceVenturaFields.TAG_NAME.stored() + ','
            + AceVenturaFields.RECORD_TYPE.stored() + ','
            + AceVenturaFields.CORP_DEPARTMENT_ID.stored() + ','
            + AceVenturaFields.CORP_DISMISSED.stored() + ','
            + AceVenturaFields.CORP_DEPARTMENTS.stored() + ','
            + AceVenturaFields.CORP_DEPARTMENT_LEVEL.stored();

    private static final String OR = " OR ";
    private static final String AND = " AND ";

    private final PersonalCorpEmailSuggestRule persContactsRule;
    private final SharedCorpEmailSuggestRule sharedContactsRule;

    public AceVenturaCorpSuggestRule(final AceVenturaProxy proxy) {
        this.persContactsRule =  new PersonalCorpEmailSuggestRule(proxy);
        this.sharedContactsRule = new SharedCorpEmailSuggestRule(proxy);
    }

    protected void buildTagsQuery(
        final AceVenturaSuggestContext context,
        final StringBuilder text)
    {
        SearchRequestText request = SearchRequestText.parseSuggest(
            context.request(),
            context.locale());
        if (!request.hasWords()) {
            context.session().logger().warning(
                "Empty request, skipping search in tags");
            return;
        }

        text.append(" OR (");
        text.append(AceVenturaFields.RECORD_TYPE.prefixed());
        text.append(':');
        text.append(AceVenturaRecordType.TAG.fieldValue());
        text.append(" AND (");
        request.fieldsQuery(text, AceVenturaFields.TAG_NAME.prefixed());

        if (context.translits().size() > 0) {
            for (String translitRequest: context.translits()) {
                request = new SearchRequestText(translitRequest);
                if (request.hasWords()) {
                    text.append(OR);
                    request.fieldsQuery(text, AceVenturaFields.TAG_NAME.prefixed());
                }
            }
        }

        text.append("))");
    }

    @Override
    public void execute(
        final T input,
        final FutureCallback<? super Map.Entry<Collection<AceVenturaContact>, List<AceVenturaResultTag>>> callback)
        throws HttpException
    {
        final AceVenturaSuggestContext context = input.searchContext();

        context.logger().info("Corp rule executed");
        QueryConstructor query =
            new QueryConstructor(
                "/search-ace-suggest?IO_PRIO=0&json-type=dollar");

        StringBuilder text = new StringBuilder(FIRST_STEP_INFO_TEXT);
        if (!input.request().isEmpty()) {
            // tags
            buildTagsQuery(context, text);
        }
        query.append("service", context.user().service());
        query.append("text", text.toString());
        query.append("get", FIRST_STEP_GET_FIELDS);
        query.append("prefix", context.user().prefix().toStringFast());

        context.session().logger().info(
            "Fetching shared lists and info " + query.toString());
        context.proxy().parallelRequest(
            context.session(),
            context,
            new BasicAsyncRequestProducerGenerator(query.toString()),
            JsonAsyncTypesafeDomConsumerFactory.OK,
            context.contextGenerator(),
            new FirstCallback(context, callback));
        //AceVenturaSuggestContext context = input.searchContext();
        StringBuilder queryText = new StringBuilder();

        queryText.append(AceVenturaFields.RECORD_TYPE.prefixed());
        queryText.append(":(");
        queryText.append(AceVenturaRecordType.TAG.fieldValue());
        queryText.append(OR);
        queryText.append(AceVenturaRecordType.CORP_STAFF.fieldValue());
        queryText.append(OR);
        queryText.append(AceVenturaRecordType.SHARED.fieldValue());
        queryText.append(":)");
    }

    private final class FirstCallback
        extends AbstractFilterFutureCallback<JsonObject, Map.Entry<Collection<AceVenturaContact>, List<AceVenturaResultTag>>>
    {
        private final AceVenturaSuggestContext context;

        public FirstCallback(
            final AceVenturaSuggestContext context,
            final FutureCallback<? super Map.Entry<Collection<AceVenturaContact>, List<AceVenturaResultTag>>> callback)
        {
            super(callback);

            this.context = context;
        }

        void executeTagsSuggest(
            final Map<Integer, String> tagMap,
            final FutureCallback<List<AceVenturaResultTag>> callback)
            throws HttpException
        {
            if (tagMap.size() == 0) {
                callback.completed(Collections.emptyList());
                return;
            }
            QueryConstructor query =
                new QueryConstructor(
                    "/search-ace-suggest-tags?IO_PRIO=0&json-type=dollar");
            context.session().logger().info("Found tags: " + tagMap);

            StringBuilder textSb = new StringBuilder();
            textSb.append(AceVenturaFields.RECORD_TYPE.prefixed());
            textSb.append(':');
            textSb.append(AceVenturaRecordType.EMAIL.fieldValue());
            textSb.append(AND);
            textSb.append(AceVenturaFields.TAGS.prefixed());
            textSb.append(":(");
            for (Map.Entry<Integer, String> entry : tagMap.entrySet()) {
                textSb.append(entry.getKey());
                textSb.append(OR);
            }
            textSb.setLength(textSb.length() - 4);
            textSb.append(')');

            query.append("service", context.user().service());
            query.append("text", textSb.toString());
            query.append("length", context.length());
            query.append("dp", AceVenturaTagsSuggestRule.LEFT_JOIN_DP);
            query.append("get", "*");
            query.append(
                "prefix",
                context.user().prefix().toStringFast());

            context.session().logger().info(
                "Tags suggest rule executed " + query.toString());
            context.proxy().sequentialRequest(
                context.session(),
                context,
                new BasicAsyncRequestProducerGenerator(query.toString()),
                context.failOverDelay(),
                false,
                JsonAsyncTypesafeDomConsumerFactory.OK,
                context.contextGenerator(),
                new TagsSuggestCallback(callback, context, tagMap));
        }

        @Override
        public void completed(final JsonObject response) {
            try {
                //long total = map.getLong("hitsCount");
                JsonList hits = response.asMap().getList("hitsArray");

                Map<AceVenturaPrefix, List<Long>> shareMap =
                    new LinkedHashMap<>();
                String departments = "";
                Map<Integer, String> tagMap = new LinkedHashMap<>(hits.size());

                for (JsonObject jo: hits) {
                    JsonMap map = jo.asMap();
                    AceVenturaRecordType rtype =
                        map.getEnum(
                            AceVenturaRecordType.class,
                            AceVenturaFields.RECORD_TYPE.stored(),
                            null);
                    if (rtype == null) {
                        context.logger().warning(
                            "Unexpected null record type " + JsonType.NORMAL.toString(map));
                        continue;
                    }
                    switch (rtype) {
                        case SHARED:
                            Long listId =
                                map.getLong(AceVenturaFields.SHARED_LIST_ID.stored());
                            Long uid =
                                map.getLong(
                                    AceVenturaFields.SHARED_OWNER_UID.stored());
                            UserType userType =
                                map.getEnum(
                                    UserType.class,
                                    AceVenturaFields.SHARED_OWNER_UTYPE.stored());
                            AceVenturaPrefix owner = new AceVenturaPrefix(uid, userType);
                            shareMap.computeIfAbsent(
                                owner,
                                (x) -> new ArrayList<>()).add(listId);
                            break;
                        case CORP_STAFF:
                            departments = map.getString(AceVenturaFields.CORP_DEPARTMENTS.stored(), "");
                            departments = departments.replaceAll("\\n", ";");
                            break;
                        case TAG:
                            int tagId = map.getInt(AceVenturaFields.TAG_ID.stored());
                            String tagName = map.getString(AceVenturaFields.TAG_NAME.stored());

                            tagMap.put(tagId, tagName);
                            break;
                        default:
                            context.logger().warning("Unexpected record type " + rtype);
                            break;
                    }
                }

                DoubleFutureCallback<
                    Collection<AceVenturaContact>, List<AceVenturaResultTag>> dfCallback
                    = new DoubleFutureCallback<>(callback);

                MultiFutureCallback<Collection<AceVenturaContact>> mfcb =
                    new MultiFutureCallback<>(
                        new DeduplicateByEmailCallback<>(context, COMPARATOR, dfCallback.first()));
                // personal
                persContactsRule.execute(
                    new CorpSuggestContext(context, departments),
                    mfcb.newCallback());
                // shared
                if (shareMap.size() == 0) {
                    context.logger().fine("No shared lists found");
                    dfCallback.second().completed(Collections.emptyList());
                } else {
                    for (Map.Entry<AceVenturaPrefix, List<Long>> entry: shareMap.entrySet()) {
                        CorpSharedSuggestContext sharedContext =
                            new CorpSharedSuggestContext(
                                context,
                                entry.getKey(),
                                departments,
                                entry.getValue());
                        sharedContactsRule.execute(sharedContext, mfcb.newCallback());
                    }
                }

                mfcb.done();

                executeTagsSuggest(tagMap, dfCallback.second());
            } catch (JsonException | HttpException e) {
                failed(e);
                return;
            }
        }
    }
}
