package ru.yandex.ace.ventura.salo;

import java.io.IOException;
import java.nio.charset.CharacterCodingException;
import java.nio.charset.CharsetEncoder;
import java.nio.charset.CodingErrorAction;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;

import ru.yandex.ace.ventura.UserType;
import ru.yandex.ace.ventura.salo.handlers.AceVenturaEmptyHandler;
import ru.yandex.ace.ventura.salo.handlers.UnknownAceVenturaHandler;
import ru.yandex.ace.ventura.salo.handlers2.CopyAbookHandler;
import ru.yandex.ace.ventura.salo.handlers2.ReindexUserWithCleanupHandler;
import ru.yandex.ace.ventura.salo.handlers2.contacts.CreateContactsHandler;
import ru.yandex.ace.ventura.salo.handlers2.contacts.DeleteContactsHandler;
import ru.yandex.ace.ventura.salo.handlers2.contacts.UpdateContactsHandler;
import ru.yandex.ace.ventura.salo.handlers2.emails.CreateEmailsHandler;
import ru.yandex.ace.ventura.salo.handlers2.emails.DeleteEmailsHandler;
import ru.yandex.ace.ventura.salo.handlers2.emails.UpdateEmailsHandler;
import ru.yandex.ace.ventura.salo.handlers2.shared.RevokeListHandler;
import ru.yandex.ace.ventura.salo.handlers2.shared.ShareListHandler;
import ru.yandex.ace.ventura.salo.handlers2.tags.CreateTagHandler;
import ru.yandex.ace.ventura.salo.handlers2.tags.DeleteTagHandler;
import ru.yandex.ace.ventura.salo.handlers2.tags.TagContactsHandler;
import ru.yandex.ace.ventura.salo.handlers2.tags.TagEmailsHandler;
import ru.yandex.ace.ventura.salo.handlers2.tags.UnTagContactsHandler;
import ru.yandex.ace.ventura.salo.handlers2.tags.UnTagEmailsHandler;
import ru.yandex.ace.ventura.salo.handlers2.tags.UpdateTagHandler;
import ru.yandex.charset.Encoder;
import ru.yandex.dbfields.CollieFields;
import ru.yandex.function.ByteArrayCopier;
import ru.yandex.function.StringBuilderVoidProcessor;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.dom.ContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.PositionSavingContainerFactory;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.search.salo.Envelope;
import ru.yandex.search.salo.EnvelopeFactory;
import ru.yandex.search.salo.EnvelopesContext;
import ru.yandex.search.salo.Mdb;

public class AceVenturaEnvelopesFactory implements EnvelopeFactory {
    private static final int DEFAULT_ENVELOPES_ARRAY_SIZE = 1000;
    private final Map<CollieChangeType, AceVenturaIndexHandler> handlers;

    private final PrefixedLogger logger;
    private final AceVenturaMdbsContext mdbsContext;
    //https://st.yandex-team.ru/MAILPG-4733
    private static final Set<Long> BANNED_ORGS =
        Collections.unmodifiableSet(
            new LinkedHashSet<>(
                Arrays.asList(1093155L, 5690183L, 5690188L, 5690231L, 5690238L)));
    private static final int BAN_TH = 1000000;


    public AceVenturaEnvelopesFactory(final AceVenturaMdbsContext context) {
        this.mdbsContext = context;

        Map<CollieChangeType, AceVenturaIndexHandler> map =
            new LinkedHashMap<>();
        map.put(CollieChangeType.COPY_ABOOK, new CopyAbookHandler(context));
        map.put(CollieChangeType.CREATE_USER, new ReindexUserWithCleanupHandler(context, false));
        map.put(CollieChangeType.DELETE_USER, new ReindexUserWithCleanupHandler(context, false));
        map.put(
            CollieChangeType.CREATE_CONTACTS,
            new CreateContactsHandler(context));
        map.put(
            CollieChangeType.UPDATE_CONTACTS,
            new UpdateContactsHandler(context));
        map.put(
            CollieChangeType.DELETE_CONTACTS,
            new DeleteContactsHandler(context));
        map.put(
            CollieChangeType.CREATE_EMAILS,
            new CreateEmailsHandler(context));
        map.put(
            CollieChangeType.UPDATE_EMAILS,
            new UpdateEmailsHandler(context));
        map.put(
            CollieChangeType.DELETE_EMAILS,
            new DeleteEmailsHandler(context));
        // tags
        map.put(CollieChangeType.CREATE_TAG, new CreateTagHandler(context));
        map.put(CollieChangeType.UPDATE_TAG, new UpdateTagHandler(context));
        map.put(CollieChangeType.DELETE_TAG, new DeleteTagHandler(context));
        map.put(CollieChangeType.TAG_EMAILS, new TagEmailsHandler(context));
        map.put(CollieChangeType.UNTAG_EMAILS, new UnTagEmailsHandler(context));
        map.put(CollieChangeType.TAG_CONTACTS, new TagContactsHandler(context));
        map.put(
            CollieChangeType.UNTAG_CONTACTS,
            new UnTagContactsHandler(context));

        map.put(CollieChangeType.CREATE_LIST, AceVenturaEmptyHandler.INSTANCE);
        map.put(CollieChangeType.DELETE_LIST, AceVenturaEmptyHandler.INSTANCE);
        map.put(
            CollieChangeType.SUBSCRIBED_LISTS,
            AceVenturaEmptyHandler.INSTANCE);
        map.put(
            CollieChangeType.SUBSCRIBE_TO_LIST,
            AceVenturaEmptyHandler.INSTANCE);
        map.put(
            CollieChangeType.REVOKE_SUBSCRIBED_LIST,
            AceVenturaEmptyHandler.INSTANCE);
        map.put(
            CollieChangeType.CREATE_DIRECTORY_ENTITIES,
            AceVenturaEmptyHandler.INSTANCE);
        map.put(
            CollieChangeType.DELETE_DIRECTORY_ENTITIES,
            AceVenturaEmptyHandler.INSTANCE);

        map.put(CollieChangeType.SHARE_LIST, new ShareListHandler(context));
        map.put(CollieChangeType.REVOKE_LIST, new RevokeListHandler(context));
        handlers = Collections.unmodifiableMap(map);
        logger = context.salo().logger();
    }

    @Override
    public void process(
        final EnvelopesContext context,
        final JsonList rows,
        final List<Envelope> storage)
        throws IOException, JsonException
    {
        int rowsSize = rows.size();
        if (rowsSize <= 0) {
            context.mdb().fetchLag(0L);
            return;
        }
        if (rows.size() < mdbsContext.salo().config().selectLength()) {
            logger.info("Using sequential process");
            for (int i = 0; i < rowsSize; ++i) {
                JsonMap row = rows.get(i).asMap();
                process(context, row, storage);
            }

            return;
        }
        logger.info("Using parallel process");

        long start = System.currentTimeMillis();

        List<Future<List<Envelope>>> futures = new ArrayList<>(rowsSize);
        for (int i = 0; i < rowsSize; ++i) {
            JsonMap row = rows.get(i).asMap();
            EnvelopesContext envelopesContext =
                new ThreadLocalEnvelopeContext(context.tokenString(), context.tokenVersion(), context.mdb());
            futures.add(mdbsContext.threadPool().submit(new AceChangeLogTask(row, this, envelopesContext)));
        }

        try {
            for (int i = 0; i < futures.size(); ++i) {
                storage.addAll(futures.get(i).get());
            }
        } catch (InterruptedException | ExecutionException ee) {
            for (int i = 0; i < futures.size(); ++i) {
                futures.get(i).cancel(true);
            }

            throw new IOException("Batch process failed", ee);
        }

        StringBuilder logSb = new StringBuilder();
        logSb.append("Batch size ");
        logSb.append(rowsSize);
        logSb.append(" processed in ");
        logSb.append((System.currentTimeMillis() - start));
        logSb.append("ms");
        logger.info(logSb.toString());
    }

    private class ThreadLocalEnvelopeContext extends EnvelopesContext {
        public ThreadLocalEnvelopeContext(
            final String tokenString,
            final long tokenVersion,
            final Mdb mdb)
        {
            super(tokenString, tokenVersion, mdb);
        }

        @Override
        public byte[] encode(final JsonObject json) throws IOException {
            StringBuilder sb = new StringBuilder();
            JsonWriter writer = new JsonWriter(new StringBuilderWriter(sb));
            json.writeValue(writer);
            CharsetEncoder charsetEncoder = mdbsContext.config().zoolooserConfig().requestCharset().newEncoder()
                .onMalformedInput(CodingErrorAction.REPLACE)
                .onUnmappableCharacter(CodingErrorAction.REPLACE);
            StringBuilderVoidProcessor<byte[], CharacterCodingException> encoder =
                 new StringBuilderVoidProcessor<>(
                    new Encoder(charsetEncoder));

            encoder.process(sb);
            return encoder.processWith(ByteArrayCopier.INSTANCE);
        }
    }

    private static final class AceChangeLogTask implements Callable<List<Envelope>> {
        private final JsonMap record;
        private final AceVenturaEnvelopesFactory factory;
        private final EnvelopesContext context;

        public AceChangeLogTask(
            final JsonMap record,
            final AceVenturaEnvelopesFactory factory,
            final EnvelopesContext context)
        {
            this.record = record;
            this.factory = factory;
            this.context = context;
        }

        @Override
        public List<Envelope> call() throws Exception {
            List<Envelope> envelopes = new ArrayList<>(DEFAULT_ENVELOPES_ARRAY_SIZE);
            factory.process(context, record, envelopes);
            return envelopes;
        }
    }

    @Override
    public void process(
        final EnvelopesContext context,
        final JsonMap json,
        final List<Envelope> storage)
        throws IOException, JsonException
    {
        AceVenturaIndexContext indexContext = null;
        try {
            double opDate = json.getDouble(CollieFields.OPERATION_DATE, 0.0);
            context.mdb().fetchLag(System.currentTimeMillis() - ((long) (opDate * 1000)));

            indexContext =
                new AceVenturaIndexContext(
                    mdbsContext,
                    context,
                    logger,
                    json);

            mdbsContext.changeTypeStat().accept(indexContext.changeType());

            AceVenturaIndexHandler handler =
                handlers.getOrDefault(
                    indexContext.changeType(),
                    UnknownAceVenturaHandler.INSTANCE);

            if (indexContext.prefix().userType() == UserType.CONNECT_ORGANIZATION
                    && BANNED_ORGS.contains(indexContext.prefix().uid()))
            {
                int length = JsonType.NORMAL.toString(indexContext.changed()).length();
                indexContext.logger().warning("Banned org " + indexContext.prefix() + " changed_size " + length + " " + indexContext.changeId());
                if (length > BAN_TH) {
                    handler = AceVenturaEmptyHandler.INSTANCE;
                    indexContext.logger().warning(
                        "Skipping change_id "+ indexContext.changeId() + " for org " + indexContext.prefix().uid());
                }
            }

            handler.handle(indexContext, storage);
        } catch (Exception e) {
            mdbsContext.errors().accept(1);
            if (indexContext != null) {
                throw new AceVenturaIndexException(
                    generateMessage(
                        indexContext,
                        e.getMessage()),
                    e);
            } else {
                throw new AceVenturaIndexException(
                    "Failed to create context from "
                        + JsonType.NORMAL.toString(json)
                        + " because of " + e.getMessage(),
                    e);
            }
        }
    }

    private static String generateMessage(
        final AceVenturaIndexContext context,
        final String message)
    {
        StringBuilder sb;
        if (message != null) {
            sb = new StringBuilder("Failed to index message context: ");
        } else {
            sb = new StringBuilder(message);
            sb.append(" with context: ");
        }

        sb.append(context.changeId());
        sb.append(" revision ");
        sb.append(context.revision());
        sb.append(" prefix ");
        sb.append(context.prefix());
        sb.append(" ");
        //sb.append(JsonType.NORMAL.toString(context.changed()));
        return sb.toString();
    }

    @Override
    public String scope() {
        return "contacts";
    }

    public AceVenturaIndexHandler handler(final CollieChangeType changeType) {
        return handlers.get(changeType);
    }

    @Override
    public ContainerFactory jsonContainerFactory() {
        return PositionSavingContainerFactory.INSTANCE;
    }
}
