package ru.yandex.ace.ventura.salo;

import java.io.IOException;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicLong;

import org.apache.http.HttpHost;
import org.apache.http.HttpStatus;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.message.BasicHttpRequest;

import ru.yandex.ace.ventura.AceVenturaPrefix;
import ru.yandex.ace.ventura.UserType;
import ru.yandex.dbfields.CollieFields;
import ru.yandex.function.BasicGenericConsumer;
import ru.yandex.function.CharArrayProcessable;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.EmptyFutureCallback;
import ru.yandex.http.util.YandexHttpStatus;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.PositionSavingContainerFactory;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.JsonParser;
import ru.yandex.json.parser.KeyInterningStringCollectorsFactory;
import ru.yandex.json.parser.StackContentHandler;
import ru.yandex.json.writer.JsonType;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.search.salo.EnvelopesContext;

public class AceVenturaIndexContext {
    private final CollieChangeType changeType;
    private final long revision;
    private final AceVenturaPrefix prefix;
    private final long changeId;
    private final double opDate;
    private final double selectDate;
    private final JsonMap changed;
    private final JsonMap arguments;
    private final EnvelopesContext envelopesContext;
    private final PrefixedLogger logger;
    private final CloseableHttpClient msalClient;
    private final AsyncClient asyncMsalClient;
    private final HttpHost msalHost;
    private final boolean skipMalformedUser;
    private final AtomicLong nextEnvelopeOffset = new AtomicLong(0);

    /**
     Here we have a contract what kind of data contain fields changed and arguments
     It matters in update_* change types. Arguments fields contains only new state field,
     so, for example, if we moved contact from one list to another in arguments field will be
     new list_id but other fields are null. Changed field contains the old data for the record
     So we should merge arguments and changed, with priority for the arguments not null fields
     **/
    public AceVenturaIndexContext(
        final AceVenturaMdbsContext mdbsContext,
        final EnvelopesContext envelopesContext,
        final PrefixedLogger logger,
        final JsonMap json)
        throws JsonException
    {
        this.changed = json.getMap("changed");
        this.arguments = json.getMap("arguments");
        this.changeType =
            json.getEnum(CollieChangeType.class, "change_type");
        Long userId  = json.getLong(CollieFields.USER_ID);
        UserType userType =
            json.getEnum(UserType.class, CollieFields.USER_TYPE);
        this.prefix = new AceVenturaPrefix(userId, userType);
        this.revision = json.getLong(CollieFields.REVISION);
        this.changeId = json.getLong(CollieFields.OPERATION_ID);
        this.opDate = json.getDouble(CollieFields.OPERATION_DATE);
        this.selectDate = json.getDouble(CollieFields.SELECT_DATE);
        this.envelopesContext = envelopesContext;
        this.logger = logger;
        this.msalClient = mdbsContext.msalClient();
        this.msalHost = mdbsContext.msalConfig().host();
        this.asyncMsalClient = ((AceVenturaSalo) mdbsContext.salo()).asyncMsalClient();
        this.skipMalformedUser = ((AceVenturaSalo) mdbsContext.salo()).aceSaloConfig().skipMalformedUser();
    }

    public AceVenturaIndexContext(
        final AceVenturaIndexContext context,
        final JsonMap changed,
        final JsonMap arguments)
    {
        this.changed = changed;
        this.arguments = arguments;
        this.prefix = context.prefix;
        this.revision = context.revision;
        this.changeId = context.changeId;
        this.opDate = context.opDate;
        this.selectDate = context.selectDate;
        this.envelopesContext = context.envelopesContext;
        this.logger = context.logger;
        this.msalClient = context.msalClient;
        this.msalHost = context.msalHost;
        this.changeType = context.changeType;
        this.asyncMsalClient = context.asyncMsalClient;
        this.skipMalformedUser = context.skipMalformedUser;
    }

    public CollieChangeType changeType() {
        return changeType;
    }

    public long revision() {
        return revision;
    }

    public AceVenturaPrefix prefix() {
        return prefix;
    }

    public long changeId() {
        return changeId;
    }

    public JsonMap changed() {
        return changed;
    }

    public EnvelopesContext envelopesContext() {
        return envelopesContext;
    }

    public PrefixedLogger logger() {
        return logger;
    }

    public double opDate() {
        return opDate;
    }

    public double selectDate() {
        return selectDate;
    }

    public JsonMap arguments() {
        return arguments;
    }

    public CloseableHttpClient msalClient() {
        return msalClient;
    }

    public HttpHost msalHost() {
        return msalHost;
    }

    public Future<JsonObject> asyncMsalRequest(
        final AceVenturaPrefix owner,
        final StringBuilder uri)
    {
        return asyncMsalRequest(owner, uri, EmptyFutureCallback.INSTANCE);
    }

    public Future<JsonObject> asyncMsalRequest(
        final AceVenturaPrefix owner,
        final StringBuilder uri,
        final FutureCallback<? super JsonObject> callback)
    {
        uri.append("&uid=");
        uri.append(owner.uid());
        uri.append("&userType=");
        uri.append(owner.userType());
        return asyncMsalClient.execute(
            msalHost,
            new BasicAsyncRequestProducerGenerator(uri.toString()),
            JsonAsyncTypesafeDomConsumerFactory.OK,
            callback);
    }

    public JsonList msalRequest(
        final AceVenturaPrefix owner,
        final StringBuilder uri)
        throws IOException
    {
        uri.append("&uid=");
        uri.append(owner.uid());
        uri.append("&userType=");
        uri.append(owner.userType());
        return msalRequest(uri.toString());
    }

    public JsonList msalRequest(
        final String uri)
        throws IOException
    {
        logger.fine("Requesting msal for " + uri);
        int status = YandexHttpStatus.SC_CLIENT_CLOSED_REQUEST;
        long start = System.currentTimeMillis();
        int maxRetries = 3;
        int retries = 0;

        while (true) {
            try (CloseableHttpResponse response
                     = msalClient.execute(
                msalHost,
                new BasicHttpRequest(HttpGet.METHOD_NAME, uri)))
            {
                CharArrayProcessable data = CharsetUtils.toDecoder(response.getEntity());

                if (response.getStatusLine().getStatusCode() != HttpStatus.SC_OK) {
                    int statusCode = response.getStatusLine().getStatusCode();
                    String dataStr = String.valueOf(data);

                    if (statusCode == HttpStatus.SC_NOT_FOUND
                        && dataStr.contains("uid not found"))
                    {
                        logger.warning("User not found " + uri);
                        return JsonList.EMPTY;
                    }

                    if (skipMalformedUser
                        && statusCode == HttpStatus.SC_INTERNAL_SERVER_ERROR
                        && (dataStr.contains("user registration already in progress")
                        || dataStr.contains("user already registered")))
                    {
                        logger.warning("Skipping malformed user " + dataStr);
                        return JsonList.EMPTY;
                    }

                    retries += 1;
                    if (retries < maxRetries) {
                        continue;
                    }
                }

                BasicGenericConsumer<JsonObject, JsonException>
                    consumer =
                    new BasicGenericConsumer<>();
                try {
                    data.processWith(
                        new JsonParser(
                            new StackContentHandler(
                                new TypesafeValueContentHandler(
                                    consumer,
                                    KeyInterningStringCollectorsFactory.INSTANCE
                                        .apply(data.length()),
                                    PositionSavingContainerFactory.INSTANCE))));
                    return consumer.get().asList();
                } catch (JsonException je) {
                    throw new AceVenturaIndexException(
                        "Bad msal response " + String.valueOf(data) + " " + uri,
                        je);
                }
            } catch (Throwable e) {
                throw new AceVenturaIndexException(
                    "Msal request failed " + uri,
                    e);
            } finally {
                this.envelopesContext.mdb().context().queueResponse(status, start);
            }
        }
    }

    public long nextEnvelopeOffset() {
        return nextEnvelopeOffset.getAndIncrement();
    }

    @Override
    public String toString() {
        return "AceVenturaIndexContext{" +
            "changeType=" + changeType +
            ", revision=" + revision +
            ", prefix=" + prefix +
            ", changeId=" + changeId +
            ", changed=" + JsonType.NORMAL.toString(changed) +
            ", arguments=" + JsonType.NORMAL.toString(arguments) +
            '}';
    }
}
