package ru.yandex.ace.ventura.salo;

import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.concurrent.ThreadPoolExecutor;

import org.apache.http.impl.client.CloseableHttpClient;

import ru.yandex.ace.ventura.salo.config.ImmutableAceVenturaSaloConfig;
import ru.yandex.ace.ventura.salo.config.ImmutableMdbConfig;
import ru.yandex.ace.ventura.salo.handlers2.HumanNamesUtil;
import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.http.util.client.ClientBuilder;
import ru.yandex.search.salo.MdbsContext;
import ru.yandex.search.salo.Salo;
import ru.yandex.stater.CountAggregatorFactory;
import ru.yandex.stater.DuplexStaterFactory;
import ru.yandex.stater.EnumStaterFactory;
import ru.yandex.stater.IntegralSumAggregatorFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;

public class AceVenturaMdbsContext extends MdbsContext {
    private final TimeFrameQueue<Integer> emailParseErrorStat;
    private final TimeFrameQueue<CollieChangeType> changeTypeStat;
    private final TimeFrameQueue<Integer> errors;
    private final ImmutableMdbConfig config;
    private final CloseableHttpClient msalClient;
    private final HumanNamesUtil humanNamesUtil;
    private final TimeFrameQueue<Long> badUsers;

    public AceVenturaMdbsContext(
        final Salo salo,
        final ThreadGroup threadGroup,
        final ImmutableMdbConfig config)
    {
        super(salo, threadGroup, config.name());

        emailParseErrorStat =
            new TimeFrameQueue<>(salo.config().metricsTimeFrame());
        salo.registerStater(
            new PassiveStaterAdapter<>(
                emailParseErrorStat,
                new DuplexStaterFactory<>(
                    new NamedStatsAggregatorFactory<>(
                        config.name() + "-email-parse-total_ammm",
                        CountAggregatorFactory.INSTANCE),
                    new NamedStatsAggregatorFactory<>(
                        config.name() + "-email-parse-errors_ammm",
                        IntegralSumAggregatorFactory.INSTANCE))));

        changeTypeStat =
            new TimeFrameQueue<>(salo.config().metricsTimeFrame());
        salo.registerStater(
            new PassiveStaterAdapter<>(
                changeTypeStat,
                new EnumStaterFactory<>(
                    type -> config.name() + "-change-type-" + type + "_ammm",
                    CollieChangeType.values())));

        errors =
            new TimeFrameQueue<>(salo.config().metricsTimeFrame());
        salo.registerStater(
            new PassiveStaterAdapter<>(
                errors,
                new NamedStatsAggregatorFactory<>(
                    config.name() + "-envelopes-process-errors_ammm",
                    IntegralSumAggregatorFactory.INSTANCE)));

        badUsers = new TimeFrameQueue<>(salo.config().metricsTimeFrame());
        salo.registerStater(
            new PassiveStaterAdapter<>(
                badUsers,
                new NamedStatsAggregatorFactory<>(
                    config.name() + "-bad-users_ammm",
                    IntegralSumAggregatorFactory.INSTANCE)));

        this.config = config;
        this.msalClient =
            ClientBuilder.createClient(
                config.msalConfig(),
                salo.config().dnsConfig());

        try {
            humanNamesUtil =
                new HumanNamesUtil(
                    new InputStreamReader(
                        this.getClass()
                            .getResourceAsStream("human.aliases.json"),
                        StandardCharsets.UTF_8));
        } catch (IOException ioe) {
            throw new RuntimeException(ioe);
        }

        if (!(salo.config() instanceof ImmutableAceVenturaSaloConfig)) {
            throw new RuntimeException("Bad salo instance");
        }
    }

    public TimeFrameQueue<Integer> emailParseStat() {
        return emailParseErrorStat;
    }

    public TimeFrameQueue<CollieChangeType> changeTypeStat() {
        return changeTypeStat;
    }

    @Override
    public CloseableHttpClient msalClient() {
        return msalClient;
    }

    @Override
    public ImmutableHttpHostConfig msalConfig() {
        return config.msalConfig();
    }

    public HumanNamesUtil humanNamesUtil() {
        return humanNamesUtil;
    }

    public TimeFrameQueue<Integer> errors() {
        return errors;
    }

    public TimeFrameQueue<Long> badUsers() {
        return badUsers;
    }

    public ImmutableMdbConfig mdbConfig() {
        return config;
    }

    public ThreadPoolExecutor threadPool() {
        return ((AceVenturaSalo) salo()).threadPool();
    }
}
