package ru.yandex.ace.ventura.salo;

import java.io.IOException;
import java.io.InputStreamReader;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Random;
import java.util.Set;
import java.util.TreeMap;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import org.apache.http.HttpException;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.entity.StringEntity;
import org.apache.http.message.BasicHttpRequest;
import org.apache.http.protocol.HttpContext;
import org.apache.http.protocol.HttpRequestHandler;

import ru.yandex.ace.ventura.AceVenturaPrefix;
import ru.yandex.ace.ventura.UserType;
import ru.yandex.ace.ventura.salo.config.ImmutableAceVenturaSaloConfig;
import ru.yandex.ace.ventura.salo.config.ImmutableMdbConfig;
import ru.yandex.ace.ventura.salo.handlers2.HumanNamesUtil;
import ru.yandex.ace.ventura.salo.handlers2.reindex.ReindexOrganizationOperation;
import ru.yandex.ace.ventura.salo.handlers2.reindex.ReindexUserOperation;
import ru.yandex.blackbox.BlackboxClient;
import ru.yandex.client.tvm2.Tvm2TicketRenewalTask;
import ru.yandex.collection.Pattern;
import ru.yandex.concurrent.LifoWaitBlockingQueue;
import ru.yandex.concurrent.NamedThreadFactory;
import ru.yandex.dispatcher.producer.Producer;
import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.http.server.sync.HttpSession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.BadResponseException;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.HttpHostParser;
import ru.yandex.http.util.YandexHttpStatus;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.SharedConnectingIOReactor;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.http.util.request.RequestInfo;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.multistarter.MultiStarter;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.parser.uri.ScanningCgiParams;
import ru.yandex.search.salo.Mdb;
import ru.yandex.search.salo.Salo;

public class AceVenturaSalo extends Salo implements HttpRequestHandler {
    private static final int MAX_WORKERS = 150;
    private static final int QUEUE_SIZE = 100;
    private static final int CORE_WORKERS = MAX_WORKERS;
    private static final int THREAD_POOL_KA = 5;

    private static final int STAFF_MILLIS_EXPIRES_FIXED =
        (int) TimeUnit.HOURS.toMillis(12);
    private static final int STAFF_MILLIS_EXPIRE_WINDOW =
        (int) TimeUnit.HOURS.toMillis(12);
    private static final int STAFF_MAX_SIZE = 100000;
    private static final int STAFF_LOGIN_MAX_SIZE = 100000;

    private final List<MdbProvider> providers;
    private final HumanNamesUtil humanNamesUtil;
    private final BlackboxClient bpBlackboxClient;
    private final BlackboxClient corpBlackboxClient;
    private final SharedConnectingIOReactor reactor;
    private final AsyncClient mailProxyClient;
    private final AsyncClient aceventuraProxyClient;
    private final Tvm2TicketRenewalTask tvm2RenewalTask;
    private final ImmutableAceVenturaSaloConfig aceSaloConfig;
    private final AsyncClient asyncMsalClient;
    private final AsyncClient staffProxyClient;
    private final ThreadPoolExecutor threadPool;
    private final Cache<String, Map.Entry<Long, JsonObject>> staffDataCache;
    private final Cache<String, String> staffUserLoginCache;

    public AceVenturaSalo(
        final ImmutableAceVenturaSaloConfig config)
        throws GeneralSecurityException,
        HttpException,
        IOException,
        JsonException,
        URISyntaxException
    {
        super(config);

        this.aceSaloConfig = config;
        providers = new ArrayList<>();
        for (ImmutableMdbConfig providerConfig: config.providers()) {
            providers.add(
                providerConfig.mdbProviderType().create(
                    this,
                    providerConfig));
        }

        reactor = new SharedConnectingIOReactor(config, config.dnsConfig());
        bpBlackboxClient =
            new BlackboxClient(reactor, config.blackboxConfig())
                .adjustStater(
                    config.bpBlackboxStatersConfig(),
                    new RequestInfo(
                        new BasicHttpRequest(RequestHandlerMapper.GET, "/userinfo")));
        registerStaters(config.bpBlackboxStatersConfig());

        corpBlackboxClient = new BlackboxClient(reactor, config.corpBlackboxConfig())
            .adjustStater(
                config.corpBlackboxStatersConfig(),
                new RequestInfo(
                    new BasicHttpRequest(RequestHandlerMapper.GET, "/userinfo")));
        registerStaters(config.corpBlackboxStatersConfig());

        mailProxyClient = new AsyncClient(reactor, config.mailProxyConfig());
        aceventuraProxyClient = new AsyncClient(reactor, config.aceventuraProxyConfig());
        asyncMsalClient = new AsyncClient(reactor, config.msalConfig());
        staffProxyClient = new AsyncClient(reactor, config.staffConfig());

        tvm2RenewalTask = new Tvm2TicketRenewalTask(
            logger().addPrefix("tvm2"),
            serviceContextRenewalTask,
            config.tvm2ClientConfig());

        register(
            new Pattern<>("/drop-position", false),
            new DropPositionHandler(),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/reindex-user", false),
            new ReindexUserHandler(),
            RequestHandlerMapper.GET);
        register(
            new Pattern<>("/reindex-tags", false),
            new ReindexTagsHandler(),
            RequestHandlerMapper.GET);

        logger().info(
            "We depending on multistarter"
                + MultiStarter.class.getSimpleName());
        logger().info(
            "We depending on producer"
                + Producer.class.getSimpleName());

        humanNamesUtil =
            new HumanNamesUtil(
                new InputStreamReader(
                    this.getClass()
                        .getResourceAsStream("human.aliases.json"),
                    StandardCharsets.UTF_8));

        threadPool = new ThreadPoolExecutor(
            CORE_WORKERS,
            MAX_WORKERS,
            THREAD_POOL_KA,
            TimeUnit.MINUTES,
            new LifoWaitBlockingQueue<>(QUEUE_SIZE),
            new NamedThreadFactory(
                getThreadGroup(),
                 "EnvelopeGenWorker-",
                true),
            new ThreadPoolExecutor.CallerRunsPolicy());

        this.staffDataCache =
            CacheBuilder.newBuilder()
                .maximumSize(STAFF_MAX_SIZE)
                .concurrencyLevel(config.workers()).build();

        this.staffUserLoginCache =
            CacheBuilder.newBuilder()
                .maximumSize(STAFF_LOGIN_MAX_SIZE)
                .concurrencyLevel(config.workers()).build();
    }

    public AsyncClient asyncMsalClient() {
        return asyncMsalClient;
    }

    public List<MdbProvider> providers() {
        return new ArrayList<>(providers);
    }

    public HumanNamesUtil humanNamesUtil() {
        return humanNamesUtil;
    }

    public BlackboxClient bpBlackboxClient() {
        return bpBlackboxClient;
    }

    public BlackboxClient corpBlackboxClient() {
        return corpBlackboxClient;
    }

    public AsyncClient aceventuraProxyClient() {
        return aceventuraProxyClient;
    }

    public HttpHost aceventuraProxyHost() {
        return aceSaloConfig.aceventuraProxyConfig().host();
    }

    public AsyncClient mailProxyClient() {
        return mailProxyClient;
    }

    public HttpHost mailProxyHost() {
        return aceSaloConfig.mailProxyConfig().host();
    }

    public ImmutableAceVenturaSaloConfig aceSaloConfig() {
        return aceSaloConfig;
    }

    public AsyncClient staffProxyClient() {
        return staffProxyClient;
    }

    public JsonObject getCachedStaffData(final String login) {
        Map.Entry<Long, JsonObject> value = staffDataCache.getIfPresent(login);
        if (value == null) {
            return null;
        }

        if (value.getKey() - System.currentTimeMillis() > 0) {
            return value.getValue();
        } else {
            staffDataCache.invalidate(login);
            return null;
        }
    }

    public void putCachedStaffData(final String login, final JsonObject data) {
        long expTs =
            System.currentTimeMillis()
                + STAFF_MILLIS_EXPIRES_FIXED
                + new Random().nextInt(STAFF_MILLIS_EXPIRE_WINDOW);
        Map.Entry<Long, JsonObject> value = new AbstractMap.SimpleEntry<>(expTs, data);
        staffDataCache.put(login, value);
    }

    public Cache<String, String> staffUserLoginCache() {
        return staffUserLoginCache;
    }

    @Override
    public void handle(
        final HttpRequest request,
        final HttpResponse response,
        final HttpContext context)
        throws HttpException, IOException
    {
    }

    @Override
    public void start() throws IOException {
        reactor.start();
        bpBlackboxClient.start();
        corpBlackboxClient.start();
        tvm2RenewalTask.start();
        mailProxyClient.start();
        aceventuraProxyClient.start();
        asyncMsalClient.start();
        staffProxyClient.start();

        for (MdbProvider provider: providers) {
            provider.start();
        }
        super.start();
    }

    @Override
    @SuppressWarnings("try")
    public void close() throws IOException {
        super.close();

        for (MdbProvider provider : providers) {
            provider.close();
        }

        threadPool.shutdown();
        reactor.close();
        bpBlackboxClient.close();
        corpBlackboxClient.close();
        mailProxyClient.close();
        asyncMsalClient.close();
        tvm2RenewalTask.cancel();
    }

    @Override
    public Map<String, Object> status(final boolean verbose) {
        Map<String, Object> status = super.status(verbose);
        if (verbose) {
            Map<String, Object> mdbsStatus = new TreeMap<>();
            for (MdbProvider provider: providers) {
                mdbsStatus.put(provider.name(), provider.status());
            }
            status.put("mdbs", mdbsStatus);
            Map<String, Object> threadPoolStatus = new LinkedHashMap<>();
            threadPoolStatus.put("queue_size", threadPool.getQueue().size());
            threadPoolStatus.put("active_count", threadPool.getActiveCount());
            threadPoolStatus.put("pool_size", threadPool.getPoolSize());
            threadPoolStatus.put("terminated", threadPool.isTerminated());
            status.put("async_pool", threadPoolStatus);
        }
        return status;
    }

    public ThreadPoolExecutor threadPool() {
        return threadPool;
    }

    public String blackboxTvm2Ticket(final boolean corp) {
        if (corp) {
            return tvm2RenewalTask.ticket(aceSaloConfig.corpBlackboxClientId());
        } else {
            return tvm2RenewalTask.ticket(aceSaloConfig.bpBlackboxClientId());
        }
    }
    public String msearchProxyTvm2Ticket() {
        return tvm2RenewalTask.ticket(aceSaloConfig.mailProxyClientId());
    }

    private class DropPositionHandler implements HttpRequestHandler {
        private DropPositionHandler() {
        }

        @Override
        public void handle(
            final HttpRequest request,
            final HttpResponse response,
            final HttpContext context)
            throws HttpException
        {
            ScanningCgiParams params = new ScanningCgiParams(request);
            String providerName = params.get(
                "type",
                null,
                NonEmptyValidator.INSTANCE);

            String shardName = params.get(
                "shard",
                null,
                NonEmptyValidator.INSTANCE);
            for (MdbProvider provider: providers) {
                if (provider.name().equalsIgnoreCase(providerName)) {
                    provider.dropPosition(shardName);
                    response.setStatusCode(HttpStatus.SC_OK);
                    return;
                }
            }

            response.setStatusCode(HttpStatus.SC_NOT_FOUND);
        }
    }

    private class ReindexUserHandler implements HttpRequestHandler {
        private ReindexUserHandler() {
        }

        @Override
        public void handle(
            final HttpRequest request,
            final HttpResponse response,
            final HttpContext context)
            throws HttpException
        {
            HttpSession session = new HttpSession(request, response, context);
            ScanningCgiParams params = new ScanningCgiParams(request);
            UserType userType = params.getEnum(
                UserType.class,
                "user_type",
                UserType.PASSPORT_USER);
            String providerName = params.get(
                "provider",
                null,
                NonEmptyValidator.INSTANCE);

            boolean shared = params.getBoolean("shared", false);

            Long uid = params.getLong("uid", null);
            if (uid == null) {
                uid = params.getLong("user_id");
            }

            AceVenturaPrefix prefix = new AceVenturaPrefix(uid, userType);
            Optional<MdbProvider> providerOpt =
                providers.stream()
                    .filter((p) -> p.name().equalsIgnoreCase(providerName))
                    .findFirst();

            if (!providerOpt.isPresent() || !(providerOpt.get() instanceof AceVenturaMdbProvider)) {
                throw new BadRequestException(
                    "Provider not found or not aceventura one" + providerName);
            }

            AceVenturaMdbProvider provider = (AceVenturaMdbProvider) providerOpt.get();

            ImmutableHttpHostConfig msalConfig = provider.config().msalConfig();

            StringBuilder msalUri = new StringBuilder(msalConfig.host().toString());
            msalUri.append("/get-user-shard?");

            if (userType == UserType.CONNECT_ORGANIZATION) {
                msalUri.append("&orgId=");
            } else {
                msalUri.append("&uid=");
            }

            msalUri.append(uid);

            String shard;
            try (CloseableHttpResponse shardResponse =
                     msalClient().execute(msalConfig.host(), new HttpGet(msalUri.toString())))
            {
                int status = shardResponse.getStatusLine().getStatusCode();
                if (status != YandexHttpStatus.SC_OK) {
                    throw new BadResponseException(msalUri, shardResponse);
                }

                String responseStr = CharsetUtils.toString(shardResponse.getEntity());
                session.logger().info("Resp str " + responseStr);
                shard = responseStr.trim();
            } catch (IOException | HttpException e) {
                throw new BadRequestException(e);
            }

            session.logger().info("Shard found " + shard);
            Mdb mdb = provider.get(shard);
            if (mdb == null || !mdb.locked()) {
                Set<HttpHost> hosts = new LinkedHashSet<>(provider.shardMap().saloHosts(shard));
                Set<HttpHost> excludeHosts =
                        session.params().getAll(
                        "exclude_host",
                        Collections.emptySet(),
                        new CollectionParser<>(HttpHostParser.INSTANCE, LinkedHashSet::new));
                excludeHosts = new LinkedHashSet<>(excludeHosts);

                hosts.removeAll(excludeHosts);
                if (hosts.isEmpty()) {
                    session.response().setStatusCode(HttpStatus.SC_NOT_FOUND);
                    session.response().setEntity(
                        new StringEntity("{\"shard\": \"" + shard + "\"}", StandardCharsets.UTF_8));
                    logger.warning("Mdb not found or not locked " + shard);
                } else {
                    HttpHost host = hosts.iterator().next();
                    excludeHosts.add(host);

                    session.params().put(
                        "exclude_host",
                        excludeHosts.stream()
                            .map(HttpHost::toHostString)
                            .collect(Collectors.toList()));
                    String uri = request.getRequestLine().getUri();
                    int paramIndex = uri.indexOf('?');
                    uri = uri.substring(0, paramIndex + 1);
                    QueryConstructor qc = new QueryConstructor(host.toString() + uri);
                    for (Map.Entry<String, List<String>> param: session.params().entrySet()) {
                        for (String value: param.getValue()) {
                            qc.append(param.getKey(), value);
                        }
                    }

                    String location = qc.toString();
                    session.logger().info(
                        "Redirecting "
                            + request.getRequestLine().getUri()
                            + " to " + location);
                   session.response().addHeader(
                        HttpHeaders.LOCATION,
                        location);
                    session.response().setStatusCode(YandexHttpStatus.SC_TEMPORARY_REDIRECT);

                }

                return;
            }

            try {
                if (shared || prefix.userType() == UserType.CONNECT_ORGANIZATION) {
                    mdb.exectueWithLock(new ReindexOrganizationOperation(prefix, provider, session.logger()));
                } else {
                    mdb.exectueWithLock(new ReindexUserOperation(prefix, provider, session.logger()));
                }
            } catch (HttpException e) {
                throw e;
            } catch (Exception e) {
                throw new BadRequestException(e);
            }

            session.response().setStatusCode(HttpStatus.SC_OK);
        }
    }
}

