package ru.yandex.ace.ventura.salo;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicLong;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.message.BasicHttpRequest;

import ru.yandex.ace.ventura.salo.config.ImmutableMdbConfig;
import ru.yandex.ace.ventura.salo.mdb.EmptyShardMap;
import ru.yandex.ace.ventura.salo.mdb.MdbShardsSupplier;
import ru.yandex.ace.ventura.salo.mdb.SaloShardMap;
import ru.yandex.ace.ventura.salo.mdb.SaloShardsMapFactory;
import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.HttpExceptionConverter;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.search.salo.Mdb;
import ru.yandex.search.salo.Salo;
import ru.yandex.stater.IntegralSumAggregatorFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;
import ru.yandex.stater.Stater;
import ru.yandex.stater.StatsConsumer;

public abstract class PgMdbProvider
    implements GenericAutoCloseable<IOException>, Runnable, MdbProvider
{
    private final Map<String, Mdb> mdbs = new ConcurrentHashMap<>();

    private final long interval;
    private final PrefixedLogger logger;
    private final Thread thread;
    private final String name;
    private final MdbShardsSupplier shardsSupplier;
    private final SaloShardsMapFactory shardsMapFactory;
    private final TimeFrameQueue<Integer> mdbError;
    private final TimeFrameQueue<Integer> mdbDelete;
    private final TimeFrameQueue<Integer> mdbAdd;
    private final AtomicLong lastUpdateTs
        = new AtomicLong(System.currentTimeMillis());

    protected final ImmutableMdbConfig config;
    private final Salo salo;
    private volatile boolean close = false;
    private volatile SaloShardMap shardMap;

    protected PgMdbProvider(
        final Salo salo,
        final ImmutableMdbConfig config)
    {
        this.salo = salo;
        this.config = config;
        this.interval = config.updateInterval();
        this.logger = salo.logger().addPrefix(config.name());
        this.name = config.name();
        mdbError = new TimeFrameQueue<>(salo.config().metricsTimeFrame());
        salo.registerStater(
            new PassiveStaterAdapter<>(
                mdbError,
                new NamedStatsAggregatorFactory<>(
                    name + "-mdb-start-error_ammm",
                    IntegralSumAggregatorFactory.INSTANCE)));

        mdbAdd = new TimeFrameQueue<>(salo.config().metricsTimeFrame());
        salo.registerStater(
            new PassiveStaterAdapter<>(
                mdbAdd,
                new NamedStatsAggregatorFactory<>(
                    name + "-mdb-added_ammm",
                    IntegralSumAggregatorFactory.INSTANCE)));

        mdbDelete = new TimeFrameQueue<>(salo.config().metricsTimeFrame());
        salo.registerStater(
            new PassiveStaterAdapter<>(
                mdbDelete,
                new NamedStatsAggregatorFactory<>(
                    name + "-mdb-deleted_ammm",
                    IntegralSumAggregatorFactory.INSTANCE)));

        salo.registerStater(new LastUpdateStater(name));

        this.thread = new Thread(
            new ThreadGroup(
                salo.getThreadGroup(),
                "MdbUpdater-" + name),
            this);
        this.shardsSupplier =
            config.shardsUpdater().create(salo, config.msalConfig());
        this.shardsMapFactory = config.shardsMapFactory();
        this.shardMap = EmptyShardMap.INSTANCE;

        salo.registerStater(new MdbsStater());
    }

    @Override
    public String name() {
        return name;
    }

    @Override
    public void start() {
        this.thread.start();
    }

    @Override
    public Map<String, Object> status() {
        Map<String, Object> status = new LinkedHashMap<>();
        Collection<Mdb> mdbs = new ArrayList<>(this.mdbs.values());
        Map<String, Object> mdbsStatus = new TreeMap<>();
        for (Mdb mdb: mdbs) {
            mdbsStatus.put(mdb.name(), mdb.status());
        }
        status.put("mdbs", mdbsStatus);

        return status;
    }

    @Override
    public void dropPosition(final String name) throws HttpException {
        Mdb mdb = mdbs.get(name);
        if (mdb == null) {
            throw new BadRequestException(
                "Mdb not found " + name + " in provider " + name());
        }

        HttpRequest producerRequest = new BasicHttpRequest(
            HttpGet.METHOD_NAME,
            "/_producer_drop_position?service=" + mdb.service()
                + "&producer-name=" + mdb.name()
                + "&positions-count=" + mdb.workersCount()
                + "&session-timeout=" + mdb.config().sessionTimeout());
        try (CloseableHttpResponse producerResponse =
                 salo.zoolooserClient().execute(
                     mdb.config().zoolooserConfig().host(),
                     producerRequest))
        {
            int status = producerResponse.getStatusLine().getStatusCode();
            if (status == HttpStatus.SC_OK) {
                synchronized (this) {
                    mdb = mdbs.remove(name);
                    mdb.interrupt();
                }
            } else {
                throw HttpExceptionConverter.toHttpException(
                    producerRequest,
                    producerResponse);
            }
        } catch (IOException e) {
            throw HttpExceptionConverter.toHttpException(e);
        }
    }

    protected void startMdbs(final Set<String> names) {
        for (String name: names) {
            Mdb mdb;
            try {
                mdb = apply(name);
            } catch (IOException ioe) {
                logger.log(Level.SEVERE, "Failed to create mdb", ioe);
                mdbError.accept(1);
                continue;
            }

            mdbAdd.accept(1);
            logger.info("Starting new mdb " + name);
            mdbs.put(name, mdb);
            mdb.start();
        }
    }

    protected Mdb get(final String name) {
        return mdbs.get(name);
    }

    protected void closeMdbs(
        final Set<String> deleted)
        throws InterruptedException
    {
        List<Mdb> deletedMdbs = new ArrayList<>();
        for (String name: deleted) {
            Mdb mdb = mdbs.remove(name);
            mdbDelete.accept(1);
            deletedMdbs.add(mdb);
            logger.info("closing " + name);
            mdb.close();
        }

        logger.info("Waiting for closed mdb to stop");
        for (Mdb mdb: deletedMdbs) {
            mdb.join();
        }

        logger.info("Deleted mdbs stopped");
    }

    @Override
    public void run() {
        try {
            while (!close) {
                logger.info("Fetching mdbs ");
                Set<String> fetched;
                try {
                    fetched = shardsSupplier.get();
                } catch (IOException ioe) {
                    logger.log(Level.WARNING, "Failed to update mdb list", ioe);
                    Thread.sleep(interval);
                    continue;
                }

                lastUpdateTs.set(System.currentTimeMillis());

                logger.info("Fetched mdbs " + fetched);

                SaloShardMap shardMap = shardsMapFactory.create(fetched);
                this.shardMap = shardMap;

                fetched = shardMap.mdbs();
                logger.info("After filtering shards " + fetched);

                synchronized (this) {
                    Set<String> deleted = new LinkedHashSet<>(mdbs.keySet());
                    deleted.removeAll(fetched);
                    if (!deleted.isEmpty()) {
                        logger.info("Mdbs for delete " + deleted);
                        closeMdbs(deleted);
                    }

                    Set<String> newOnes = new LinkedHashSet<>(fetched);
                    newOnes.removeAll(mdbs.keySet());

                    if (!newOnes.isEmpty()) {
                        logger.info("New mdbs to consume " + newOnes);
                        startMdbs(newOnes);
                    }
                }

                Thread.sleep(interval);
            }
        } catch (InterruptedException ie) {
            logger.log(Level.WARNING, "Mdb update interrupted");
            return;
        }

        try {
            closeMdbs(mdbs.keySet());
        } catch (InterruptedException ie) {
        }

    }

    @Override
    public void close() throws IOException {
        this.close = true;
        this.thread.interrupt();
        try {
            this.thread.join();
        } catch (InterruptedException ie) {
        }
    }

    private final class LastUpdateStater implements Stater {
        private final String name;

        private LastUpdateStater(final String name) {
            this.name = name + "-mdb-last-update-time-passed_axxx";
        }

        @Override
        public <E extends Exception> void stats(
            final StatsConsumer<? extends E> statsConsumer)
            throws E
        {
            statsConsumer.stat(
                name,
                System.currentTimeMillis() - lastUpdateTs.get());
        }
    }

    private class MdbsStater implements Stater {
        @Override
        public <E extends Exception> void stats(
            final StatsConsumer<? extends E> statsConsumer)
            throws E
        {
            int lockedShards = 0;
            int pinholes = 0;
            long maxTransferLag = 0;
            long maxLag = 0;
            for (Mdb mdb: mdbs.values()) {
                if (mdb.locked()) {
                    ++lockedShards;
                }
                if (mdb.pinhole()) {
                    ++pinholes;
                }

                long lastTransferLag = mdb.lastTransferLag();
                if (lastTransferLag > maxTransferLag) {
                    maxTransferLag = lastTransferLag;
                }

                long transferLag = mdb.transferLag();
                if (transferLag > maxLag) {
                    maxLag = transferLag;
                }

                long fetchLag = mdb.fetchLag();
                if (fetchLag > maxLag) {
                    maxLag = fetchLag;
                }
            }
            statsConsumer.stat(name() + "-locked-shards_ammv", lockedShards);
            statsConsumer.stat(name() + "-pinholes_ammv", pinholes);
            statsConsumer.stat(name() + "-max-lag_axxx", maxLag);
            statsConsumer.stat(name() + "-last-transfer-lag_axxx", maxTransferLag);
        }
    }

    @Override
    public ImmutableMdbConfig config() {
        return config;
    }

    @Override
    public SaloShardMap shardMap() {
        return shardMap;
    }
}
