package ru.yandex.ace.ventura.salo.handlers2;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.logging.Level;
import java.util.regex.Pattern;

import ru.yandex.ace.ventura.AceVenturaFields;
import ru.yandex.ace.ventura.AceVenturaPrefix;
import ru.yandex.ace.ventura.AceVenturaRecordType;
import ru.yandex.ace.ventura.ListType;
import ru.yandex.ace.ventura.UserType;
import ru.yandex.ace.ventura.salo.AceVenturaIndexContext;
import ru.yandex.ace.ventura.salo.AceVenturaIndexException;
import ru.yandex.ace.ventura.salo.AceVenturaIndexHandler;
import ru.yandex.ace.ventura.salo.AceVenturaMdbsContext;
import ru.yandex.ace.ventura.salo.AceVenturaSalo;
import ru.yandex.ace.ventura.salo.AceventuraSaloEmail;
import ru.yandex.ace.ventura.salo.EmailInfo;
import ru.yandex.ace.ventura.salo.TagInfo;
import ru.yandex.ace.ventura.salo.UpdateAceVenturaEnvelope;
import ru.yandex.ace.ventura.salo.handlers.ReindexContext;
import ru.yandex.blackbox.BlackboxClient;
import ru.yandex.blackbox.BlackboxDbfield;
import ru.yandex.blackbox.BlackboxUserinfo;
import ru.yandex.blackbox.BlackboxUserinfoRequest;
import ru.yandex.blackbox.BlackboxUserinfos;
import ru.yandex.dbfields.CollieFields;
import ru.yandex.function.BasicGenericConsumer;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.EmptyFutureCallback;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonBoolean;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonLong;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.JsonString;
import ru.yandex.json.dom.PositionSavingContainerFactory;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.JsonParser;
import ru.yandex.json.parser.StackContentHandler;
import ru.yandex.json.parser.StringCollectorsFactory;
import ru.yandex.json.writer.JsonType;
import ru.yandex.parser.string.EnumParser;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.salo.Mdb;
import ru.yandex.search.salo.PingEnvelope;

public abstract class AbstractAceVenturaIndexHandler2
    implements AceVenturaIndexHandler
{
    public static final String GET_CONTACTS =
            "/get-user-contacts?&scope=contacts";
    public static final String GET_EMAILS_INFO =
            "/get-user-emails?&scope=contacts";
    public static final String GET_LIST_INFO =
            "/get-contacts-list?&scope=contacts";
    public static final String GET_TAGS =
            "/get-contacts-tags?&scope=contacts";
    public static final String GET_TAGGED_EMAILS =
            "/get-tagged-emails?&scope=contacts";
    public static final String GET_TAGGED_CONTACTS =
            "/get-tagged-contacts?&scope=contacts";
    public static final String GET_SHARED_LISTS =
        "/get-shared-lists?&scope=contacts";
    public static final String GET_ABOOK_INDEX_STATS =
        "/sequential/search?&service=abook&text=abook_last_contacted:*" +
            "&get=abook_email,abook_last_contacted&json-type=dollar";
    public static final String GET_ACEVENTURA_INDEX_STATS =
        "/sequential/search?&service="
            + System.getenv().getOrDefault("ACEVENTURA_QUEUE", "")
            + "&text=av_last_usage:*"
            + "&get=av_last_usage,av_email_cid,av_email,av_email_id&json-type=dollar";

    public static final int CORP_STAFF_ORG_ID = 2;
    public static final EnumParser<UserType> USER_TYPE_PARSER =
        new EnumParser<>(UserType.class);
    public static final EnumParser<ListType> LIST_TYPE_PARSER =
        new EnumParser<>(ListType.class);

    protected static final JsonObject LAST_USAGE_FUNCTION;
    public static final Pattern DOMAIN_VALIDATOR =
            Pattern.compile("^([a-z0-9]+(-[a-z0-9]+)*\\.)+[a-z]{2,}$");
    private static final String YANDEX_TEAM_LOGIN_REGEXP = "[a-z0-9\\-]+";

    static {
        JsonMap maxFunction = new JsonMap(BasicContainerFactory.INSTANCE);
        JsonMap getFunction = new JsonMap(BasicContainerFactory.INSTANCE);
        JsonList maxArgs = new JsonList(BasicContainerFactory.INSTANCE);
        JsonList getArgs = new JsonList(BasicContainerFactory.INSTANCE);
        getArgs.add(new JsonString(AceVenturaFields.LAST_USAGE.stored()));
        getFunction.put("function", new JsonString("get"));
        getFunction.put("args", getArgs);
        maxFunction.put("function", new JsonString("max"));
        maxArgs.add(getFunction);
        maxArgs.add(new JsonLong(0L));
        maxFunction.put("args", maxArgs);

        LAST_USAGE_FUNCTION = maxFunction;
    }

    protected final AceVenturaMdbsContext context;
    protected final AceVenturaSalo salo;

    protected AbstractAceVenturaIndexHandler2(
        final AceVenturaMdbsContext context)
    {
        this.context = context;
        this.salo = (AceVenturaSalo) context.salo();
    }

    public static boolean isCorpStaffOrgId(final AceVenturaPrefix prefix) {
        return prefix.uid() == CORP_STAFF_ORG_ID
            && prefix.userType() == UserType.CONNECT_ORGANIZATION;
    }

    public static boolean isCorpUser(final AceVenturaPrefix prefix) {
        return BlackboxUserinfo.corp(prefix.uid())
            && prefix.userType() == UserType.PASSPORT_USER;
    }

    public JsonMap buildContact(
        final JsonMap map,
        final AceVenturaPrefix user,
        final AceVenturaPrefix owner,
        final List<TagInfo> contactTags,
        final boolean update)
        throws JsonException, AceVenturaIndexException
    {
        JsonMap contact = new JsonMap(BasicContainerFactory.INSTANCE);
        Long contactId = map.getLong(CollieFields.CONTACT_ID);

        if (owner.userType() == UserType.PASSPORT_USER
            && BlackboxUserinfo.corp(owner.uid()))
        {
            contact.put(AceVenturaFields.IS_CORP.stored(), JsonBoolean.TRUE);
        }

        contact.put(
            AceVenturaFields.ID.stored(),
            new JsonString(
                AceVenturaFields.contactUrl(
                    contactId.toString(),
                    user,
                    owner)));

        contact.put(
            AceVenturaFields.RECORD_TYPE.stored(),
            new JsonString(AceVenturaRecordType.CONTACT.fieldValue()));
        contact.put(
            AceVenturaFields.USER_ID.stored(),
            new JsonLong(user.uid()));
        contact.put(
            AceVenturaFields.USER_TYPE.stored(),
            new JsonString(user.userType().toString()));
        contact.put(
            AceVenturaFields.CID.stored(),
            new JsonLong(contactId));

        if (owner.uid() != user.uid()) {
            contact.put(AceVenturaFields.SHARED.stored(), JsonBoolean.TRUE);
            contact.put(
                AceVenturaFields.OWNER_ID.stored(),
                new JsonLong(owner.uid()));
            contact.put(
                AceVenturaFields.OWNER_TYPE.stored(),
                new JsonString(owner.userType().toString()));
        }

        JsonMap vcard;
        JsonObject vcardObj = map.get(CollieFields.VCARD);
        if (vcardObj.type() == JsonObject.Type.STRING) {
            String vcardStr = vcardObj.asString();
            BasicGenericConsumer<JsonObject, JsonException> consumer =
                new BasicGenericConsumer<>();
            JsonParser parser = new JsonParser(
                new StackContentHandler(
                    new TypesafeValueContentHandler(
                        consumer,
                        StringCollectorsFactory.INSTANCE.apply(
                            vcardStr.length()),
                        PositionSavingContainerFactory.INSTANCE)));
            parser.parse(vcardStr);
            vcard = consumer.get().asMap();
        } else {
            vcard = vcardObj.asMap();
        }

        buildVcard(vcard, context.humanNamesUtil(), contact);

        contact.put(
            AceVenturaFields.LIST_ID.stored(),
            map.get(CollieFields.LIST_ID));

        if (!update) {
            contact.put(
                AceVenturaFields.LIST_NAME.stored(),
                map.get(CollieFields.LIST_NAME));
            contact.put(
                AceVenturaFields.LIST_TYPE.stored(),
                map.get(
                    CollieFields.LIST_TYPE));

            StringBuilder tagsSb = new StringBuilder();
            for (TagInfo tagInfo: contactTags) {
                tagsSb.append(tagInfo.id());
                tagsSb.append('\n');
            }

            if (tagsSb.length() > 0) {
                tagsSb.setLength(tagsSb.length() - 1);

                contact.put(
                    AceVenturaFields.TAGS.stored(),
                    new JsonString(tagsSb.toString()));
                contact.put(
                    AceVenturaFields.CONTACT_TAGS.stored(),
                    new JsonString(tagsSb.toString()));
            }
        }

        return contact;
    }

    protected JsonMap buildEmail(
        final AceVenturaPrefix user,
        final AceVenturaPrefix owner,
        final EmailInfo emailInfo)
        throws JsonException, AceVenturaIndexException
    {
        JsonMap result =
            buildEmail(user, owner, emailInfo.emailId(), emailInfo.email());
        if (result == null) {
            return null;
        }

        if (owner.userType() == UserType.PASSPORT_USER
            && BlackboxUserinfo.corp(owner.uid()))
        {
            result.put(AceVenturaFields.IS_CORP.stored(), JsonBoolean.TRUE);
        }

        result.put(
            AceVenturaFields.EMAIL_CID.stored(),
            new JsonLong(emailInfo.contactId()));
        if (emailInfo.label() != null){
            result.put(
                AceVenturaFields.EMAIL_LABEL.stored(),
                new JsonString(emailInfo.label()));
        } else {
            result.put(
                AceVenturaFields.EMAIL_LABEL.stored(),
                JsonNull.INSTANCE);
        }

        if (emailInfo.type() != null) {
            result.put(
                AceVenturaFields.EMAIL_TYPE.stored(),
                new JsonString(emailInfo.type()));
        } else {
            result.put(
                AceVenturaFields.EMAIL_TYPE.stored(),
                JsonNull.INSTANCE);
        }

        result.put(
            AceVenturaFields.REVISION.stored(),
            new JsonLong(emailInfo.revision()));

        result.put(AceVenturaFields.LAST_USAGE.stored(), LAST_USAGE_FUNCTION);

        if (emailInfo.tags() != null) {
            StringBuilder tagSb = new StringBuilder();
            for (TagInfo tag: emailInfo.tags()) {
                tagSb.append(tag.id());
                tagSb.append('\n');
            }

            if (tagSb.length() > 0) {
                tagSb.setLength(tagSb.length() - 1);
            }

            result.put(
                AceVenturaFields.TAGS.stored(),
                new JsonString(tagSb.toString()));
            result.put(
                AceVenturaFields.EMAIL_TAGS.stored(),
                new JsonString(tagSb.toString()));
        }

        return result;
    }

    protected JsonMap buildEmail(
        final AceVenturaPrefix user,
        final AceVenturaPrefix owner,
        final long emailId,
        final String email)
        throws JsonException, AceVenturaIndexException
    {
        JsonMap result = createEmail(user, owner, emailId, email);
        if (result == null) {
            context.logger().warning(
                "Invalid email, no @ in address " + email);
            context.emailParseStat().accept(1);
        } else {
            context.emailParseStat().accept(0);
        }

        return result;
    }

    public JsonMap createEmail(
        final AceVenturaPrefix user,
        final AceVenturaPrefix owner,
        final long emailId,
        final String email)
        throws JsonException, AceVenturaIndexException
    {
        AceventuraSaloEmail saloEmail =
            AceventuraSaloEmail.parse(email, context.logger());

        if (saloEmail == null) {
            return null;
        }

        boolean corp = owner.userType() == UserType.PASSPORT_USER
            && BlackboxUserinfo.corp(owner.uid());
        JsonMap result = new JsonMap(BasicContainerFactory.INSTANCE);
        result.put(
            AceVenturaFields.EMAIL_ID.stored(),
            new JsonLong(emailId));
        result.put(
            AceVenturaFields.RECORD_TYPE.stored(),
            new JsonString(AceVenturaRecordType.EMAIL.fieldValue()));

        if (corp) {
            result.put(AceVenturaFields.IS_CORP.stored(), JsonBoolean.TRUE);
        }

        result.put(AceVenturaFields.EMAIL.stored(), new JsonString(email));
        result.put(
            AceVenturaFields.USER_ID.stored(),
            new JsonLong(user.uid()));
        result.put(
            AceVenturaFields.USER_TYPE.stored(),
            new JsonString(user.userType().toString()));
        if (!user.equals(owner)) {
            result.put(
                AceVenturaFields.OWNER_ID.stored(),
                new JsonLong(user.uid()));
            result.put(
                AceVenturaFields.OWNER_TYPE.stored(),
                new JsonString(user.userType().toString()));
            result.put(AceVenturaFields.SHARED.stored(), JsonBoolean.TRUE);
        }

        if (corp && saloEmail.yaTeamEmail() && saloEmail.login().matches(YANDEX_TEAM_LOGIN_REGEXP)) {
            JsonObject staffDataObj = salo.getCachedStaffData(saloEmail.login());
            Map<String, JsonObject> staffData;
            if (staffDataObj != null) {
                staffData = staffDataObj.asMap();
            } else {
                try {
                    QueryConstructor qc = new QueryConstructor("/staff/info?");
                    qc.append("login", saloEmail.login());
                    staffDataObj = salo.staffProxyClient().execute(
                        salo.aceSaloConfig().staffConfig().host(),
                        new BasicAsyncRequestProducerGenerator(qc.toString()),
                        JsonAsyncTypesafeDomConsumerFactory.OK,
                        EmptyFutureCallback.INSTANCE).get();
                    staffData = staffDataObj.asMap();
                } catch (BadRequestException | ExecutionException | InterruptedException e) {
                    throw new AceVenturaIndexException(
                        "Failed to get staff info for " + owner.uid() + " " + saloEmail.login(), e);
                }

                salo.putCachedStaffData(saloEmail.login(), staffDataObj);
            }

            result.putAll(staffData);
        }

        result.put(
            AceVenturaFields.DOMAIN.stored(),
            new JsonString(saloEmail.domainNormalized()));
        result.put(
            AceVenturaFields.DOMAIN_NT.stored(),
            new JsonString(saloEmail.domainNoTopLevel()));
        result.put(
            AceVenturaFields.LOGIN.stored(),
            new JsonString(saloEmail.login()));
        result.put(
            AceVenturaFields.LOGINS.stored(),
            new JsonString(saloEmail.loginAliases()));

        result.put(
            AceVenturaFields.ID.stored(),
            new JsonString(
                AceVenturaFields.emailUrl(
                    String.valueOf(emailId),
                    user,
                    owner)));

        return result;
    }

    protected static void buildVcard(
        final JsonMap vcard,
        final HumanNamesUtil util,
        final JsonMap result)
        throws JsonException, AceVenturaIndexException
    {
        JsonList phones = vcard.getListOrNull(CollieFields.PHONES);
        StringBuilder phonesSb = new StringBuilder();
        StringBuilder phonesNormSb = new StringBuilder();
        if (phones != null && !phones.isEmpty()) {
            for (JsonObject jo: phones) {
                JsonMap map = jo.asMap();
                String phone = map.getString(CollieFields.PHONE, null);
                if (phone != null) {
                    phonesSb.append(phone);
                    phonesSb.append('\n');
                    phonesNormSb.append(phone);
                    phonesNormSb.append('\n');
                    String norm = PhonesUtil.cutPrefixOrNull(phone);
                    if (norm != null) {
                        phonesNormSb.append(norm);
                        phonesNormSb.append('\n');
                    }
                }
            }

            if (phonesSb.length() > 0) {
                phonesSb.setLength(phonesSb.length() - 1);
                phonesNormSb.setLength(phonesNormSb.length() - 1);

                result.put(
                    AceVenturaFields.PHONES.stored(),
                    new JsonString(phonesSb.toString()));
                result.put(
                    AceVenturaFields.PHONES_N.stored(),
                    new JsonString(phonesNormSb.toString()));
                result.put(
                    AceVenturaFields.HAS_PHONES.stored(),
                    JsonBoolean.TRUE);
            } else {
                result.put(
                    AceVenturaFields.HAS_PHONES.stored(),
                    JsonBoolean.FALSE);
            }
        } else {
            result.put(
                AceVenturaFields.HAS_PHONES.stored(),
                JsonBoolean.FALSE);
        }

        JsonList namesList = vcard.getListOrNull("names");
        if (namesList == null || namesList.isEmpty()) {
            result.put(
                AceVenturaFields.VCARD.stored(),
                new JsonString(JsonType.NORMAL.toString(vcard)));
            return;
        }

        StringBuilder namesSb = new StringBuilder();
        StringBuilder namesAlias = new StringBuilder();

        String value;
        for (JsonObject name: namesList) {
            JsonMap map = name.asMap();
            value = map.getString(CollieFields.FIRST, "");
            if (!value.isEmpty()) {
                value = value.trim();
                namesSb.append(value);
                namesSb.append('\t');

                String[] aliases = util.aliases(value);
                if (aliases != null) {
                    // alias should contain original one
                    for (String alias: aliases) {
                        namesAlias.append(alias);
                        namesAlias.append('\n');
                    }
                } else {
                    namesAlias.append(value);
                    namesAlias.append('\n');
                }
            }

            value = map.getString(CollieFields.MIDDLE, "");
            if (!value.isEmpty()) {
                value = value.trim();
                namesSb.append(value);
                namesSb.append('\t');
            }

            value = map.getString(CollieFields.LAST, "");
            if (!value.isEmpty()) {
                value = value.trim();
                namesSb.append(value);
            }

            namesSb.append('\n');
        }

        if (namesSb.length() > 0) {
            namesSb.setLength(namesSb.length() - 1);
        }

        if (namesAlias.length() > 0) {
            namesAlias.setLength(namesAlias.length() - 1);
        }

        result.put(
            AceVenturaFields.NAMES.stored(),
            new JsonString(namesSb.toString()));

        result.put(
            AceVenturaFields.NAMES_ALIAS.stored(),
            new JsonString(namesAlias.toString()));

        result.put(
            AceVenturaFields.VCARD.stored(),
            new JsonString(JsonType.NORMAL.toString(vcard)));
    }

    protected List<EmailInfo> emails(
        final AceVenturaIndexContext context,
        final Long contactId)
        throws IOException, JsonException
    {
        StringBuilder uri = new StringBuilder(GET_EMAILS_INFO);
        uri.append("&contactId=");
        uri.append(contactId);
        JsonList emailList =
            context.msalRequest(context.prefix(), uri);

        List<EmailInfo> emails = new ArrayList<>(emailList.size());
        for (JsonObject jo: emailList) {
            emails.add(new EmailInfo(jo.asMap(), Collections.emptyList()));
        }

        return emails;
    }

    public JsonMap buildTag(final AceVenturaPrefix prefix, final TagInfo info) {
        JsonMap result = new JsonMap(BasicContainerFactory.INSTANCE);
        result.put(
            AceVenturaFields.ID.stored(),
            new JsonString(
                AceVenturaFields.tagUrl(
                    String.valueOf(info.id()),
                    prefix)));
        result.put(
            AceVenturaFields.RECORD_TYPE.stored(),
            new JsonString(AceVenturaRecordType.TAG.fieldValue()));
        result.put(
            AceVenturaFields.TAG_NAME.stored(),
            new JsonString(info.name()));
        result.put(
            AceVenturaFields.REVISION.stored(),
            new JsonLong(info.revision()));
        result.put(
            AceVenturaFields.TAG_ID.stored(),
            new JsonLong(info.id()));
        if (info.type() != null) {
            result.put(
                AceVenturaFields.TAG_TYPE.stored(),
                new JsonString(info.type()));
        }

        result.put(
            AceVenturaFields.USER_ID.stored(),
            new JsonLong(prefix.uid()));
        result.put(
            AceVenturaFields.USER_TYPE.stored(),
            new JsonString(prefix.userType().lowName()));

        return result;
    }

    public void reindexTags(
        final ReindexContext context,
        final Collection<TagInfo> tagInfos)
        throws IOException, JsonException
    {
        JsonMap result = new JsonMap(BasicContainerFactory.INSTANCE);
        result.put(
            "prefix",
            new JsonString(context.context().prefix().toStringFast()));
        result.put("AddIfNotExists", JsonBoolean.TRUE);
        JsonList list = new JsonList(BasicContainerFactory.INSTANCE);

        context.logger().info("Reindexing tags for " + context.owner());
        if (tagInfos.size() == 0) {
            context.logger().info(
                "No tags to index for " + context.owner().toStringFast());
            return;
        } else {
            for (TagInfo tagInfo: tagInfos) {
                list.add(buildTag(context.context().prefix(), tagInfo));
            }

            context.logger().info("Reindexed tags " + list.size());
            context.logger().info(
                "Reindexed tags " + JsonType.NORMAL.toString(list));
        }
        result.put("docs", list);

        context.storage().add(
            new UpdateAceVenturaEnvelope(
                context.context(),
                result,
                list.size()));
    }

    public Map<String, Long> fetchCurrentEmailsUsage(
        final AceVenturaPrefix prefix)
        throws IOException
    {
        if (prefix.userType() != UserType.PASSPORT_USER) {
            return Collections.emptyMap();
        }

        StringBuilder uri = new StringBuilder(GET_ACEVENTURA_INDEX_STATS);
        uri.append("&prefix=");
        uri.append(prefix);

        BasicAsyncRequestProducerGenerator producerGenerator =
            new BasicAsyncRequestProducerGenerator(uri.toString());

        Future<JsonObject> aceProxyRequest = salo.aceventuraProxyClient().execute(
            salo.aceventuraProxyHost(),
            producerGenerator,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            EmptyFutureCallback.INSTANCE);

        try {
            JsonMap root = aceProxyRequest.get().asMap();
            Integer hitsCount = root.getInt("hitsCount");
            Map<String, Long> result = new LinkedHashMap<>(hitsCount);
            context.logger().info("Fetched " + hitsCount + " for " + prefix);
            for (JsonObject itemObj: root.getList("hitsArray")) {
                JsonMap item = itemObj.asMap();
                String emailStr = item.getString("av_email", "");
                //Long emailId = item.getLong("av_email_id", -1L);
                Long lastContacted = item.getLong("av_last_usage", 0L);
                if (lastContacted > 0L && !emailStr.isEmpty()) {
                    String[] emails = emailStr.trim().split("\\n");
                    for (String email: emails) {
                        if (!email.isEmpty()) {
                            result.put(email.trim().toLowerCase(Locale.ROOT), lastContacted);
                        }
                    }
                }
            }

            context.logger().info("Fetched from aceventura index " + result.size());
            return result;
        } catch (InterruptedException | ExecutionException | NumberFormatException | JsonException e) {
            context.logger().log(
                Level.WARNING,
                "Failed to fetch aceventura index stats",
                e);
            throw new IOException(e);
        }
    }

    public Map<String, Long> fetchOldAbookEmailsUsage(
        final AceVenturaPrefix prefix)
        throws IOException
    {
        if (prefix.userType() != UserType.PASSPORT_USER) {
            return Collections.emptyMap();
        }

        BlackboxClient client;
        boolean corp = BlackboxUserinfo.corp(prefix.uid());
        if (corp) {
            client = salo.corpBlackboxClient();
        } else {
            client = salo.bpBlackboxClient();
        }

        Future<BlackboxUserinfos> future = client.userinfo(
            new BlackboxUserinfoRequest(prefix.uid())
                .requiredDbfields(BlackboxDbfield.SUID)
                .addHeader(
                    YandexHeaders.X_YA_SERVICE_TICKET,
                    salo.blackboxTvm2Ticket(corp)),
            EmptyFutureCallback.INSTANCE);

        Long suid = -1L;
        try {
            List<BlackboxUserinfo> infos = future.get();
            if (infos.size() > 0) {
                suid = Long.parseLong(
                    infos.get(0).dbfields().getOrDefault(BlackboxDbfield.SUID, suid.toString()));
            }
        } catch (InterruptedException | ExecutionException | NumberFormatException e) {
            context.logger().log(Level.WARNING, "Failed to fetch emails usage", e);
        }

        if (suid < 0) {
            return Collections.emptyMap();
        }

        StringBuilder uri = new StringBuilder(GET_ABOOK_INDEX_STATS);
        uri.append("&prefix=");
        uri.append(suid);

        BasicAsyncRequestProducerGenerator producerGenerator =
                new BasicAsyncRequestProducerGenerator(uri.toString());

        producerGenerator.addHeader(
                YandexHeaders.X_YA_SERVICE_TICKET,
                salo.msearchProxyTvm2Ticket());

        Future<JsonObject> mailProxyResp = salo.mailProxyClient().execute(
                salo.mailProxyHost(),
                producerGenerator,
                JsonAsyncTypesafeDomConsumerFactory.OK,
                EmptyFutureCallback.INSTANCE);

        try {
            JsonMap root = mailProxyResp.get().asMap();
            Integer hitsCount = root.getInt("hitsCount");
            Map<String, Long> result = new LinkedHashMap<>(hitsCount);
            context.logger().info("Fetched " + hitsCount + " for " + prefix);
            for (JsonObject itemObj: root.getList("hitsArray")) {
                JsonMap item = itemObj.asMap();
                String emailStr = item.getString("abook_email", "");
                Long lastContacted = item.getLong("abook_last_contacted", 0L);
                if (lastContacted > 0L && !emailStr.isEmpty()) {
                    String[] emails = emailStr.trim().split("\\n");
                    for (String email: emails) {
                        if (!email.isEmpty()) {
                            result.put(email.trim().toLowerCase(Locale.ROOT), lastContacted * 1000);
                        }
                    }
                }
            }

            context.logger().info("Fetched from old index " + result.size());
            return result;
        } catch (InterruptedException | ExecutionException | NumberFormatException | JsonException e) {
            context.logger().log(
                Level.WARNING,
                "Failed to fetch old abook index stats",
                e);
            throw new IOException(e);
        }
    }

    public void reindexList(
        final ReindexContext reindexContext,
        final JsonMap listInfoMap,
        final boolean keepUsageStats)
        throws IOException, JsonException
    {
        Map<String, Long> emailsStats = Collections.emptyMap();
        if (keepUsageStats) {
            if (salo.aceSaloConfig().fetchOldAbookUsageOnReindex()) {
                emailsStats = fetchOldAbookEmailsUsage(reindexContext.owner());
            } else {
                emailsStats = fetchCurrentEmailsUsage(reindexContext.owner());
            }
        }

        Long listId = listInfoMap.getLong(CollieFields.LIST_ID);
        reindexContext.logger().info(
            "Reindexing list id  " + listId);

        AceVenturaIndexContext context = reindexContext.context();
        StringBuilder contactsUri = new StringBuilder(GET_CONTACTS);
        contactsUri.append("&listId=");
        contactsUri.append(listId);
        JsonList contacts =
            context.msalRequest(reindexContext.owner(), contactsUri);

        reindexContext.logger().fine(
            "Contacts fetched " + contacts.size());

        Map<Long, JsonMap> contactsMap = new LinkedHashMap<>(contacts.size());
        for (JsonObject jo: contacts) {
            JsonMap cmap = jo.asMap();
            cmap.put(
                CollieFields.LIST_NAME,
                listInfoMap.get(CollieFields.LIST_NAME));
            cmap.put(
                CollieFields.LIST_TYPE,
                listInfoMap.get(CollieFields.LIST_TYPE));
            contactsMap.put(cmap.getLong(CollieFields.CONTACT_ID), cmap);
        }

        JsonMap result = new JsonMap(BasicContainerFactory.INSTANCE);
        result.put("prefix", new JsonString(reindexContext.owner().toStringFast()));
        result.put("AddIfNotExists", JsonBoolean.TRUE);
        JsonList list = new JsonList(BasicContainerFactory.INSTANCE);
        int nonEmptylastUsage = 0;

        for (Map.Entry<Long, JsonMap> contactEntry: contactsMap.entrySet()) {
            Long cid = contactEntry.getKey();

            JsonMap contact = buildContact(
                contactEntry.getValue(),
                reindexContext.owner(),
                reindexContext.owner(),
                reindexContext.contactTags(cid),
                false);
            list.add(contact);

            List<EmailInfo> emailInfos = reindexContext.emails(cid);
            if (emailInfos.size() == 0) {
                context.logger().warning("No emails found for " + cid);
                continue;
            }

            addEnNamesToStaffContact(context, contact, emailInfos, cid);

            for (EmailInfo emailInfo: emailInfos) {
                JsonMap emapRes = buildEmail(
                    reindexContext.owner(),
                    reindexContext.owner(),
                    emailInfo);

                if (emapRes != null) {
                    Long lastUsage = emailsStats.get(emailInfo.email());
                    if (lastUsage != null) {
                        nonEmptylastUsage += 1;
                        emapRes.put(
                            AceVenturaFields.LAST_USAGE.stored(),
                            new JsonLong(lastUsage));
                    }

                    adjustEmailWithContacFields(emapRes, contact);

                    list.add(emapRes);
                } else {
                    reindexContext.logger().warning(
                        "Failed to create email entry from " + emailInfo);
                }
            }
        }

        context.logger().info("Items to update " + list.size() + " with last usage " + nonEmptylastUsage);
        if (list.size() <= 0) {
            reindexContext.storage().add(pingEnvelope(context));
        } else {
            result.put("docs", list);

            reindexContext.storage().add(
                new UpdateAceVenturaEnvelope(context, result, list.size()));
        }
    }

    protected void adjustEmailWithContacFields(
        final JsonMap email,
        final JsonMap contact)
    {
        for (AceVenturaFields field: AceVenturaFields.EMAIL_CONTACTS_FIELDS) {
            JsonObject jo = contact.getOrDefault(field.field(), null);
            if (jo != null) {
                email.put(field.stored(), jo);
            }
        }
    }

    public void reindexList(
        final ReindexContext reindexContext,
        final long listId,
        final boolean keepUsageStats)
        throws IOException, JsonException
    {
        AceVenturaIndexContext context = reindexContext.context();

        StringBuilder listInfoUri = new StringBuilder(GET_LIST_INFO);
        listInfoUri.append("&listId=");
        listInfoUri.append(listId);
        JsonList listInfo =
            context.msalRequest(reindexContext.owner(), listInfoUri);

        if (listInfo.size() != 1) {
            throw new AceVenturaIndexException(
                "Bad list info " + JsonType.NORMAL.toString(listInfo));
        }
        JsonMap listInfoMap = listInfo.get(0).asMap();
        reindexContext.logger().info(
            "List id " + listId + " contacts " + listInfo.size());

        reindexList(reindexContext, listInfoMap, keepUsageStats);
    }

    protected static JsonMap merge(final JsonMap old, JsonMap update) {
        JsonMap result = new JsonMap(update.containerFactory());
        result.putAll(old);

        for (Map.Entry<String, JsonObject> entry: update.entrySet()) {
            JsonObject value = entry.getValue();
            if (value != null && value.type() != JsonObject.Type.NULL) {
                result.put(entry.getKey(), value);
            }
        }

        return result;
    }

    protected static JsonObject merge(final JsonMap old, JsonMap update, final String field) {
        JsonObject value = update.get(field);
        if (value == null || value.type() == JsonObject.Type.NULL) {
            return old.get(field);
        }

        return value;
    }

    protected static PingEnvelope pingEnvelope(final AceVenturaIndexContext context) {
        StringBuilder sb = new StringBuilder();
        sb.append("&change_type=");
        sb.append(context.changeType().lowName());
        sb.append("&change_user=");
        sb.append(context.prefix().toStringFast());

        return new PingEnvelope(
            context.changeId(),
            sb.toString(),
            context.envelopesContext(),
            (int) (context.prefix().hash() % Mdb.SHARDS));
    }


    protected void addEnNamesToStaffContact(
        final AceVenturaIndexContext context,
        final JsonMap luceneContact,
        final Long cid)
        throws IOException, JsonException
    {
        addEnNamesToStaffContact(context, luceneContact, null, cid);
    }

    protected void addEnNamesToStaffContact(
        final AceVenturaIndexContext context,
        final JsonMap luceneContact,
        final List<EmailInfo> emailInfoList,
        final Long cid)
        throws IOException, JsonException
    {
        if (isCorpStaffOrgId(context.prefix()) || isCorpUser(context.prefix())) {
            String loginCacheKey = context.prefix().toString() + "_" + cid;
            String login = salo.staffUserLoginCache().getIfPresent(loginCacheKey);
            JsonObject staffData = null;
            if (login == null) {
                List<EmailInfo> emailInfos = emailInfoList;
                if (emailInfos == null) {
                    emailInfos = emails(context, cid);
                }

                if (emailInfos.size() > 0) {
                    for (EmailInfo emailInfo : emailInfos) {
                        AceventuraSaloEmail email =
                            AceventuraSaloEmail.parse(emailInfo.email(), context.logger());
                        if (email == null) {
                            continue;
                        }

                        // if it is staff shared userid, use mild check for email
                        boolean staffEmail = isCorpStaffOrgId(context.prefix()) && email.domainNormalized().startsWith("yandex-team.");
                        // if personal corp book, use strict one
                        staffEmail |= isCorpUser(context.prefix()) && email.yaTeamEmail();

                        if (!staffEmail) {
                            context.logger().warning(
                                "corpOrgStaff email is not yateam " + cid + " " + emailInfo.email());
                            continue;
                        }

                        if (!email.login().matches(YANDEX_TEAM_LOGIN_REGEXP)) {
                            context.logger().warning(
                                "corpOrgStaff login is not valid login " + cid + " " + emailInfo.email() + " " + email.login());
                            // https://wiki.yandex-team.ru/intranet/nanimator/new/loginreq/
                            continue;
                        }

                        salo.staffUserLoginCache().put(loginCacheKey, email.login());
                        login = email.login();
                        break;
                    }
                }
            }

            if (login != null) {
                staffData = salo.getCachedStaffData(login);
                if (staffData == null) {
                    int retries = 0;
                    int[] sleeps = new int[]{0, 100, 1000, 5000, 10000, 30000};
                    try {
                        while (true) {
                            try {
                                QueryConstructor qc = new QueryConstructor("/staff/info?");
                                qc.append("login", login);
                                staffData = salo.staffProxyClient().execute(
                                    salo.aceSaloConfig().staffConfig().host(),
                                    new BasicAsyncRequestProducerGenerator(qc.toString()),
                                    JsonAsyncTypesafeDomConsumerFactory.OK,
                                    EmptyFutureCallback.INSTANCE).get().asMap();
                                break;
                            } catch (BadRequestException | ExecutionException e) {
                                retries += 1;
                                if (retries >= sleeps.length) {
                                    throw new AceVenturaIndexException(
                                        "Failed to get staff info for " + cid + " " + login, e);
                                }

                                Thread.sleep(sleeps[retries]);
                            }
                        }

                        if (staffData != null) {
                            salo.putCachedStaffData(login, staffData);
                        }
                    } catch (InterruptedException e) {
                        throw new AceVenturaIndexException(
                            "Failed to get staff info for " + cid + " " + login, e);
                    }
                }
            } else {
                context.logger().warning("corpOrgStaff No emails found for " + cid);
            }

            JsonObject enNames = JsonNull.INSTANCE;
            if (staffData != null) {
                enNames = staffData.get(AceVenturaFields.EN_NAMES.stored());
                context.logger().warning("corpOrgStaff for " + cid + " found en names: " + JsonType.NORMAL.toString(enNames));
            }

            luceneContact.put(AceVenturaFields.EN_NAMES.stored(), enNames);
        }
    }

}

