package ru.yandex.ace.ventura.salo.handlers2.reindex;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;

import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpUriRequest;

import ru.yandex.ace.ventura.AceVenturaFields;
import ru.yandex.ace.ventura.AceVenturaPrefix;
import ru.yandex.ace.ventura.AceVenturaRecordType;
import ru.yandex.ace.ventura.salo.AceVenturaIndexContext;
import ru.yandex.ace.ventura.salo.AceVenturaMdbProvider;
import ru.yandex.ace.ventura.salo.CollieChangeType;
import ru.yandex.dbfields.CollieFields;
import ru.yandex.function.GenericBiConsumer;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.BadResponseException;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonLong;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonString;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.search.salo.Envelope;
import ru.yandex.search.salo.EnvelopesContext;
import ru.yandex.search.salo.HttpLockHolder;
import ru.yandex.search.salo.Mdb;
import ru.yandex.search.salo.Salo;
import ru.yandex.search.salo.TokenException;

public class ReindexUserOperation implements GenericBiConsumer<HttpLockHolder, EnvelopesContext, Exception> {
    protected final ContactsReindexUserHandler reindexHandler;
    protected final AceVenturaPrefix prefix;
    protected final AceVenturaMdbProvider provider;
    protected final PrefixedLogger logger;
    protected final Salo salo;

    public ReindexUserOperation(
        final AceVenturaPrefix prefix,
        final AceVenturaMdbProvider provider,
        final Logger logger)
    {
        this.prefix = prefix;
        this.provider = provider;
        this.logger = new PrefixedLogger(logger, "", " ");
        this.reindexHandler = new ContactsReindexUserHandler(provider.mdbsContext(), true);
        this.salo = provider.mdbsContext().salo();
    }

    protected void sendDelete(
        final AceVenturaPrefix prefix,
        final EnvelopesContext ennvelopesContext)
        throws Exception
    {
        // dropping contacts/tags/emails not dropping shared ones
        StringBuilder uri = new StringBuilder();
        uri.append("/delete?&prefix=");
        uri.append(prefix);
        uri.append("&reindex");
        uri.append("&operation-date=");
        uri.append(System.currentTimeMillis());
        uri.append("&text=");
        uri.append(AceVenturaFields.RECORD_TYPE.prefixed());
        uri.append(":(");
        uri.append(AceVenturaRecordType.CONTACT.fieldValue());
        uri.append("+OR+");
        uri.append(AceVenturaRecordType.EMAIL.fieldValue());
        uri.append("+OR+");
        uri.append(AceVenturaRecordType.TAG.fieldValue());
        uri.append(")");

        HttpGet request = new HttpGet(uri.toString());
        applyHeaders(request, ennvelopesContext, prefix);
        sendRequest(request);
    }

    protected void sendRequest(final HttpRequest request) throws Exception {
        try (CloseableHttpResponse response = salo.zoolooserClient().execute(
            salo.config().zoolooserConfig().host(),
            request))
        {
            String body = CharsetUtils.toString(response.getEntity());
            int status = response.getStatusLine().getStatusCode();
            if (status == HttpStatus.SC_FORBIDDEN) {
                throw new TokenException(
                    "Token rejected by zoolooser with message: " + body,
                    "reindex-worker",
                    String.valueOf(request.getFirstHeader(YandexHeaders.LOCKID)));
            } else if (status == HttpStatus.SC_CONFLICT) {
                throw new Exception("Conflict impossible for reindex requests");
            } else if (status != HttpStatus.SC_OK) {
                throw new BadResponseException(request, response, body);
            } else {
                logger.info("Success on " + request.getRequestLine().getUri());
            }
        }
    }

    protected void applyHeaders(
        final HttpUriRequest request,
        final EnvelopesContext envelopesContext,
        final AceVenturaPrefix prefix)
    {
        request.addHeader(YandexHeaders.LOCKID, envelopesContext.tokenString());
        request.addHeader(YandexHeaders.SERVICE, envelopesContext.mdb().service());
        request.addHeader(
            YandexHeaders.ZOO_SHARD_ID,
            String.valueOf(prefix.hash() % Mdb.SHARDS));
    }

    protected List<Envelope> prepareEnvelopes(
        final AceVenturaIndexContext context,
        final HttpLockHolder holder,
        final EnvelopesContext envelopesContext)
        throws Exception
    {
        List<Envelope> envelopes = new ArrayList<>();
        reindexHandler.handle(context, envelopes);
        return envelopes;
    }

    protected void doAccept(
        final AceVenturaIndexContext indexContext,
        final HttpLockHolder holder,
        final EnvelopesContext envelopesContext)
        throws Exception
    {
        List<Envelope> envelopes = prepareEnvelopes(indexContext, holder, envelopesContext);
        logger.info("For reindex envelopes generated: " + envelopes.size());
        sendDelete(prefix, envelopesContext);
        logger.info("Delete dispacthed");

        int left = 0;
        int right = salo.config().requestsBatchSize();
        int sent = 0;
        while (left < envelopes.size()) {
            List<Envelope> batch = envelopes.subList(left, Math.min(envelopes.size(), right));
            HttpRequest indexRequest =
                holder.makeSetRequest(batch, "reindex-worker", false);
            sendRequest(indexRequest);
            sent += batch.size();
            left = right;
            right += salo.config().requestsBatchSize();
        }

        logger.info("Reindex successfull, sent envelopes: " + sent);
    }

    @Override
    public void accept(final HttpLockHolder holder, final EnvelopesContext envelopesContext) throws Exception {
        if (!envelopesContext.mdb().locked()) {
            throw new BadRequestException("Mdb is not locked");
        }

        JsonMap map = new JsonMap(BasicContainerFactory.INSTANCE);
        map.put("change_type", new JsonString(CollieChangeType.REINDEX_USER.lowName()));
        map.put(CollieFields.USER_TYPE, new JsonString(prefix.userType().lowName()));
        map.put(CollieFields.USER_ID, new JsonLong(prefix.uid()));
        map.put(CollieFields.OPERATION_ID, new JsonLong(-1L));
        map.put(CollieFields.OPERATION_DATE, new JsonLong(System.currentTimeMillis()));
        map.put(CollieFields.SELECT_DATE, new JsonLong(System.currentTimeMillis()));
        map.put(CollieFields.REVISION, new JsonLong(-1L));
        map.put("changed", new JsonMap(BasicContainerFactory.INSTANCE));
        map.put("arguments", new JsonMap(BasicContainerFactory.INSTANCE));

        AceVenturaIndexContext indexContext =
            new AceVenturaIndexContext(provider.mdbsContext(), envelopesContext, logger, map);
        doAccept(indexContext, holder, envelopesContext);
    }
}
