package ru.yandex.ace.ventura.salo.mdb;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import org.apache.http.HttpHost;

import ru.yandex.ace.ventura.salo.ShardRemainFilter;
import ru.yandex.http.util.HttpHostParser;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;

public class RemainsBasedSaloShardsMapFactory implements SaloShardsMapFactory {
    private final Map<HttpHost, Function<Set<String>, Set<String>>> hostFilters;
    private final HttpHost currentHost;

    public RemainsBasedSaloShardsMapFactory(
        final IniConfig config)
        throws ConfigException
    {
        File file = config.getInputFile("file");
        currentHost = config.get("host", HttpHostParser.INSTANCE);

        Map<HttpHost, Function<Set<String>, Set<String>>> saloHostMap
            = new LinkedHashMap<>();

        Map<HttpHost, Integer> remains = new LinkedHashMap<>();
        int maxShard = -1;
        try {
            List<String> lines =
                Files.readAllLines(Paths.get(file.toURI()));
            for (String line: lines) {
                if (line.isEmpty()) {
                    continue;
                }

                line = line.trim();
                if (line.startsWith("#")) {
                    continue;
                }

                String[] split = line.split("\t");
                if (split.length != 3) {
                    throw new ConfigException(
                        "Bad formatted line " + split[1]);
                }

                Integer shard = Integer.parseInt(split[1].trim());
                HttpHost host = HttpHostParser.INSTANCE.apply(split[0]);

                if (shard > maxShard) {
                    maxShard = shard;
                }

                remains.put(host, shard);
            }
        } catch (IOException ioe) {
            throw new ConfigException("Bad formatted file " + file, ioe);
        }

        if (maxShard <= 0) {
            throw new ConfigException(
                "Host not found in searchmap or bad formatted map "
                    + currentHost + ' ' + maxShard);
        }

        if (remains.get(currentHost) == null) {
            throw new ConfigException("Current host not found in shardsmap");
        }

        for (Map.Entry<HttpHost, Integer> entry: remains.entrySet()) {
            saloHostMap.put(
                entry.getKey(),
                new ShardRemainFilter(maxShard + 1, entry.getValue()));
        }

        hostFilters = Collections.unmodifiableMap(saloHostMap);
    }

    @Override
    public SaloShardMap create(final Set<String> mdbs) {
        Map<String, Set<HttpHost>> mdbToHost = new LinkedHashMap<>();
        Map<HttpHost, Set<String>> hostToMdb = new LinkedHashMap<>();
        for (Map.Entry<HttpHost, Function<Set<String>, Set<String>>> entry
            : hostFilters.entrySet())
        {
            Set<String> hostMdbs = entry.getValue().apply(mdbs);
            for (String mdb: hostMdbs) {
                mdbToHost.computeIfAbsent(mdb, (m) -> new LinkedHashSet<>()).add(entry.getKey());
            }

            hostToMdb.computeIfAbsent(entry.getKey(), (h) -> new LinkedHashSet<>()).addAll(hostMdbs);
        }

        return new BasicSaloMdbMap(currentHost, mdbToHost, hostToMdb);
    }
}
