package ru.yandex.mail.search.staff.consumer;

import java.io.IOException;
import java.net.InetAddress;
import java.net.URISyntaxException;
import java.net.UnknownHostException;
import java.util.List;
import java.util.Locale;
import java.util.TimerTask;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.logging.Level;

import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.FormBodyPartBuilder;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.entity.mime.content.StringBody;

import ru.yandex.ace.ventura.AceVenturaFields;
import ru.yandex.ace.ventura.AceVenturaPrefix;
import ru.yandex.ace.ventura.AceVenturaRecordType;
import ru.yandex.ace.ventura.UserType;
import ru.yandex.http.util.BadResponseException;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.EmptyFutureCallback;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.BasicAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.StatusCheckAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.mail.search.staff.Person;
import ru.yandex.mail.search.staff.config.ImmutableStaffConsumerConfig;
import ru.yandex.search.request.util.SearchRequestText;

public class FullStaffConsumer extends TimerTask {
    private static final int MAX_SHARDS = 65534;
    private static final StatusCheckAsyncResponseConsumerFactory<HttpResponse>
        PRODUCER_CONSUMER_FACTORY =
        new StatusCheckAsyncResponseConsumerFactory<>(
            x -> x == HttpStatus.SC_OK || x == HttpStatus.SC_CONFLICT,
            BasicAsyncResponseConsumerFactory.INSTANCE);

    private static final String STAFF_URI =
        "/v3/persons?&_sort=id";
    private static final String FIELDS =
        "login,work_email,official.is_dismissed,official.is_robot,official.position," +
            "department_group.id,department_group.ancestors.id,department_group.level," +
            "language,accounts,uid,_meta.message_id";
    private static final int LIMIT = 500;
    private static final int MAX_RETRIES = 10;
    private static final int DELAY = 10000;

    private static final String PRODUCER_NAME = "staff_consumer";
    //private static final String PRODUCER_NAME_PARAM = "&producer-name=";

    private final String staffUri;
    private final AsyncClient staffClient;
    private final AsyncClient producerClient;
    private final ImmutableStaffConsumerConfig config;
    private final PrefixedLogger logger;
    private final String name;
    private final StaffProducerLock lock;
    private final String sharedShard;

    public FullStaffConsumer(final StaffConsumerServer server) {
        staffUri = STAFF_URI + "&_fields=" + FIELDS + "&_limit=" + LIMIT;
        staffClient = server.staffClient();
        producerClient = server.producerClient();
        config = server.config();
        logger = server.logger();

        String name = System.getProperty("BSCONFIG_IHOST");
        if (name == null) {
            try {
                name = InetAddress.getLocalHost().getHostAddress();
            } catch (UnknownHostException ue) {
                name = "unknown";
            }
        }

        this.name = name;
        this.lock = new StaffProducerLock(server);
        this.sharedShard =
            Long.toString(config.updatePrefix().hash() % MAX_SHARDS);
    }

    @Override
    public void run() {
        logger.info("Starting consumer");
        String nextUri = staffUri;
        int retries = 0;
        long delay = 0;

        if (!lock.tryLock()) {
            logger.info("Lock failed");
            return;
        }

        logger.info("Locked");
        while (nextUri != null && retries < MAX_RETRIES) {
            try {
                if (delay > 0) {
                    logger.info("Delaying for " + delay);
                    Thread.sleep(delay);
                }
                Future<StaffPage> future = staffClient.execute(
                    new StaffRequestProducerSupplier(nextUri, config.token()),
                    StaffRecordsConsumerFactory.OK,
                    staffClient.httpClientContextGenerator(),
                    EmptyFutureCallback.INSTANCE);
                StaffPage current = future.get();
                process(current.people());
                logger.info("Uri processed " + nextUri + " next is " + current.next());
                nextUri = current.next();
            } catch (URISyntaxException | InterruptedException e) {
                logger.log(Level.WARNING, "Staff unrecoverable exeception", e);
                break;
            } catch (ExecutionException | IOException e) {
                logger.log(Level.WARNING, "Staff request execution error on url " + nextUri, e);
                retries += 1;
                delay = DELAY;
                continue;
            }

            retries = 0;
            delay = 0;
        }

        logger.info("Staff consumer finished " + retries);
    }

    protected Exception producerRequest(final BasicAsyncRequestProducerGenerator generator) {
        return producerRequest(generator, true);
    }

    protected Exception producerRequest(
        final BasicAsyncRequestProducerGenerator generator,
        final boolean relockOnExpired)
    {
        BasicAsyncRequestProducerGenerator request =
            new BasicAsyncRequestProducerGenerator(generator);

        request.addHeader(YandexHeaders.SERVICE, config.service());
        request.addHeader(YandexHeaders.PRODUCER_NAME, PRODUCER_NAME);
        request.addHeader(YandexHeaders.LOCKID, lock.token());

        logger.fine("Producer request " + generator);

        Future<HttpResponse> future =
            producerClient.execute(
                config.producerConfig().host(),
                request,
                PRODUCER_CONSUMER_FACTORY,
                EmptyFutureCallback.INSTANCE);

        Exception result = null;
        try {
            HttpResponse response = future.get();
            int status = response.getStatusLine().getStatusCode();
            if (relockOnExpired && status == HttpStatus.SC_FORBIDDEN) {
                logger.warning("Lock expired, trying to obtain " + lock.token());
                if (lock.tryLock()) {
                    return producerRequest(generator, false);
                }
            }

            if (status == HttpStatus.SC_CONFLICT) {
                logger.warning("Skipping, messageId that already indexed");
            } else if (status != HttpStatus.SC_OK) {
                return new BadResponseException(
                    request,
                    response,
                    CharsetUtils.toString(response.getEntity()));
            }
        } catch (ExecutionException e) {
            Throwable original = e.getCause();
            if (original instanceof BadResponseException) {
                int status = ((BadResponseException) original).statusCode();
                if (relockOnExpired && status == HttpStatus.SC_FORBIDDEN) {
                    logger.warning("Lock expired, trying to obtain " + lock.token());
                    if (lock.tryLock()) {
                        return producerRequest(generator, false);
                    }
                }

                logger.log(
                    Level.WARNING,
                    "Producer request failed " + request.toString(),
                    original);
                return (BadResponseException) original;
            }

            result = e;
        } catch (Exception e) {
            result = e;
        }

        return result;
    }

    protected void process(final List<Person> people) throws IOException {
        logger.info(
            "Processing staff records " + people.size());

        //first update shared one
        StringBuilder uri = new StringBuilder("/update?");
        uri.append("&staffloader=");
        uri.append(name);
        uri.append("&prefix=");
        uri.append(config.updatePrefix());
        uri.append("&service=");
        uri.append(config.service());
        uri.append("&batch-size=");
        uri.append(people.size());

        StringBuilder personalUri = new StringBuilder("/modify?");
        personalUri.append("&staffloader=");
        personalUri.append(name);
        personalUri.append("&service=");
        personalUri.append(config.service());
        personalUri.append("&personal");
        personalUri.append("&prefix=");

        String updateQueryPrefix = AceVenturaFields.EMAIL.prefixed() + ':';

        int uriLength = uri.length();
        int persLength = personalUri.length();

        MultipartEntityBuilder builder = MultipartEntityBuilder.create();
        builder.setMimeSubtype("mixed");
        StringBuilderWriter sbw = new StringBuilderWriter();
        JsonWriter writer = JsonType.NORMAL.create(sbw);
        for (Person person : people) {
            uri.setLength(uriLength);
            writer.reset();
            sbw.clear();
            personalUri.setLength(persLength);

            uri.append("&login=");
            uri.append(person.login());

            writer.startObject();
            writer.key("prefix");
            writer.value(config.updatePrefix());
            writer.key("query");
            writer.value(
                updateQueryPrefix + SearchRequestText.fullEscape(person.email(), false));
            //writer.value(updateQueryPrefix + person.email());
            writer.key("docs");
            writer.startArray();
            writer.startObject();
            writePerson(person, writer);
            writer.endObject();
            writer.endArray();
            writer.endObject();

            FormBodyPartBuilder partBuilder =
                FormBodyPartBuilder
                    .create()
                    .addField(YandexHeaders.ZOO_SHARD_ID, sharedShard)
                    .addField(YandexHeaders.URI, new String(uri))
                    .setBody(new StringBody(sbw.toString(), ContentType.APPLICATION_JSON))
                    .setName("envelope.json");

            builder.addPart(partBuilder.build());

            //logger.info("Update Shared " + uri.toString() + ' ' + sbw.toString());
            // now add personal record
            writer.reset();
            sbw.clear();

            AceVenturaPrefix prefix =
                new AceVenturaPrefix(person.uid(), UserType.PASSPORT_USER);
            personalUri.append(prefix);
            personalUri.append("&login=");
            personalUri.append(person.login());

            writer.startObject();
            writer.key("prefix");
            writer.value(prefix);
            writer.key("docs");
            writer.startArray();
            writer.startObject();

            writer.key(AceVenturaFields.ID.stored());
            writer.value(AceVenturaFields.corpStaffUrl(prefix));
            writer.key(AceVenturaFields.RECORD_TYPE.stored());
            writer.value(AceVenturaRecordType.CORP_STAFF.fieldValue());
            writePerson(person, writer);

            writer.endObject();
            writer.endArray();
            writer.endObject();
            partBuilder =
                FormBodyPartBuilder
                    .create()
                    .addField(
                        YandexHeaders.ZOO_SHARD_ID,
                        Long.toString(prefix.hash() % MAX_SHARDS))
                    .addField(YandexHeaders.URI, new String(personalUri))
                    .setBody(new StringBody(sbw.toString(), ContentType.APPLICATION_JSON))
                    .setName("envelope.json");

            builder.addPart(partBuilder.build());
            //logger.info("Update personal " + personalUri
            //    .toString() + ' ' + sbw.toString());
        }

        StringBuilder batchUri = new StringBuilder("/notify?&staffloader=");
        batchUri.append(name);
        batchUri.append("&service=");
        batchUri.append(config.service());
        batchUri.append("&batch-size=");
        batchUri.append(people.size());

        BasicAsyncRequestProducerGenerator request =
            new BasicAsyncRequestProducerGenerator(
                batchUri.toString(),
                builder.build());

        logger.info("Index request " + batchUri.toString());

        Exception e = producerRequest(request);
        if (e != null) {
            throw new IOException(e);
        }
    }


    protected void writePerson(
        final Person person,
        final JsonWriter writer)
        throws IOException
    {
        writer.key(AceVenturaFields.CORP_DISMISSED.stored());
        writer.value(person.dismissed());

        writer.key(AceVenturaFields.CORP_MESSENGERS_NICKS.stored());
        writer.value(person.messengersLogins());

        writer.key(AceVenturaFields.CORP_POSITION_NAME.stored());
        writer.value(person.positionName());

        writer.key(AceVenturaFields.CORP_POSITION_TYPE.stored());
        writer.value(person.positionType().name().toLowerCase(Locale.ENGLISH));

        writer.key(AceVenturaFields.CORP_DEPARTMENT_ID.stored());
        writer.value(person.depId());

        writer.key(AceVenturaFields.CORP_DEPARTMENT_LEVEL.stored());
        writer.value(person.depLevel());

        writer.key(AceVenturaFields.CORP_DEPARTMENTS.stored());
        writer.value(person.departments());
    }
}
