package ru.yandex.mail.search.staff.consumer;

import java.io.IOException;
import java.net.URISyntaxException;
import java.util.ArrayDeque;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;

import org.apache.http.HttpException;

import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.BadResponseException;
import ru.yandex.http.util.EmptyFutureCallback;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.mail.search.staff.StaffMessageIdGrouppedPage;
import ru.yandex.mail.search.staff.config.ImmutableStaffConsumerConfig;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.stater.IntegralSumAggregatorFactory;
import ru.yandex.stater.MaxAggregatorFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;

public class StaffClient {
    public static final String EQUALS_OPERATION = "==";
    public static final String GREATER_OR_EQUALS_OPERATION = ">=";
    public static final String GREATER_OPERATION = ">";

    private static final String FIELDS
        = "name,login,work_email,official,department_group,language,accounts,uid,_meta.message_id";
    private static final String MESSAGE_ID = "_meta.message_id";
    private enum State {
        NONE,
        SINGLE,
        RANGE
    }

    private final ImmutableStaffConsumerConfig config;
    private final AsyncClient staffClient;
    private final PrefixedLogger logger;
    private final ArrayDeque<StaffMessage> queue = new ArrayDeque<>();
    private final TimeFrameQueue<Long> messageIdStater;
    private final TimeFrameQueue<Integer> recordsToIndex;
    private final TimeFrameQueue<Long> errors;
    private long messageId = -1L;

    public StaffClient(
        final StaffConsumerServer server)
    {
        this.config = server.config();
        this.logger = server.logger().addPrefix("StaffClient");
        this.staffClient = server.staffClient();

        messageIdStater = new TimeFrameQueue<>(server.config().metricsTimeFrame());
        server.registerStater(
            new PassiveStaterAdapter<>(
                messageIdStater,
                new NamedStatsAggregatorFactory<>(
                    "staff-message-id_axxx",
                    new MaxAggregatorFactory(0L))));
        recordsToIndex = new TimeFrameQueue<>(server.config().metricsTimeFrame());
        server.registerStater(
            new PassiveStaterAdapter<>(
                recordsToIndex,
                new NamedStatsAggregatorFactory<>(
                    "staff-records-for-index_amm",
                    IntegralSumAggregatorFactory.INSTANCE)));
        errors = new TimeFrameQueue<>(server.config().metricsTimeFrame());
        server.registerStater(
            new PassiveStaterAdapter<>(
                errors,
                new NamedStatsAggregatorFactory<>(
                    "staff-errors_amm",
                    IntegralSumAggregatorFactory.INSTANCE)));
    }

    public void reset(final long messageId) {
        this.messageId = messageId;
    }

    protected StringBuilder singleMessageIdRequest(final long messageId) throws BadRequestException {
        QueryConstructor qc =
            new QueryConstructor(config.staffClientConfig().uri().toString());

        qc.append("_fields", FIELDS);
        qc.append(
            "_query",
            MESSAGE_ID + EQUALS_OPERATION + messageId);
        return qc.sb();
    }

    protected StringBuilder rangeRequest(final long messageId) throws BadRequestException {
        QueryConstructor qc =
            new QueryConstructor(config.staffClientConfig().uri().toString());

        qc.append("_fields", FIELDS);
        qc.append(
            "_query",
            MESSAGE_ID + GREATER_OR_EQUALS_OPERATION + messageId);
        qc.append("_sort", MESSAGE_ID);
        return qc.sb();
    }

    private void fetch() throws IOException, HttpException {
        State state = State.RANGE;
        String nextUri = rangeRequest(messageId).toString();
        StaffMessageIdGrouppedPage result = null;
        while (true) {
            try {
                messageIdStater.accept(messageId);

                logger.info("Staff request " + nextUri);
                Future<StaffMessageIdGrouppedPage> future = staffClient.execute(
                    new StaffRequestProducerSupplier(nextUri, config.token()),
                    PersonsConsumerFactory.OK,
                    staffClient.httpClientContextGenerator(),
                    EmptyFutureCallback.INSTANCE);
                StaffMessageIdGrouppedPage current = future.get();

                if (state == State.SINGLE) {
                    if (result != null) {
                        result.appendNext(current);
                    } else {
                        result = current;
                    }

                    if (current.next() != null) {
                        nextUri = result.next();
                        continue;
                    }

                    for (Map.Entry<Long, StaffMessage> entry: result.messages().entrySet()) {
                        queue.add(entry.getValue());
                        recordsToIndex.accept(entry.getValue().people().size());
                    }

                    logger.info(
                        "Single mode, retrieved records: "
                            + result.messages().size() + " " + result.maxId());

                    if (result.total() > 0) {
                        if (result.maxId() < messageId) {
                            logger.severe(
                                "Max message id from batch is less than current, something gone wrong "
                                    + result.total()
                                    + ' ' + messageId);
                            errors.accept(1L);
                        } else {
                            this.messageId = result.maxId() + 1;
                        }
                    } else {
                        logger.info("No records in response, increasing messageId b one");
                        this.messageId += 1;
                        nextUri = singleMessageIdRequest(this.messageId).toString();
                        continue;
                    }

                    break;
                } else {
                    if (result != null) {
                        result.appendNext(current);
                    } else {
                        result = current;
                    }

                    if (result.messages().size() <= 1 && result.next() != null) {
                        nextUri = result.next();
                        logger.info("Not enough results, message ids " + result.messages().size()
                            + " trying next " + nextUri);

                        continue;
                    }

                    if (current.next() != null) {
                        int i = 0;
                        for (Map.Entry<Long, StaffMessage> entry: result.messages().entrySet()) {
                            this.messageId = entry.getKey();

                            if (i == result.messages().size() - 1) {
                                // except last message
                                break;
                            }

                            this.queue.add(entry.getValue());
                            recordsToIndex.accept(entry.getValue().people().size());
                            i++;
                        }
                        logger.info(
                            "Range mode, reached more than 1 messageIds, message ids: "
                                + result.messages().size()
                                + " message id now is " + messageId);
                    } else {
                        for (Map.Entry<Long, StaffMessage> entry
                            : result.messages().entrySet())
                        {
                            this.queue.add(entry.getValue());
                            recordsToIndex.accept(entry.getValue().people().size());
                            this.messageId = entry.getKey();
                        }

                        if (result.total() > 0) {
                            this.messageId += 1;
                        }

                        logger.info(
                            "Range mode, reached page limit, message ids: "
                                + result.messages().size()
                                + " extracted from records "
                                + result.total()
                                + ", message id now is " + messageId);
                    }

                    break;
                }
            } catch (URISyntaxException | InterruptedException e) {
                throw new IOException("Staff request failed", e);
            } catch (ExecutionException e) {
                Throwable original = e.getCause();
                if (original instanceof BadResponseException) {
                    BadResponseException bre = (BadResponseException) original;
                    if (bre.response() != null
                        && (bre.response().contains("Overflow sort stage buffered data usage")
                        || bre.response().contains("Sort operation used more than the maximum")))
                    {
                        logger.info("Overflow in db due to sort, trying fetch single message " + messageId);
                        state = State.SINGLE;
                        if (result != null && result.total() > 0 && result.minId() < Long.MAX_VALUE) {
                            logger.info(
                                "Minimal id in scope was " + result.minId() + " use it as starting point");
                            messageId = result.minId();
                        }

                        nextUri = singleMessageIdRequest(messageId).toString();
                        result = null;
                        continue;
                    }
                }

                throw new HttpException("Staff request failed " + nextUri, original);
            }
        }
    }

    public StaffMessage next() throws IOException, HttpException {
        if (queue.size() > 0) {
            return queue.poll();
        }

        logger.info("Fetching starting from message id " + messageId);
        fetch();
        logger.info("Fetched, message id  " + messageId + " queue size " + queue.size());
        return queue.poll();
    }
}
