package ru.yandex.mail.search.staff.consumer;

import java.net.URISyntaxException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;

import org.apache.http.HttpException;

import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.EmptyFutureCallback;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.mail.search.staff.Person;
import ru.yandex.mail.search.staff.config.ImmutableStaffConsumerConfig;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.stater.IntegralSumAggregatorFactory;
import ru.yandex.stater.MaxAggregatorFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;

public class StaffClient2 {
    private static final String FIELDS =
        "login,work_email,official.is_dismissed,official.is_robot,official.position," +
            "department_group.id,department_group.ancestors.id,department_group.level," +
            "language,accounts,uid,id,name";

    private final ImmutableStaffConsumerConfig config;
    private final AsyncClient staffClient;
    private final PrefixedLogger logger;
    private final TimeFrameQueue<Long> messageIdStater;
    private final TimeFrameQueue<Long> errors;

    public StaffClient2(final StaffConsumerServer server) {
        this.config = server.config();
        this.logger = server.logger().addPrefix("StaffClient");
        this.staffClient = server.staffClient();

        messageIdStater = new TimeFrameQueue<>(server.config().metricsTimeFrame());
        server.registerStater(
            new PassiveStaterAdapter<>(
                messageIdStater,
                new NamedStatsAggregatorFactory<>(
                    "staff-message-id_axxx",
                    new MaxAggregatorFactory(0L))));
        errors = new TimeFrameQueue<>(server.config().metricsTimeFrame());
        server.registerStater(
            new PassiveStaterAdapter<>(
                errors,
                new NamedStatsAggregatorFactory<>(
                    "staff-errors_amm",
                    IntegralSumAggregatorFactory.INSTANCE)));
    }

    public Person one(final String login) throws HttpException, InterruptedException {
        try {
            QueryConstructor qc =
                new QueryConstructor(config.staffClientConfig().uri().toString());

            qc.append("_fields", FIELDS);
            qc.append("login", login);
            //qc.append("_one", "1");

            String nextUri = qc.toString();

            logger.info("Staff request " + nextUri);
            Future<StaffPage> future = staffClient.execute(
                new StaffRequestProducerSupplier(nextUri, config.token()),
                StaffRecordsConsumerFactory.OK,
                staffClient.httpClientContextGenerator(),
                EmptyFutureCallback.INSTANCE);
            StaffPage page = future.get();
            if (page.people().size() == 0) {
                return null;
            } else {
                return page.people().get(0);
            }
        } catch (URISyntaxException | BadRequestException nre) {
            errors.accept(1L);
            throw new HttpException("Unable to prepare staff api request", nre);
        } catch (ExecutionException ee) {
            errors.accept(1L);
            throw new HttpException("Execute request failed", ee);
        }
    }

    protected StaffPage next(final long id) throws InterruptedException, HttpException {
        try {
            QueryConstructor qc =
                new QueryConstructor(config.staffClientConfig().uri().toString());

            qc.append("_fields", FIELDS);
            qc.append(
                "_query",
                "id>" + id);
            qc.append("_sort", "id");

            String nextUri = qc.toString();

            logger.info("Staff request " + nextUri);
            Future<StaffPage> future = staffClient.execute(
                new StaffRequestProducerSupplier(nextUri, config.token()),
                StaffRecordsConsumerFactory.OK,
                staffClient.httpClientContextGenerator(),
                EmptyFutureCallback.INSTANCE);
            return future.get();
        } catch (URISyntaxException | BadRequestException nre) {
            errors.accept(1L);
            throw new HttpException("Unable to prepare staff api request", nre);
        } catch (ExecutionException ee) {
            errors.accept(1L);
            throw new HttpException("Execute request failed", ee);
        }
    }
}
