package ru.yandex.mail.search.staff.consumer;

import java.io.IOException;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.logging.Level;

import org.apache.http.HttpEntity;
import org.apache.http.HttpException;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.FormBodyPartBuilder;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.entity.mime.content.StringBody;

import ru.yandex.ace.ventura.AceVenturaFields;
import ru.yandex.ace.ventura.AceVenturaPrefix;
import ru.yandex.ace.ventura.AceVenturaRecordType;
import ru.yandex.ace.ventura.UserType;
import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.http.util.BadResponseException;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.EmptyFutureCallback;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.BasicAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.StatusCheckAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.mail.search.staff.Person;
import ru.yandex.mail.search.staff.config.ImmutableStaffConsumerConfig;
import ru.yandex.parser.searchmap.SearchMap;
import ru.yandex.parser.searchmap.SearchMapRow;
import ru.yandex.parser.searchmap.SearchMapShard;
import ru.yandex.search.prefix.StringPrefix;
import ru.yandex.search.request.util.SearchRequestText;
import ru.yandex.stater.IntegralSumAggregatorFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;

public class StaffConsumer implements Runnable {
    private static final int MAX_SHARDS = 65534;
    private static final StatusCheckAsyncResponseConsumerFactory<HttpResponse>
        PRODUCER_CONSUMER_FACTORY =
        new StatusCheckAsyncResponseConsumerFactory<>(
            x -> x == HttpStatus.SC_OK || x == HttpStatus.SC_CONFLICT,
            BasicAsyncResponseConsumerFactory.INSTANCE);


    private static final String PRODUCER_NAME = "staff_consumer";
    private static final String PRODUCER_NAME_PARAM = "&producer-name=";

    private final StaffClient2 staffClient;
    private final AsyncClient producerClient;

    private final String name;

    private final StaffProducerLock lock;
    private final String getPositionRequest;

    private final PrefixedLogger logger;
    private final ImmutableStaffConsumerConfig config;
    private final Thread thread;

    private final List<StaffIndexModule> indexers;
    private final TimeFrameQueue<Integer> recordsToIndex;

    private volatile boolean stopped = false;

    public StaffConsumer(final StaffConsumerServer server, final StaffClient2 staffClient) throws IOException {
        this.logger = server.logger();
        this.config = server.config();
        this.lock = new StaffProducerLock(server);

        try {
            SearchMap searchMap = config.searchmap().build();

            this.indexers =
                Collections.unmodifiableList(
                    Arrays.asList(
                        new OwnInfoIndexer(),
                        new SharedListInfoIndexer(config.updatePrefix()),
                        new PersonalBooksUpdater(config, searchMap)));
        } catch (ParseException pe) {
            throw new IOException(pe);
        }

        String name = System.getProperty("BSCONFIG_IHOST");
        if (name == null) {
            try {
                name = InetAddress.getLocalHost().getHostAddress();
            } catch (UnknownHostException ue) {
                name = "unknown";
            }
        }

        this.staffClient = staffClient;
        producerClient = server.producerClient();

        getPositionRequest =
            "/_producer_position?service=" + config.service()
                + PRODUCER_NAME_PARAM + PRODUCER_NAME;

        this.name = name;

        thread = new Thread(this);

        recordsToIndex = new TimeFrameQueue<>(server.config().metricsTimeFrame());
        server.registerStater(
            new PassiveStaterAdapter<>(
                recordsToIndex,
                new NamedStatsAggregatorFactory<>(
                    "staff-records-fetched_ammm",
                    IntegralSumAggregatorFactory.INSTANCE)));
    }

    public PrefixedLogger logger() {
        return logger;
    }

    public void close() {
        this.stopped = true;
    }

    public void start() {
        thread.start();
    }

    protected void doRun() throws InterruptedException {
        if (!lock.tryLock()) {
            return;
        }

        Long messageId = lastMessageId();
        if (messageId == null) {
            return;
        }

        while (!stopped && lock.locked()) {
            StaffPage message;
            try {
                message = staffClient.next(messageId);
            } catch (HttpException e) {
                logger.log(Level.WARNING, "Failed to fetch staff data", e);
                Thread.sleep(config.checkInterval());
                continue;
            }
            recordsToIndex.accept(message.people().size());

            if (message.people().size() == 0) {
                logger.fine("No new staff records");
                Thread.sleep(2 * config.checkInterval());
                continue;
            }

            logger.fine(
                "Consuming staff record " + message.people().size()
                    + " request from with message id " + messageId);

            try {
                long maxMessageId = consume(message.people(), messageId);
                if (maxMessageId < messageId) {
                    logger.severe("Message id " + maxMessageId + " is less than previous " + messageId);
                    Thread.sleep(2 * config.checkInterval());
                    continue;
                }

                messageId = maxMessageId;
            } catch (IOException ioe) {
                logger.log(Level.WARNING, "Index failed for " + messageId, ioe);
            }

            Thread.sleep(config.checkInterval());
        }
    }

    @Override
    public void run() {
        try {
            while (!stopped) {
                try {
                    doRun();
                } catch (Exception e) {
                    logger.log(Level.WARNING, "Unexpected error in StaffConsumer", e);
                }

                Thread.sleep(config.checkInterval());
            }
        } catch (InterruptedException ie) {
            logger.log(Level.WARNING, "Staff consumer thread interruped", ie);
        }
    }


    protected static void writePerson(
        final Person person,
        final JsonWriter writer)
        throws IOException
    {
        writer.key(AceVenturaFields.CORP_DISMISSED.stored());
        if (person.dismissed()) {
            writer.value(person.dismissed());
        } else {
            writer.value(JsonNull.INSTANCE);
        }

        writer.key(AceVenturaFields.CORP_MESSENGERS_NICKS.stored());
        writer.value(person.messengersLogins());

        writer.key(AceVenturaFields.EN_NAMES.stored());
        writer.value(person.englishNames());

        writer.key(AceVenturaFields.CORP_POSITION_NAME.stored());
        writer.value(person.positionName());

        writer.key(AceVenturaFields.CORP_POSITION_TYPE.stored());
        writer.value(person.positionType().name().toLowerCase(Locale.ENGLISH));

        writer.key(AceVenturaFields.CORP_DEPARTMENT_ID.stored());
        writer.value(person.depId());

        writer.key(AceVenturaFields.CORP_DEPARTMENT_LEVEL.stored());
        writer.value(person.depLevel());

        writer.key(AceVenturaFields.CORP_DEPARTMENTS.stored());
        writer.value(person.departments());
    }

    protected long consume(
        final List<Person> people,
        final long currentMessageId)
        throws IOException
    {
        long maxId = Long.MIN_VALUE;

        MultipartEntityBuilder builder = MultipartEntityBuilder.create();
        builder.setMimeSubtype("mixed");

        StringBuilder commonUriBuilder = new StringBuilder();
        commonUriBuilder.append("&staffloader=");
        commonUriBuilder.append(name);
        commonUriBuilder.append("&service=");
        commonUriBuilder.append(config.service());
        commonUriBuilder.append("&message-id=");
        commonUriBuilder.append(currentMessageId);

        String commonUri = commonUriBuilder.toString();

        for (Person person: people) {
            if (person.messageId() > maxId) {
                maxId = person.messageId();
            }

            if (!person.email().toLowerCase(Locale.ENGLISH).contains("@yandex-team.")) {
                logger.warning("skipping, not yateam " + person);
                continue;
            }

            for (StaffIndexModule indexer: indexers) {
                indexer.index(builder, person, commonUri);
            }
        }

        StringBuilder batchUri = new StringBuilder("/notify?&staffloader=");
        batchUri.append(name);
        batchUri.append("&service=");
        batchUri.append(config.service());
        batchUri.append("&message-id=");
        batchUri.append(currentMessageId);
        batchUri.append("&batch-size=");
        batchUri.append(people.size());

        HttpEntity entity = builder.build();
        BasicAsyncRequestProducerGenerator request =
            new BasicAsyncRequestProducerGenerator(
                batchUri.toString(),
                entity);

        logger().info("Index request " + batchUri.toString());

        Exception e = producerRequest(request);
        if (e != null) {
            throw new IOException(e);
        }

        return maxId;
    }

    protected Exception producerRequest(final BasicAsyncRequestProducerGenerator generator) {
        return producerRequest(generator, true);
    }

    protected Exception producerRequest(
        final BasicAsyncRequestProducerGenerator generator,
        final boolean relockOnExpired)
    {
        BasicAsyncRequestProducerGenerator request =
            new BasicAsyncRequestProducerGenerator(generator);

        request.addHeader(YandexHeaders.SERVICE, config.service());
        request.addHeader(YandexHeaders.PRODUCER_NAME, PRODUCER_NAME);
        request.addHeader(YandexHeaders.LOCKID, lock.token());

        logger.fine("Producer request " + generator);

        Future<HttpResponse> future =
            producerClient.execute(
                config.producerConfig().host(),
                request,
                PRODUCER_CONSUMER_FACTORY,
                EmptyFutureCallback.INSTANCE);

        Exception result = null;
        try {
            HttpResponse response = future.get();
            int status = response.getStatusLine().getStatusCode();
            if (relockOnExpired && status == HttpStatus.SC_FORBIDDEN) {
                logger.warning("Lock expired, trying to obtain " + lock.token());
                if (lock.tryLock()) {
                    return producerRequest(generator, false);
                }
            }

            if (status == HttpStatus.SC_CONFLICT) {
                logger.warning("Skipping, messageId that already indexed");
            } else if (status != HttpStatus.SC_OK) {
                return new BadResponseException(
                    request,
                    response,
                    CharsetUtils.toString(response.getEntity()));
            }
        } catch (ExecutionException e) {
            Throwable original = e.getCause();
            if (original instanceof BadResponseException) {
                int status = ((BadResponseException) original).statusCode();
                if (relockOnExpired && status == HttpStatus.SC_FORBIDDEN) {
                    logger.warning("Lock expired, trying to obtain " + lock.token());
                    if (lock.tryLock()) {
                        return producerRequest(generator, false);
                    }
                }

                logger().log(
                    Level.WARNING,
                    "Producer request failed " + request.toString(),
                    original);
                return (BadResponseException) original;
            }

            result = e;
        } catch (Exception e) {
            result = e;
        }

        return result;
    }

    protected Long lastMessageId() {
        Future<HttpResponse> future =
            producerClient.execute(
                config.producerConfig().host(),
                new BasicAsyncRequestProducerGenerator(getPositionRequest),
                EmptyFutureCallback.INSTANCE);
        try {
            HttpResponse response = future.get();
            int status = response.getStatusLine().getStatusCode();
            if (status != HttpStatus.SC_OK) {
                throw new BadResponseException(
                    new HttpGet(getPositionRequest),
                    response);
            }
            String body = CharsetUtils.toString(response.getEntity());

            long messageId = Long.parseLong(body);
            logger().info(
                "Last message id " + messageId);
            return messageId;
        } catch (IOException
            | HttpException
            | NumberFormatException
            | InterruptedException
            | ExecutionException e) {
            logger().log(Level.WARNING, "Failed to get position", e);
        }

        return null;
    }

    private interface StaffIndexModule {
        void index(
            final MultipartEntityBuilder builder,
            final Person person,
            final String commonUri)
            throws IOException;
    }

    private static class SharedListInfoIndexer implements StaffIndexModule {
        private static final String UPDATE_QUERY_BASE =
                AceVenturaFields.EMAIL.prefixed() + ':';

        private final AceVenturaPrefix prefix;

        public SharedListInfoIndexer(AceVenturaPrefix prefix) {
            this.prefix = prefix;
        }

        @Override
        public void index(
            final MultipartEntityBuilder builder,
            final Person person,
            final String commonUri)
            throws IOException
        {
            StringBuilder uri = new StringBuilder("/update?");
            uri.append(commonUri);
            uri.append("&login=");
            uri.append(person.login());
            uri.append("&prefix=");
            uri.append(prefix);

            StringBuilderWriter sbw = new StringBuilderWriter();
            try (JsonWriter writer = JsonType.NORMAL.create(sbw)) {
                writer.startObject();
                writer.key("prefix");
                writer.value(prefix);
                writer.key("query");
                writer.value(
                    UPDATE_QUERY_BASE + SearchRequestText.fullEscape(person.email(), false));
                //writer.value(updateQueryPrefix + person.email());
                writer.key("docs");
                writer.startArray();
                writer.startObject();
                writePerson(person, writer);
                writer.endObject();
                writer.endArray();
                writer.endObject();
            }
            System.err.println(sbw.toString());

            builder.addPart(
                FormBodyPartBuilder
                    .create()
                    .addField(
                        YandexHeaders.ZOO_SHARD_ID,
                        Long.toString(prefix.hash() % MAX_SHARDS))
                    .addField(YandexHeaders.URI, new String(uri))
                    .addField(
                        YandexHeaders.PRODUCER_POSITION,
                        String.valueOf(person.messageId()))
                    .setBody(new StringBody(sbw.toString(), ContentType.APPLICATION_JSON))
                    .setName("envelope.json")
                    .build());
        }
    }

    private static class OwnInfoIndexer implements StaffIndexModule {
        @Override
        public void index(
            final MultipartEntityBuilder builder,
            final Person person,
            final String commonUri)
            throws IOException
        {
            StringBuilder personalUri = new StringBuilder("/modify?");
            personalUri.append(commonUri);
            personalUri.append("&personal");
            personalUri.append("&prefix=");
            AceVenturaPrefix prefix =
                new AceVenturaPrefix(person.uid(), UserType.PASSPORT_USER);
            personalUri.append(prefix);
            personalUri.append("&login=");
            personalUri.append(person.login());

            StringBuilderWriter sbw = new StringBuilderWriter();
            try (JsonWriter writer = JsonType.NORMAL.create(sbw)) {
                writer.startObject();
                writer.key("prefix");
                writer.value(prefix);
                writer.key("docs");
                writer.startArray();
                writer.startObject();

                writer.key(AceVenturaFields.ID.stored());
                writer.value(AceVenturaFields.corpStaffUrl(prefix));
                writer.key(AceVenturaFields.RECORD_TYPE.stored());
                writer.value(AceVenturaRecordType.CORP_STAFF.fieldValue());
                writePerson(person, writer);

                writer.endObject();
                writer.endArray();
                writer.endObject();
            }

            System.err.println(sbw.toString());
            builder.addPart(
                FormBodyPartBuilder
                    .create()
                    .addField(
                        YandexHeaders.ZOO_SHARD_ID,
                        Long.toString(prefix.hash() % MAX_SHARDS))
                    .addField(YandexHeaders.URI, new String(personalUri))
                    .addField(
                        YandexHeaders.PRODUCER_POSITION,
                        String.valueOf(person.messageId()))
                    .setBody(new StringBody(sbw.toString(), ContentType.APPLICATION_JSON))
                    .setName("envelope.json")
                    .build());
        }
    }

    private static class PersonalBooksUpdater implements StaffIndexModule {
        private static final String UPDATE_QUERY_BASE =
            AceVenturaFields.IS_CORP.global() + ":1 AND "
            + AceVenturaFields.EMAIL.global() + ':';

        private final List<Integer> shards;
        private final List<StringPrefix> prefixes;
        //private final int backendShards;

        public PersonalBooksUpdater(
            final ImmutableStaffConsumerConfig config,
            final SearchMap searchmap)
        {
            SearchMapRow row = searchmap.row(config.service());
            Map<Integer, Integer> map = new LinkedHashMap<>();
            for (int i = 0; i < SearchMap.SHARDS_COUNT; i++) {
                SearchMapShard shard = row.get(i);
                map.putIfAbsent(shard.iNum(), i);
            }

            this.shards = Collections.unmodifiableList(new ArrayList<>(map.values()));
            //on indexation part - all prefixes are parsed as string prefix
            Map<Integer, StringPrefix> prefixes = new LinkedHashMap<>();
            for (int i = 0; i < config.searchBackendShards() * 100; i++) {
                StringPrefix prefix = new StringPrefix(i + "$staff_update");
                int hash = (int) (prefix.hash() % config.searchBackendShards());

                prefixes.putIfAbsent(hash, prefix);

                if (prefixes.size() == config.searchBackendShards()) {
                    break;
                }
            }

            if (prefixes.size() != config.searchBackendShards()) {
                throw new RuntimeException("Can not generate enough prefixes");
            }
            this.prefixes = Collections.unmodifiableList(new ArrayList<>(prefixes.values()));
        }

        @Override
        public void index(
            final MultipartEntityBuilder builder,
            final Person person,
            final String commonUri)
            throws IOException
        {
            StringBuilder personalBooksUri = new StringBuilder("/update?");
            personalBooksUri.append(commonUri);
            personalBooksUri.append("&personal_books");

            personalBooksUri.append("&login=");
            personalBooksUri.append(person.login());
            personalBooksUri.append("&email=");
            personalBooksUri.append(person.email());
            int length = personalBooksUri.length();

            StringBuilderWriter sbw = new StringBuilderWriter();
            for (Integer shard: shards) {
                for (StringPrefix prefix: prefixes) {
                    sbw.clear();
                    try (JsonWriter writer = JsonType.NORMAL.create(sbw)) {
                        writer.startObject();
                        writer.key("prefix");
                        writer.value(prefix);
                        writer.key("query");
                        writer.value(
                            UPDATE_QUERY_BASE + SearchRequestText.fullEscape(person.email(), false));
                        writer.key("docs");
                        writer.startArray();
                        writer.startObject();
                        writePerson(person, writer);
                        writer.endObject();
                        writer.endArray();
                        writer.endObject();
                    }

                    personalBooksUri.setLength(length);
                    personalBooksUri.append("&shard=");
                    personalBooksUri.append(shard);
                    personalBooksUri.append("&prefix=");
                    personalBooksUri.append(prefix);

                    System.err.println(sbw.toString());
                    builder.addPart(
                        FormBodyPartBuilder
                            .create()
                            .addField(
                                YandexHeaders.ZOO_SHARD_ID,
                                Long.toString(shard % MAX_SHARDS))
                            .addField(YandexHeaders.URI, new String(personalBooksUri))
                            .addField(
                                YandexHeaders.PRODUCER_POSITION,
                                String.valueOf(person.messageId()))
                            .setBody(new StringBody(sbw.toString(), ContentType.APPLICATION_JSON))
                            .setName("envelope.json")
                            .build());
                }
            }
        }
    }
}
