package ru.yandex.mail.search.staff.consumer;

import java.io.IOException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.methods.HttpGet;

import ru.yandex.http.util.BadResponseException;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.EmptyFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.mail.search.staff.config.ImmutableStaffConsumerConfig;

public class StaffProducerLock {
    private static final String PRODUCER_NAME = "staff_consumer";
    private static final String PRODUCER_NAME_PARAM = "&producer-name=";

    private String token = null;
    private long expireTs = -1;
    private final ImmutableStaffConsumerConfig config;
    private final StaffConsumerServer server;
    private final String lockRequest;

    public StaffProducerLock(final StaffConsumerServer server) {
        this.config = server.config();
        this.server = server;
        lockRequest =
            "/_producer_lock?service=" + config.service()
                + "&session-timeout=" + config.lockTimeout()
                + PRODUCER_NAME_PARAM + PRODUCER_NAME;
    }

    public boolean tryLock() {
        reset();

        HttpGet request = new HttpGet(lockRequest);

        Future<HttpResponse> future =
            server.producerClient().execute(
                config.producerConfig().host(),
                new BasicAsyncRequestProducerGenerator(lockRequest),
                EmptyFutureCallback.INSTANCE);
        try {
            HttpResponse response = future.get();
            int status = response.getStatusLine().getStatusCode();
            if (status == HttpStatus.SC_FORBIDDEN) {
                server.logger().fine(
                    "Failed to obtain lock, current is "
                        + CharsetUtils.toString(response.getEntity()));
            } else if (status != HttpStatus.SC_OK) {
                throw new BadResponseException(request, response);
            } else {
                String lockId = CharsetUtils.toString(response.getEntity());
                server.logger().fine("Locked with " + lockId);

                this.token = lockId;
                this.expireTs = System.currentTimeMillis() + config.lockTimeout();

                return true;
            }
        } catch (IOException
            | HttpException
            | InterruptedException
            | ExecutionException e)
        {
            server.logger().log(Level.WARNING, "Failed to get lock", e);
        }

        return false;
    }

    public boolean locked() {
        return token != null && System.currentTimeMillis() < expireTs;
    }

    public String token() {
        return token;
    }

    public void reset() {
        token = null;
        expireTs = -1L;
    }
}
