package ru.yandex.search.backpack.client;

import java.io.IOException;
import java.net.URISyntaxException;
import java.text.ParseException;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.http.HttpException;

import ru.yandex.client.tvm2.Tvm2TicketRenewalTask;
import ru.yandex.collection.Pattern;
import ru.yandex.http.proxy.HttpProxy;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.parser.JsonException;
import ru.yandex.msearch.Index;
import ru.yandex.parser.searchmap.SearchMap;
import ru.yandex.search.backpack.client.handlers.BackpackClientIndexHandler;
import ru.yandex.search.backpack.client.handlers.BackpackClientIndexRestoreHandler;
import ru.yandex.search.backpack.client.handlers.BackpackClientPingHandler;
import ru.yandex.search.backpack.client.handlers.BackpackClientShardControlHandler;
import ru.yandex.search.backpack.client.handlers.BackpackClientSimpleHandler;
import ru.yandex.search.backpack.client.handlers.BackpackClientSimpleRestoreHandler;
import ru.yandex.search.backpack.client.handlers.BackpackClientStatHandler;

public class BackPackClient extends HttpProxy<ImmutableBackPackClientConfig> {
    //private static final int MAX_POOL_SIZE = 10;
    private static final String PING = "/ping";
    //private static final String TVM2 = "tvm2";

    private final AsyncClient mdsWriterClient;
    private final AsyncClient mdsReaderClient;
    private final AsyncClient metaServerClient;
    private final AsyncClient consumerClient;
    private final String namespace;
    private final String expire;
    private final boolean directupload;
    private final boolean toSalt;
    private final int directproxyport;
    private final int clientinum;
    private final String clienthostname;
    private final java.util.regex.Pattern nohashext;
    private final java.util.regex.Pattern passext;

    public BackPackRequestContext context;
    private ConcurrentHashMap<String, ConcurrentHashMap<String, Map<String,String>>> backupstat;

    private final SearchMap searchMap;

    private final Tvm2TicketRenewalTask tvm2RenewalTask;

    public BackPackClient(final ImmutableBackPackClientConfig config)
        throws HttpException, IOException, ParseException, URISyntaxException, JsonException
    {
        super(config);
        this.logger().info("BackPack default client starting.");
        this.logger().info("BackPack mds client id: " + config.mdsWriterTvmClientId());

        mdsWriterClient = client("MDS-Writer", config.mdsWriterConfig());

        mdsReaderClient = client("MDS-Reader", config.mdsWriterConfig());

        metaServerClient = client("META-server", config.metaServerConfig());

        consumerClient =
            client("CONSUMER-server", config.consumerServerConfig());

        namespace = config.nameSpace();
        directupload = config.directUpload();
        toSalt = config.toSalt();
        expire = config.expire();
        directproxyport = config.directProxyPort();
        clientinum = config.clientInum();
        clienthostname = config.clientHostname();
        nohashext = java.util.regex.Pattern.compile(config.noHashExt());
        passext = java.util.regex.Pattern.compile(config.passExt());

        this.backupstat = new ConcurrentHashMap<>();
        this.searchMap = config.searchMapConfig().build();

        tvm2RenewalTask = new Tvm2TicketRenewalTask(
                logger().addPrefix("tvm2"),
                serviceContextRenewalTask,
                config.tvm2ClientConfig());

        //TODO: Check tvm task is started
        //tvm2RenewalTask.start();

        register(new Pattern<>("/backup", true),
                new BackpackClientSimpleHandler(this), "POST");

        register(new Pattern<>("/restore", true),
                new BackpackClientSimpleRestoreHandler(this), "POST");

        BackpackClientStatHandler statHandler = new BackpackClientStatHandler(this);
        register(new Pattern<>("/backup/backupstat", true), statHandler);

        register(new Pattern<>(PING, true), new BackpackClientPingHandler(this), "GET");
    }


    public BackPackClient(final ImmutableBackPackClientConfig config, Index index)
            throws HttpException, IOException, ParseException, URISyntaxException, JsonException
    {
        this(config);

        this.logger().info("BackPack lucene detected. Lucene backup client starting.");

        register(new Pattern<>("/backup/lucene", true),
                new BackpackClientIndexHandler(this, index), "POST");

        register(new Pattern<>("/backup/lucene/shard", true),
                new BackpackClientShardControlHandler(this, index), "GET");

        register(new Pattern<>("/restore", true),
                new BackpackClientIndexRestoreHandler(this, index), "POST");

        this.logger().info("Search backend handler initialized for path:" + index.config().indexPath().getPath());

    }

    public SearchMap searchMap() {
        return searchMap;
    }

    public ConcurrentHashMap<String, ConcurrentHashMap<String, Map<String,String>>> backupStat() {
        return backupstat;
    }

    public AsyncClient getMdsWriterClient() {
        return mdsWriterClient;
    }

    public AsyncClient getMdsReaderClient() {
        return mdsReaderClient;
    }

    public AsyncClient getMetaServerClient() {
        return metaServerClient;
    }

    public AsyncClient getConsumerClient() {
        return consumerClient;
    }

    public String getNamespace() {
        return namespace;
    }

    public String getClienthostname() {
        return clienthostname;
    }

    public int getClientinum() {
        return clientinum;
    }

    public boolean getDirectUpload() {
        return directupload;
    }

    public boolean getToSalt() {
        return toSalt;
    }

    public String getExpire() {
        return expire;
    }

    public java.util.regex.Pattern getNoHashExt(){
        return nohashext;
    }

    public java.util.regex.Pattern getPassExt(){
        return passext;
    }

    public int getDirecProxyPort() {
        return directproxyport;
    }

    public String mdsTvm2Ticket() {
        return tvm2RenewalTask.ticket(config.tvm2ClientConfig().destinationClientId());
    }

    @Override
    public void start() throws IOException {
        tvm2RenewalTask.start();
        super.start();
    }

    @Override
    public void close() throws IOException {
        tvm2RenewalTask.cancel();
        super.close();
    }
}
