package ru.yandex.search.backpack.client.handlers;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Paths;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Supplier;

import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumer;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumer;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.StringCollectorsFactory;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.msearch.Index;
import ru.yandex.search.backpack.client.BackPackClient;
import ru.yandex.search.backpack.client.BackPackRequestContext;
import ru.yandex.search.backpack.client.handlers.callbacks.BackpackMetaInfoCallback;
import ru.yandex.search.backpack.client.handlers.callbacks.MetaServerCallback;


public class BackpackClientMainHandler implements HttpAsyncRequestHandler<JsonObject> {
    protected String backupversion;
    protected String service;
    protected String shard;
    protected Index index;
    protected final BackPackClient backpack;
    public static final String STATUSFINISHED = "FINISHED";
    public static final String STATUSPENDING = "PENDING";
    public static final String STATUSERROR = "ERROR";
    public static final String STATUSPASSED = "PASSED";
    public static final String OKMESSAGE = "success";
    public static final String DEFAULTKEY = "none";
    public static final String DEFAULTSIZE = "0";
    public static final String DEFAULTMD5 = "0";
    // for testing sandbox purporses only:
    //private static final String OAUTH_TOKEN = "c2FuZGJveC10bXA6YjUyZDVkZjk0ZDA0NTU2MTRiZDZmOWI3NDA3Mzk0OWI=";
    protected String namespace;
    protected String backuppath;
    protected String automove;
    protected ConcurrentHashMap<String, Map<String, String>> versBackupStatus;
    protected ConcurrentHashMap<String, Map<String, String>> versRestoreStatus;

    public final String sizeKey = "size";
    public final String md5Key = "md5";
    public final String statusKey = "status";
    public final String mdsKey = "mdskey";
    public final String msgKey = "msg";
    public final String pathKey = "path";
    public final String inumKey = "inum";
    public final String shardKey = "shard";
    public final String docsKey = "numdocs";
    public final String fsizeKey = "fsize";
    public final String hostnameKey = "hostname";
    public final String rootPathKey = "rootpath";
    private Long docsnum = 0L;


    public BackpackClientMainHandler(BackPackClient backpack) {
          this.backpack = backpack;
          this.namespace = backpack.getNamespace();
          this.versBackupStatus = new ConcurrentHashMap<>();
          this.versRestoreStatus = new ConcurrentHashMap<>();
    }

    public BackpackClientMainHandler(BackPackClient backpack, Index index) {
        this(backpack);
        this.index = index;
    }


    @Override
    public HttpAsyncRequestConsumer<JsonObject> processRequest(
            final HttpRequest request,
            final HttpContext context)
            throws HttpException {
        if (request instanceof HttpEntityEnclosingRequest) {
            HttpEntity entity =
                    ((HttpEntityEnclosingRequest) request).getEntity();
            if (entity.getContentLength() != 0) {
                return new JsonAsyncTypesafeDomConsumer(
                        entity,
                        StringCollectorsFactory.INSTANCE,
                        BasicContainerFactory.INSTANCE);
            }
        }
        return new EmptyAsyncConsumer<>(JsonNull.INSTANCE);
    }

    @Override
    public void handle(
            final JsonObject payload,
            final HttpAsyncExchange exchange,
            final HttpContext context)
            throws HttpException {
        ProxySession session = new BasicProxySession(this.backpack, exchange, context);
        this.handle(new BackPackRequestContext(this.backpack, session), payload);
    }

    public void handle(BackPackRequestContext context, JsonObject payload) throws HttpException {
    }

    public void initBackupStatus(Map.Entry<String, Map<String, String>> fileEntry) {
        Map<String,String> pData = versBackupStatus.computeIfAbsent(fileEntry.getKey(), k -> new LinkedHashMap<>());
        for ( Map.Entry<String,String> fdata : fileEntry.getValue().entrySet()) {
            if (fdata.getKey().equals(sizeKey)
                    && fdata.getValue().equals(DEFAULTSIZE)){
                pData.put(statusKey, STATUSPASSED);
            }

            pData.put(fdata.getKey(),fdata.getValue());
        }
        pData.put(msgKey, DEFAULTKEY);
    }

    public void setBackupStatus(String path, String mdskeyval, String statusval, String message) {
            Map<String, String> datapath = versBackupStatus.get(path);
            datapath.put(mdsKey, mdskeyval);
            datapath.put(statusKey, statusval);
            datapath.put(msgKey, message);
    }

//    public void setRestoreStatus(String path, String mdskeyval, String statusval, String message) {
//        Map<String, String> datapath = versRestoreStatus.get(path);
//        datapath.put(mdsKey, mdskeyval);
//        datapath.put(statusKey, statusval);
//        datapath.put(msgKey, message);
//    }

    protected void mdsFileUpload(Map<String, Map<String, String>> filemap,
                                 AsyncClient client,
                                 BackPackClient backpack,
                                 BackPackRequestContext context,
                                 Long docsnum
    ) throws IOException {
        this.docsnum = docsnum;
        mdsFileUpload(filemap, client, backpack, context);
    }

    protected void mdsFileUpload(Map<String, Map<String, String>> filemap,
                                 AsyncClient client,
                                 BackPackClient backpack,
                                 BackPackRequestContext context
    ) throws IOException {

        context.session().logger().info("Prepare to upload files.");

        StringBuilderWriter mainSbw = new StringBuilderWriter();
        String inum = String.valueOf(backpack.getClientinum());
        String hostname = backpack.getClienthostname();

        try (JsonWriter writer = JsonType.NORMAL.create(mainSbw)) {
            writer.startObject();
            writer.key("meta");
            writer.startObject();
            writer.key("service");
            writer.value(service);
            writer.key("version");
            writer.value(backupversion);
            writer.key(shardKey);
            writer.value(shard);
            writer.key(inumKey);
            writer.value(inum);
            writer.key(hostnameKey);
            writer.value(hostname);
            writer.endObject();

            writer.key("fileinfo");
            writer.startObject();

            for (Map.Entry<String, Map<String, String>> mapKey : filemap.entrySet()) {

                String path = mapKey.getKey();
                writer.key(path);
                writer.startObject();
                writer.key(rootPathKey);
                writer.value(mapKey.getValue().get(rootPathKey));
                writer.key(msgKey);
                writer.value(OKMESSAGE);
                writer.key(mdsKey);
                writer.value(DEFAULTKEY);
                writer.key(statusKey);

                if(mapKey.getValue().get(sizeKey).equals(DEFAULTSIZE)){
                    context.session().logger().warning("Do not backup zero size file, passing: "
                            + mapKey.getValue().get(path));
                    writer.value(STATUSPASSED);
                } else if (backpack.getPassExt().matcher(Paths.get(path).getFileName().toString()).matches()) {
                    context.session().logger().warning("Do not backup file matches the regexp pattern, passing: "
                            + mapKey.getValue().get(path));
                    writer.value(STATUSPASSED);
                }
                else {
                    writer.value(STATUSPENDING);
                }
                writer.key(md5Key);
                writer.value(mapKey.getValue().get(md5Key));
                writer.key(fsizeKey);
                writer.value(mapKey.getValue().get(sizeKey));
                writer.key(shardKey);
                writer.value(shard);
                writer.key(docsKey);
                writer.value(this.docsnum);
                writer.key(inumKey);
                writer.value(inum);
                writer.key(hostnameKey);
                writer.value(hostname);
                writer.endObject();
            }

            writer.endObject();
            writer.endObject();
        }

        Supplier<? extends HttpClientContext> contextGenerator =
                context.session().listener().createContextGeneratorFor(client);

        MetaServerCallback callback;
        callback = new MetaServerCallback(filemap, context, this);

        BasicAsyncRequestProducerGenerator updateReqGenerator =
                new BasicAsyncRequestProducerGenerator(
                        "/v1/backup/init",
                        mainSbw.toString(),
                        StandardCharsets.UTF_8);

        client.execute(backpack.config().metaServerConfig().host(),
                updateReqGenerator,
                contextGenerator,
                callback
        );


    }

    public static void setBackupStat(BackPackRequestContext context,
                                     BackpackClientMainHandler handler,
                                     String path,
                                     String mdskeyval,
                                     String size,
                                     String md5hash,
                                     String status,
                                     String message
    ){
        final AsyncClient client = context.backpack().getMetaServerClient().adjust(context.session().context());

        PrefixedLogger logger = context.session().logger();
        BackPackClient backpack = context.backpack();

        handler.setBackupStatus(path,
                mdskeyval,
                status,
                message);

        StringBuilderWriter mainSbw = new StringBuilderWriter();

        try {
            try (JsonWriter writer = JsonType.NORMAL.create(mainSbw)) {
                writer.startObject();
                writer.key("path");
                writer.value(path);
                writer.key("msg");
                writer.value(message);
                writer.key("mdskey");
                writer.value(mdskeyval);
                writer.key("status");
                writer.value(status);
                writer.key("md5");
                writer.value(md5hash);
                writer.key("fsize");
                writer.value(size);
                writer.key("shard");
                writer.value(handler.getShard());
                writer.key("inum");
                writer.value(backpack.getClientinum());
                writer.key("hostname");
                writer.value(backpack.getClienthostname());
                writer.key("version");
                writer.value(handler.getBackupversion());
                writer.key("service");
                writer.value(handler.getService());
                writer.endObject();
            }
        } catch (IOException e) {
            String msg = "Cannot make json for upload stat to meta server " + e.getMessage();
            logger.severe(msg);
            e.printStackTrace();
        }

        BackpackMetaInfoCallback callback = new BackpackMetaInfoCallback(context, handler, path);

        client.execute(backpack.config().metaServerConfig().host(),
                new BasicAsyncRequestProducerGenerator(
                        "/v1/backup/file/update",
                        mainSbw.toString(),
                        StandardCharsets.UTF_8),
                context.session().listener().createContextGeneratorFor(client),
                callback
                //EmptyFutureCallback.INSTANCE
        );

    }

    public static void setRestoreStat(BackPackRequestContext context,
                                      BackpackClientMainHandler handler,
                                      String path,
                                      String mdskeyval,
                                      String size,
                                      String md5hash,
                                      String rootPath,
                                      String status,
                                      String message
    ){
        final AsyncClient client = context.backpack().getMetaServerClient().adjust(context.session().context());

        PrefixedLogger logger = context.session().logger();
        BackPackClient backpack = context.backpack();

//        handler.setRestoreStatus(path,
//                mdskeyval,
//                status,
//                message);

        StringBuilderWriter mainSbw = new StringBuilderWriter();

        try {
            try (JsonWriter writer = JsonType.NORMAL.create(mainSbw)) {
                writer.startObject();
                writer.key("path");
                writer.value(path);
                writer.key("msg");
                writer.value(message);
                writer.key("mdskey");
                writer.value(mdskeyval);
                writer.key("status");
                writer.value(status);
                writer.key("md5");
                writer.value(md5hash);
                writer.key("rootpath");
                writer.value(rootPath);
                writer.key("fsize");
                writer.value(size);
                writer.key("shard");
                writer.value(handler.getShard());
                writer.key("inum");
                writer.value(backpack.getClientinum());
                writer.key("hostname");
                writer.value(backpack.getClienthostname());
                writer.key("version");
                writer.value(handler.getBackupversion());
                writer.key("service");
                writer.value(handler.getService());
                writer.endObject();
            }
        } catch (IOException e) {
            String msg = "Cannot make json for upload stat to meta server " + e.getMessage();
            logger.severe(msg);
            e.printStackTrace();
        }

        BackpackMetaInfoCallback callback = new BackpackMetaInfoCallback(context, handler, path);

//        logger.info("Try to make info req for path: " + path + " status: " + status);

        client.execute(backpack.config().metaServerConfig().host(),
                new BasicAsyncRequestProducerGenerator(
                        "/v1/restore/file/update",
                        mainSbw.toString(),
                        StandardCharsets.UTF_8),
                context.session().listener().createContextGeneratorFor(client),
                callback
        );
    }

    public String getShard() {
        return shard;
    }

    public String getBackupversion() {
        return backupversion;
    }

    public String getService() {
        return service;
    }

    public String getPath() {
        return backuppath;
    }

    public String getAutoMove() {
        return automove;
    }
}
