package ru.yandex.search.backpack.client.handlers.callbacks;

import java.io.IOException;
import java.net.URLEncoder;
import java.util.function.Supplier;

import org.apache.http.HttpHost;
import org.apache.http.HttpResponse;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.util.EntityUtils;

import ru.yandex.http.util.nio.BasicAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.search.backpack.client.BackPackClient;
import ru.yandex.search.backpack.client.BackPackRequestContext;
import ru.yandex.search.backpack.client.handlers.BackpackClientMainHandler;

import static ru.yandex.search.backpack.client.handlers.BackpackClientMainHandler.DEFAULTMD5;
//import ru.yandex.search.backpack.client.handlers.callbacks.ZeroCopySupplier;

public final class MdsGetHostCallback
        implements FutureCallback<HttpResponse> {
    private final String path;
    private final String spath;
    private final String md5Hash;
    private final BackPackRequestContext context;
    private final BackPackClient backpack;
    private final PrefixedLogger logger;
    private final BackpackClientMainHandler handler;

    public MdsGetHostCallback(String path,
                              String md5Hash,
                              BackPackRequestContext context,
                              String backupversion,
                              BackpackClientMainHandler backpackHandler) {
        // TODO: turn saulted path in config
        this.path = path;
        this.spath = backupversion + ":" + path;
        this.md5Hash = md5Hash;
        this.context = context;
        this.backpack = context.backpack();
        this.logger = context.session().logger();
        this.handler = backpackHandler;
    }

    @Override
    public void completed(final HttpResponse response) {
        final int code = response.getStatusLine().getStatusCode();
        try {
            if (code >= 400 || code < 200) {
                context.session().logger().severe("Cannot get upload server for: "
                        + path + " Invalid http response code: "
                        + code);
                if (response.getEntity() != null) {
                    final String reply =
                            EntityUtils.toString(response.getEntity());
                    if (reply.length() != 0) {
                        context.session().logger().severe("Path: "
                                + path
                                + " Server reply: " + reply);
                    }
                }
            } else {
                HttpHost host = new HttpHost(EntityUtils.toString(response.getEntity())
                        + ":"
                        + backpack.getDirecProxyPort());
                String filekey;

                if ( backpack.getToSalt() ) {
                    filekey = URLEncoder.encode(spath, "UTF-8");
                    context.session().logger().info("Key sault setting to true, key will be: " + filekey);
                } else {
                    filekey = URLEncoder.encode(path, "UTF-8");
                    context.session().logger().info("Key sault setting to false, key will be: " + filekey);
                }

                context.session().logger().info("Host for upload:: " + host);

                String uri = host
                        + "/upload-"
                        + backpack.getNamespace()
                        + "/"
                        + filekey
                        + "?expire="
                        + backpack.getExpire();

                //TODO: Looks like we dont need ZeroCopyPost and we can use simple
                //TODO: BasicAsyncRequestProducer here (and write bandwith ratelimiter for it)

                ZeroCopySupplier zeroCopySupp = new ZeroCopySupplier(uri, path, md5Hash, context, handler);

                // Check that supplier can reach file
                if (!zeroCopySupp.getError()) {

                    final AsyncClient client =
                            backpack.getMdsWriterClient().adjust(context.session().context());

                    Supplier<? extends HttpClientContext> contextGenerator =
                            context.session().listener().createContextGeneratorFor(client);

                    client.execute(zeroCopySupp,
                            BasicAsyncResponseConsumerFactory.INSTANCE,
                            contextGenerator,
                            new MdsUploadCallback(context, md5Hash, path, handler));
                }
            }
        } catch (IOException e) {
            logger.severe("Cannot get upload host for: " + path + " error: "
                    + e.getMessage());

            BackpackClientMainHandler.setBackupStat(context,
                    handler,
                    path,
                    "none",
                    BackpackClientMainHandler.DEFAULTSIZE,
                    DEFAULTMD5,
                    BackpackClientMainHandler.STATUSERROR,
                    "Exception reached: " + e.getMessage());

            e.printStackTrace();
        }
    }

    @Override
    public void failed(final Exception e) {
        logger.severe("Cannot determine upload host for: " + path + " error: "
                + e.getMessage());

        BackpackClientMainHandler.setBackupStat(context,
                handler,
                path,
                "none",
                BackpackClientMainHandler.DEFAULTSIZE,
                DEFAULTMD5,
                BackpackClientMainHandler.STATUSERROR,
                "Cannot determine upload host for: " + path + " error: " + e.getMessage());
    }

    @Override
    public void cancelled() {
    }
}
