package ru.yandex.search.backpack.client.handlers.callbacks;

import java.nio.file.Paths;
import java.util.Map;
import java.util.function.Supplier;

import org.apache.http.HttpResponse;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.search.backpack.client.BackPackClient;
import ru.yandex.search.backpack.client.BackPackRequestContext;
import ru.yandex.search.backpack.client.handlers.BackpackClientMainHandler;

public final class MetaServerCallback
        implements FutureCallback<HttpResponse> {
    private final BackPackRequestContext context;
    private final PrefixedLogger logger;
    private final BackpackClientMainHandler handler;

    private final Map<String, Map<String, String>> filemap;
    private final String backupversion;
    private final BackPackClient backpack;

    public MetaServerCallback(Map<String, Map<String, String>> filemap,
                              BackPackRequestContext context,
                              BackpackClientMainHandler backpackHandler) {
        this.logger = context.session().logger();
        this.handler = backpackHandler;
        this.context = context;
        this.filemap = filemap;
        this.backupversion = backpackHandler.getBackupversion();
        this.backpack = context.backpack();
    }

    @Override
    public void completed(final HttpResponse response) {

        for (Map.Entry<String, Map<String, String>> mapKey : filemap.entrySet()) {
            String path = mapKey.getKey();
            String md5Hash = mapKey.getValue().get(handler.md5Key);
            String fileSize = mapKey.getValue().get(handler.sizeKey);
            String rootPath = mapKey.getValue().get(handler.rootPathKey);

            if (fileSize.equals(BackpackClientMainHandler.DEFAULTSIZE)){
                logger.info("File size is zero, do not backup file: " + path );
                continue;
            } else if (backpack.getPassExt().matcher(Paths.get(path).getFileName().toString()).matches()) {
                logger.info("During pass ext regexp passing backup for: " + path );
                continue;
            }

            if (backpack.getNoHashExt().matcher(Paths.get(path).getFileName().toString()).matches()) {
                logger.info("During no hash regexp doesn't set md5 hash for: " + path );
                md5Hash = "";
            }

            logger.info("Start callback to upload: "
                    + path
                    + " Md5sum: "
                    + md5Hash
                    + " File size: "
                    + fileSize
                    + " rootPath: "
                    + rootPath);

            final AsyncClient client =
                    backpack.getMdsWriterClient().adjust(context.session().context());

            Supplier<? extends HttpClientContext> contextGenerator =
                    context.session().listener().createContextGeneratorFor(client);

            MdsGetHostCallback callback;

            callback = new MdsGetHostCallback(path, md5Hash, context, backupversion, handler);

            BasicAsyncRequestProducerGenerator producerGenerator =
                    new BasicAsyncRequestProducerGenerator("/hostname");

            producerGenerator.addHeader(
                    YandexHeaders.X_YA_SERVICE_TICKET,
                    backpack.mdsTvm2Ticket());

            client.execute(backpack.config().mdsWriterConfig().host(),
                    producerGenerator,
                    contextGenerator,
                    callback
            );
        }

    }

    @Override
    public void failed(final Exception e) {
        logger.severe("Cannot upload meta info and init backup: " + " error: "
                + e.getMessage());
    }

    @Override
    public void cancelled() {
    }
}
