package ru.yandex.search.backpack.client.handlers.providers;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URLEncoder;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.security.DigestInputStream;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;

import ru.yandex.base64.Base64Encoder;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.search.backpack.client.BackPackRequestContext;
import ru.yandex.search.backpack.client.handlers.BackpackClientMainHandler;

public class FilePathBackupProvider {
    private final Map<String, Map<String, String>> filemap;
    private final Path rootPath;
    private final PrefixedLogger logger;
    private final BackpackClientMainHandler handler;

    public FilePathBackupProvider(final String path, BackPackRequestContext context, final BackpackClientMainHandler handler) {
        this.logger = context.session().logger();
        this.rootPath = Paths.get(path).getParent();
        this.handler = handler;
        filemap = new HashMap<>();
        logger.info("Start walking around upload dir.");

        try {
            Files.walkFileTree(
                    Paths.get(path),
                    new FilePathBackupProvider.ListDirAdvisor(this));
        } catch (IOException e) {
            logger.severe("Error access to file" + e.getMessage());
        }
    }

    public Map<String, Map<String, String>> getFilemap() {
        return filemap;
    }

    private static class ListDirAdvisor extends SimpleFileVisitor<Path> {
        private final FilePathBackupProvider provider;

        ListDirAdvisor(FilePathBackupProvider provider) {
            this.provider = provider;
        }

        @Override
        public FileVisitResult visitFile(
                final Path path,
                final BasicFileAttributes attrs)
                throws IOException {
            String size = String.valueOf(path.toFile().length());
            //String filekey = URLEncoder.encode(path.toString(), "UTF-8");
            String filepath = path.toString();
            String md5Hash = computeContentMD5Header(new FileInputStream(new File(filepath)));

            Map<String, String> storedFile = provider.filemap.get(filepath);

            if (storedFile == null) {
                LinkedHashMap<String, String> map = new LinkedHashMap<String, String>();
                map.put(provider.handler.sizeKey, size);
                map.put(provider.handler.md5Key, md5Hash);
                map.put(provider.handler.rootPathKey, provider.rootPath.toString());
                provider.filemap.put(filepath, map);

                provider.logger.info("Stored file path: "
                        + filepath
                        + " File key: "
                        + URLEncoder.encode(filepath, "UTF-8")
                        + " Md5 base64 hash: "
                        + md5Hash
                        + " Root path: "
                        + provider.rootPath.toString());
            } else {
                provider.logger.warning("Collision happened! "
                        + filepath
                        + " File name: "
                        + path.getFileName()
                        + " File key: "
                        + URLEncoder.encode(filepath, "UTF-8")
                        + " Md5 base64 hash: "
                        + md5Hash
                        + " Md5 base64 found:"
                        + storedFile.get(provider.handler.md5Key)
                        + " Root path: "
                        + provider.rootPath.toString()
                );
            }

            return FileVisitResult.CONTINUE;
        }
    }

    public static String computeContentMD5Header(InputStream inputStream) {
        // Consume the stream to compute the MD5 as a side effect.
        DigestInputStream s;
        try {
            s = new DigestInputStream(inputStream,
                    MessageDigest.getInstance("MD5"));
            // drain the buffer, as the digest is computed as a side-effect
            byte[] buffer = new byte[8192];
            if (s.read(buffer) > 0) {
                while(true) {
                    if (s.read(buffer) < 0) {
                        break;
                    }
                }
            }
            Base64Encoder encoder = new Base64Encoder();
            encoder.process(s.getMessageDigest().digest());
            return encoder.toString();

        } catch (NoSuchAlgorithmException e) {
            throw new RuntimeException(e);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }
}
