package ru.yandex.search.so.bbnn_plus;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.TimerTask;
import java.util.logging.Logger;

import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;

import ru.yandex.collection.Pattern;
import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.http.proxy.HttpProxy;
import ru.yandex.http.util.HttpStatusPredicates;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.StatusCheckAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.http.util.server.UpstreamStater;
import ru.yandex.http.util.server.UpstreamStaterFutureCallback;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.Utf8JsonWriter;
import ru.yandex.logger.HandlersManager;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.stater.CountAggregatorFactory;
import ru.yandex.stater.DuplexStaterFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;
import ru.yandex.stater.Stater;
import ru.yandex.stater.StatsConsumer;

public class BBNNPlus extends HttpProxy<ImmutableBBNNPlusConfig> {
    private static final int PUSH_DELAY = 10000;

    private final TimeFrameQueue<Long> recordsProcessed;
    private final TimeFrameQueue<Long> uuidsDeposited;
    private final RecordHandler handler;
    private final AsyncClient supClient;
    private final Logger depositLogger;
    private final PrefixedLogger depositDebugLogger;
    private final UpstreamStater supStater;
    private final Table allowedUuids;

    public BBNNPlus(final ImmutableBBNNPlusConfig config)
        throws ConfigException, IOException
    {
        super(config);

        allowedUuids = new Table(config.tablesPath(), "allowed-uuids.txt");

        supClient = client("Sup", config.sup());

        final HandlersManager handlersManager = new HandlersManager();
        depositLogger = config.depositLog().build(handlersManager);
        depositDebugLogger = config.depositDebugLog().buildPrefixed(handlersManager);
        registerLoggerForLogrotate(depositLogger);
        registerLoggerForLogrotate(depositDebugLogger);


        handler = new RecordHandler(
            this,
            logger().replacePrefix("HANDLER"),
            config.tablesPath());

        register(
            new Pattern<>("/logbroke-it", true),
            new LogbrokeItHandler(this, handler),
            RequestHandlerMapper.POST);

        register(
            new Pattern<>("/send-promo-push", true),
            new SendPushHandler(this),
            RequestHandlerMapper.GET);

        recordsProcessed = new TimeFrameQueue<>(config.metricsTimeFrame());
        uuidsDeposited = new TimeFrameQueue<>(config.metricsTimeFrame());

        registerStater(
            new PassiveStaterAdapter<>(
                recordsProcessed,
                new DuplexStaterFactory<>(
                    new NamedStatsAggregatorFactory<>(
                        "records_processed_ammm",
                        CountAggregatorFactory.INSTANCE),
                    new NamedStatsAggregatorFactory<>(
                        "records_processed_axxx",
                        CountAggregatorFactory.INSTANCE))));
        registerStater(
            new PassiveStaterAdapter<>(
                uuidsDeposited,
                new DuplexStaterFactory<>(
                    new NamedStatsAggregatorFactory<>(
                        "uuids_deposited_ammm",
                        CountAggregatorFactory.INSTANCE),
                    new NamedStatsAggregatorFactory<>(
                        "uuids_deposited_axxx",
                        CountAggregatorFactory.INSTANCE))));
        registerStater(new Huyater());
        supStater =
            new UpstreamStater(config.metricsTimeFrame(), "sup");
        registerStater(supStater);
    }

    public void allowUuid(final String uuid) {
        allowedUuids.add(uuid);
    }

    public void uuidDeposited() {
        uuidsDeposited.accept(1L);
    }

    public boolean uuidAllowed(final String uuid) {
        return allowedUuids.contains(uuid);
    }

    @Override
    public void close() throws IOException {
        super.close();
        supClient.close();
        handler.close();
        allowedUuids.close();
    }

    public AsyncClient sup() {
        return supClient;
    }

    public Logger depositLogger() {
        return depositLogger;
    }

    public PrefixedLogger depositDebugLogger() {
        return depositDebugLogger;
    }

    public void processRecord() {
        recordsProcessed.accept(1L);
    }

/*
{
    "receiver" :
    [
        "uuid: '$UUID'"
    ],
    "notification" :
    {
        "title": "'$TITLE'",
        "body": "'$BODY'",
        "link": "https://id.yandex.ru/promo"
    },
    "project" : "zalogin",
    "data":
    {
        "push_id" : "passport_zalogin_pp_push"
    },
    "is_data_only": true
}'
*/
    public final void schedulePush(
        final UserContext ctx,
        final FutureCallback<JsonObject> callback,
        final boolean promo)
    {
        supClient.scheduleRetry(
            new PushDelayTask(ctx, callback, promo),
            PUSH_DELAY);
    }

    public final void sendPush(
        final UserContext ctx,
        final FutureCallback<JsonObject> callback,
        final boolean promo)
        throws IOException
    {
        AsyncClient client = supClient;
        final String uri = "/pushes?dry_run=0";
        final byte[] postData;
        final ByteArrayOutputStream baos = new ByteArrayOutputStream();
        baos.reset();
        try (
            Utf8JsonWriter writer = JsonType.NORMAL.create(baos))
        {
            writer.startObject();

            writer.key("receiver");
            writer.startArray();
            writer.value("uuid: " + ctx.uuid());
            writer.endArray();

            if (promo) {
                writer.key("adjust_time_zone");
                writer.value(true);
            }

            writer.key("notification");
            writer.startObject();
            writer.key("title");
            if (promo) {
                writer.value(config().push1Title());
            } else {
                writer.value(config().push2Title());
            }
            writer.key("body");
            if (promo) {
                writer.value(config().push1Body());
            } else {
                writer.value(config().push2Body());
            }
            writer.key("link");
            if (promo) {
                writer.value(config().push1Link());
            } else {
                writer.value(config().push2Link());
            }
            writer.endObject();

            writer.key("project");
            writer.value("zalogin");

            writer.key("data");
            writer.startObject();
            writer.key("push_id");
            writer.value("passport_zalogin_pp_push");
            writer.endObject();

            writer.key("is_data_only");
            writer.value(true);

            writer.endObject();

            writer.flush();
            postData = baos.toByteArray();
        }
        final BasicAsyncRequestProducerGenerator post =
            new BasicAsyncRequestProducerGenerator(
                uri,
                postData,
                ContentType.APPLICATION_JSON);
        post.addHeader("Authorization", "OAuth " + config().supOAuthToken());
        ctx.userLog().info("postData: " + new String(postData, "UTF-8"));
        client.execute(
            config().sup().host(),
            post,
            new StatusCheckAsyncResponseConsumerFactory<JsonObject>(
                HttpStatusPredicates.NON_PROTO_FATAL,
                JsonAsyncTypesafeDomConsumerFactory.INSTANCE),
//            session.listener().createContextGeneratorFor(client),
            new UpstreamStaterFutureCallback<>(
                callback,
                supStater));
    }

    private final class PushDelayTask extends TimerTask {
        private final UserContext ctx;
        private final FutureCallback<JsonObject> callback;
        private final boolean promo;

        private PushDelayTask(
            final UserContext ctx,
            final FutureCallback<JsonObject> callback,
            final boolean promo)
        {
            this.ctx = ctx;
            this.callback = callback;
            this.promo = promo;
        }

        @Override
        public void run() {
            try {
                ctx.userLog().info("Sending second push");
                sendPush(ctx, callback, promo);
            } catch (IOException e) {
                callback.failed(e);
            }
        }
    }

    private class Huyater implements Stater {
        @Override
        public <E extends Exception> void stats(
            final StatsConsumer<? extends E> statsConsumer)
            throws E
        {
            statsConsumer.stat(
                "promo-pushes-sent_axxx",
                allowedUuids.size());
            statsConsumer.stat(
                "pluses-enlarged_axxx",
                handler.depositedCount());
        }
    }

}

