package ru.yandex.search.so.bbnn_plus;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import org.apache.http.Header;
import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.protocol.HTTP;
import org.apache.http.util.EntityUtils;
import org.apache.james.mime4j.stream.EntityState;
import org.apache.james.mime4j.stream.MimeTokenStream;
//import org.apache.http.concurrent.FutureCallback;

import ru.yandex.compress.GzipInputStream;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.NotImplementedException;
import ru.yandex.mail.mime.DefaultMimeConfig;
import ru.yandex.mail.mime.OverwritingBodyDescriptorBuilder;
import ru.yandex.mail.mime.Utf8FieldBuilder;
import ru.yandex.tskv.BasicTskvParser;
//import ru.yandex.http.util.BadRequestException;
//import ru.yandex.http.util.nio.BasicAsyncResponseProducerGenerator;
//import ru.yandex.logger.PrefixedLogger;

public class LogbrokeItHandler implements ProxyRequestHandler
{
    private static final int TEMP_BUFFER_SIZE = 4096;
    private static final ThreadLocal<BasicTskvParser> TSKV_PARSER =
        new ThreadLocal<>();
    private final BBNNPlus bbnn;
    private final ThreadPoolExecutor executor;
    private final RecordHandler handler;

    public LogbrokeItHandler(
        final BBNNPlus bbnn,
        final RecordHandler handler)
    {
        this.bbnn = bbnn;
        this.handler = handler;
        ImmutableBBNNPlusConfig config = bbnn.config();
        executor =
            new ThreadPoolExecutor(
                config.parseThreads(),
                config.parseThreads(),
                1,
                TimeUnit.HOURS,
                new ArrayBlockingQueue<>(config.parseQueue()));
    }

    public BBNNPlus bbnn() {
        return bbnn;
    }

    public BasicTskvParser parser() {
        BasicTskvParser parser = TSKV_PARSER.get();
        if (parser == null) {
            parser = new BasicTskvParser(handler);
            TSKV_PARSER.set(parser);
        }
        return parser;
    }

    @Override
    public void handle(final ProxySession session)
        throws HttpException, IOException
    {
        HttpEntityEnclosingRequest request =
            (HttpEntityEnclosingRequest) session.request();
        byte[] body = EntityUtils.toByteArray(request.getEntity());
        Header encoding = request.getFirstHeader("Content-Encoding");
        if (encoding != null && "gzip".equals(encoding.getValue())) {
            body = gzipDecompress(body);
        }

        final String contentTypeHeader =
            session.headers().getString(HTTP.CONTENT_TYPE, null);
        ContentType contentType = null;
        if (contentTypeHeader != null) {
            try {
                contentType = ContentType.parse(contentTypeHeader);
            } catch (Exception e) {
                throw new BadRequestException("Bad content-type header");
            }
        }
        final boolean multiPart;
        if (contentType != null
            && contentType.getMimeType().equals("multipart/mixed"))
        {
            multiPart = true;
        } else {
            multiPart = false;
        }
        List<byte[]> bodies;
        if (multiPart) {
            bodies = parseMultipart(session, body);
            session.logger().info("Multipart bodies: " + bodies.size());
        } else {
            bodies = Collections.singletonList(body);
        }

        executor.execute(
            new TskvBufferParser(
                bodies,
                new Callback(session)));
    }

    private static byte[] gzipDecompress(byte[] data) throws IOException {
        try (ByteArrayInputStream bis = new ByteArrayInputStream(data);
            ByteArrayOutputStream bos =
                new ByteArrayOutputStream(data.length << 1);
            GzipInputStream gzipIS = new GzipInputStream(bis))
        {
            byte[] buffer = new byte[4096];
            int len;
            while ((len = gzipIS.read(buffer)) != -1) {
                bos.write(buffer, 0, len);
            }
            return bos.toByteArray();
        }
    }

    private List<byte[]> parseMultipart(
        final ProxySession session,
        final byte[] data)
        throws HttpException
    {
        List<byte[]> parts = new ArrayList<>();
        MimeTokenStream stream = new MimeTokenStream(
            DefaultMimeConfig.INSTANCE,
            null,
            new Utf8FieldBuilder(),
            new OverwritingBodyDescriptorBuilder());
        byte[] tempBuffer = new byte[TEMP_BUFFER_SIZE];
        try {
            HttpEntity entity =
                ((HttpEntityEnclosingRequest) session.request()).getEntity();
            stream.parseHeadless(
                new ByteArrayInputStream(data),
                entity.getContentType().getValue());
            EntityState state = stream.getState();
            while (state != EntityState.T_END_OF_STREAM) {
                switch (state) {
                    case T_BODY:
                        int pos = 0;
                        try (InputStream is = stream.getDecodedInputStream())
                                //stream.getBodyDescriptor().getCharset()))
                        {
                            int read =
                                is.read(
                                    tempBuffer,
                                    pos,
                                    tempBuffer.length - pos);
                            while (read != -1) {
                                pos += read;
                                if (pos == tempBuffer.length) {
                                    tempBuffer = Arrays.copyOf(
                                        tempBuffer,
                                        tempBuffer.length << 1);
                                }
                                read = is.read(
                                    tempBuffer,
                                    pos,
                                    tempBuffer.length - pos);
                            }
                        }
                        byte[] body = Arrays.copyOf(tempBuffer, pos);
                        parts.add(body);
                        break;
                    default:
                        break;
                }
                state = stream.next();
            }
        } catch (Throwable t) {
            throw new NotImplementedException(t);
        }
        return parts;
    }


    private static class Callback extends AbstractProxySessionCallback<Void> {
        Callback(final ProxySession session) {
            super(session);
        }

        @Override
        public void completed(final Void v) {
            session.response(HttpStatus.SC_OK);
        }
    }

    private class TskvBufferParser implements Runnable {
        private final List<byte[]> bodies;
        private final FutureCallback<Void> callback;

        TskvBufferParser(
            final List<byte[]> bodies,
            final FutureCallback<Void> callback)
        {
            this.bodies = bodies;
            this.callback = callback;
        }

        @Override
        public void run() {
            BasicTskvParser parser = parser();
            try {
                for (byte[] body: bodies) {
                    String str = new String(body, StandardCharsets.UTF_8);
                    parser.parse(new StringReader(str));
                }
                callback.completed(null);
            } catch (Exception e) {
                callback.failed(e);
            }
        }
    }
}
