package ru.yandex.ps.disk.search;

import java.io.IOException;
import java.util.AbstractMap;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.DoubleFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyNHttpEntity;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.proxy.universal.BasicUniversalSearchProxyRequestContext;
import ru.yandex.util.string.UnhexStrings;

public class CompareFacesHandler implements ProxyRequestHandler {
    private final Diface server;

    public CompareFacesHandler(final Diface server) {
        this.server = server;
    }

    @Override
    public void handle(
        final ProxySession session)
        throws HttpException, IOException
    {
        Callback callback = new Callback(session);
        List<String> vectors = session.params().getAll("vector");
        if (vectors.size() > 0) {
            session.response(
                HttpStatus.SC_OK,
                String.valueOf(
                    Face.dotProduct(
                        Face.bytesToDoubleArray(UnhexStrings.unhex(vectors.get(0))),
                        Face.bytesToDoubleArray(UnhexStrings.unhex(vectors.get(1))))));
            return;
        }

        List<String> faces = session.params().getAll("face_id");
        if (faces.size() > 0) {
            long uid = session.params().getLong("prefix");
            QueryConstructor qc = new QueryConstructor("/search?faces&IO_PRIO=1000");
            try {
                StringBuilder textSb = new StringBuilder();
                textSb.append(FaceBackendFields.FACE_ID.prefixed());
                textSb.append(":(");
                textSb.append(faces.get(0).replaceAll(":", "\\\\:"));
                textSb.append(" ");
                textSb.append(faces.get(1).replaceAll(":", "\\\\:"));
                textSb.append(")");
                qc.append("text", textSb.toString());
                qc.append("service", server.config().faceIndexQueue());
                qc.append("prefix", uid);
                qc.append("get", "*");
            } catch (BadRequestException bre) {
                callback.failed(bre);
                return;
            }

            session.params().putIfAbsent(
                "service",
                Collections.singletonList(server.config().faceIndexQueue()));

            BasicUniversalSearchProxyRequestContext context =
                new BasicUniversalSearchProxyRequestContext(
                    server,
                    session,
                    EmptyNHttpEntity.INSTANCE,
                    false);
            server.sequentialRequest(
                session,
                new BasicUniversalSearchProxyRequestContext(
                    server,
                    session,
                    EmptyNHttpEntity.INSTANCE,
                    false),
                new BasicAsyncRequestProducerGenerator(qc.toString()),
                TimeUnit.SECONDS.toMillis(2),
                false,
                JsonAsyncTypesafeDomConsumerFactory.OK,
                session.listener().createContextGeneratorFor(context.client()),
                new FromIndexCallback(callback));
        } else {
            List<String> stids = session.params().getAll("stid");
            if (stids.size() != 2) {
                callback.failed(new BadRequestException("Expecting 2 stids"));
                return;
            }

            DoubleFutureCallback<List<Face>, List<Face>> dfcb =
                new DoubleFutureCallback<>(new ExtractCallback(callback));

            Context context = new Context(server, session);
            server.extractFaces(context, stids.get(0), new DiskDoc(stids.get(0), stids.get(0), stids.get(0), 1, 1), dfcb.first());
            server.extractFaces(context, stids.get(1), new DiskDoc(stids.get(1), stids.get(1), stids.get(1), 1, 1), dfcb.second());
        }
    }

    private static final class Callback
        extends AbstractProxySessionCallback<Map.Entry<Face, Face>>
    {
        public Callback(final ProxySession session) {
            super(session);
        }

        @Override
        public void completed(final Map.Entry<Face, Face> entry) {
            session.response(
                HttpStatus.SC_OK,
                String.valueOf(
                    Face.dotProduct(
                        entry.getKey().dvector(),
                        entry.getValue().dvector())));
        }
    }

    private static final class FromIndexCallback
        extends AbstractFilterFutureCallback<JsonObject, Map.Entry<Face, Face>>
    {
        public FromIndexCallback(final FutureCallback<? super Map.Entry<Face, Face>> callback) {
            super(callback);
        }

        @Override
        public void completed(final JsonObject resultObj) {
            try {
                JsonList list = resultObj.asMap().getList("hitsArray");
                if (list.size() != 2) {
                    failed(new BadRequestException(
                        "Some faecs not found: " + JsonType.NORMAL.toString(resultObj)));
                    return;
                }

                callback.completed(
                    new AbstractMap.SimpleEntry<>(
                        Face.parseFromBackend(list.get(0)),
                        Face.parseFromBackend(list.get(1))));
            } catch (JsonException je) {
                failed(je);
            }
        }
    }


    private static final class ExtractCallback
        extends AbstractFilterFutureCallback<Map.Entry<List<Face>, List<Face>>, Map.Entry<Face, Face>>
    {
        public ExtractCallback(final FutureCallback<? super Map.Entry<Face, Face>> callback) {
            super(callback);
        }

        @Override
        public void completed(final Map.Entry<List<Face>, List<Face>> entry) {
            if (entry.getKey().size() <= 0) {
                failed(new Exception("First resource - no faces"));
                return;
            }

            if (entry.getValue().size() <= 0) {
                failed(new Exception("Second resource - no faces"));
                return;
            }

            callback.completed(
                new AbstractMap.SimpleEntry<>(
                    entry.getKey().get(0),
                    entry.getValue().get(0)));
        }
    }

    private static class Context extends AbstractDifaceContext {
        public Context(
            final Diface server,
            final ProxySession session)
            throws BadRequestException
        {
            super(server, session);
        }
    }
}
