package ru.yandex.ps.disk.search;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

import ru.yandex.http.proxy.ProxySession;

public class DbScanClusterizer implements Clusterizer {
    private final String baseId;
    private final long version;
    private final double minClusterSize;
    private final DpThreshold similarThreshold;
    private final List<DpThreshold> thresholds;
    private final ProxySession session;
    private final DifaceContext context;
    private final String cachetype;
    private final DpThreshold[] neigbourThresholds;

    private static final double[] AGE = new double[200];
    static {
        setAgeRange(0, 2, 1);
        setAgeRange(3, 4, 1.5);
        setAgeRange(5, 6, 2);
        setAgeRange(7, 9, 3);
        setAgeRange(10, 11, 3.5);
        setAgeRange(12, 15, 4);
        setAgeRange(16, 18, 4.5);
        setAgeRange(19, 22, 5);
        setAgeRange(23, 30, 6);
        setAgeRange(31, 37, 7);
        setAgeRange(38, 50, 8);
        setAgeRange(50, 199, 10);
    }

    private static void setAgeRange(final int start, final int end, final double value) {
        for (int i = start; i <= end; i++) {
            AGE[i] = value;
        }
    }

    /**
     SearchServer-96
     SearchServer-77
     SearchServer-94
     SearchServer-98
     SearchServer-95
     SearchServer-71
     SearchServer-93
     SearchServer-78
     SearchServer-67
     SearchServer-80
    **/

    public DbScanClusterizer(
        final DifaceContext context,
        final String baseId,
        final long version)
    {
        this.baseId = baseId;
        this.version = version;
        this.session = context.session();
        DpThreshold neighbourTh =
            new DpThreshold(context.clusterThreshold(), "Neighbour", 0);

        this.similarThreshold = new DpThreshold(0.8, "Similar", 1);
        AgeThreshold toddlerAgeTh =
            new AgeThreshold(
                0.93,
                Integer.parseInt(session.params().getString("toddler-age", "4")),
                "ToddlerNeighbour",
                2);
        AgeThreshold childTh =
            new AgeThreshold(
                0.9,
                Integer.parseInt(session.params().getString("child-age", "12")),
                "ChildNeighbour",
                3);
        this.minClusterSize = context.minClusterSize();

        this.cachetype = session.params().getString("cachetype", "bit");

        this.context = context;
        this.neigbourThresholds = new DpThreshold[] {toddlerAgeTh, childTh, neighbourTh};
        StringBuilder sb = new StringBuilder("Thresholds: ");
        for (int i = 0; i < neigbourThresholds.length; i++) {
            sb.append(neigbourThresholds[i]);
            sb.append(", ");
        }
        context.logger().info(sb.toString());
        this.thresholds = Arrays.asList(neighbourTh, similarThreshold, toddlerAgeTh, childTh);
    }

    public DbScanClusterizer(
        final DifaceContext context,
        final String baseId,
        final long version,
        final List<DpThreshold> neigbourThs,
        final DpThreshold similarThreshold,
        final String cachetype)
    {
        this.baseId = baseId;
        this.version = version;
        this.minClusterSize = context.minClusterSize();

        this.similarThreshold = similarThreshold;
        this.session = context.session();

        this.context = context;
        this.cachetype = cachetype;
        this.thresholds = new ArrayList<>(neigbourThs);
        this.thresholds.add(similarThreshold);
        this.neigbourThresholds = new DpThreshold[neigbourThs.size()];
        StringBuilder sb = new StringBuilder("Thresholds: ");
        neigbourThs.toArray(neigbourThresholds);

        for (int i = 0; i < neigbourThresholds.length; i++) {
            sb.append(neigbourThresholds[i]);
            sb.append(", ");
        }
        context.logger().info(sb.toString());
    }


    @Override
    public List<Cluster> clusterize(
        final List<Cluster> clusters,
        final Collection<Face> faces)
    {
        DpCache cache;
        if (cachetype.equalsIgnoreCase("array")) {
            cache = new ArrayCache(faces.size());
        } else {
            cache = new BitCache(faces.size(), thresholds);
        }

        List<Cluster> result = clusterize(clusters, faces, cache);
        context.logger().info(
            "Cache hit " + cache.cacheHit()
                + " total " + cache.total()
                + " miss " + cache.cacheMiss());

        if (cache.total() > 0) {
            context.logger().info(
                "Cache hit " + 100 * cache.cacheHit() / cache.total()
                    + " miss " + 100 * cache.cacheMiss() / cache.total());
        }

        context.logger().info("Clusters made " + clusters.size());
        return result;
    }

    public static int countSides(final Collection<Face> faces) {
        int count = 0;
        for (Face face: faces) {
            if (face.side()) {
                count += 1;
            }
        }

        return count;
    }

    private List<Cluster> clusterizeInternal(
        final List<Cluster> clusters,
        final Collection<Face> faces,
        final DpCache cache,
        final List<Cluster> diff,
        final AtomicInteger i)
    {
        int processed = 0;
        int progressPrint = 0;
        if (faces.size() > 10000) {
            progressPrint = faces.size() / 10;
        }
        if (faces.size() > 50000) {
            progressPrint = faces.size() / 100;
        }

        if (faces.size() > 80000) {
            progressPrint = faces.size() / 1000;
        }
        for (final Face face: faces) {
            if (progressPrint > 0 && processed % progressPrint == 0 ) {
                context.logger().info("Processed " + processed + "/" + faces.size() + " Cache hit " + cache.cacheHit()
                    + " total " + cache.total()
                    + " miss " + cache.cacheMiss());
            }
            processed += 1;
            if (face.status() != null) {
                continue;
            }

            final Set<Face> neighbors = getNeighbors(null, face, faces, cache);
            int sideFaces = countSides(neighbors);

            if (neighbors.size() - sideFaces >= minClusterSize) {
                final Cluster cluster = new Cluster(baseId + '_' + i.incrementAndGet(), face.faceId(), version);

                if (context.debug()) {
                    session.logger().info(
                        "New cluster " + cluster.id()
                            + " sides " + sideFaces
                            + " Neighbors size " + neighbors.size());
                }
                long ts = System.currentTimeMillis();
                Cluster newCluster = expandCluster1(cluster, face, neighbors, faces, cache);
                clusters.add(newCluster);
                if (!cluster.names().isEmpty()) {
                    session.logger().info("Cluster " + cluster.id()
                        + " Name " + face.name() + " Names " + cluster.namesAsString());
                }
                diff.add(newCluster);
                context.logger().info("Cluster expanded in " + (System.currentTimeMillis() - ts));
                context.logger().info(
                    "Cluster " + newCluster.id() + " faces "
                        + newCluster.faces().size() + " expanded in " + (System.currentTimeMillis() - ts));

                context.logger().info(
                    "Cluster " + newCluster.id() + " name " + cluster.name() + " names " + cluster.names());
                context.logger().info(
                    "Cluster " + newCluster.id() + " faces "
                        + cluster.faces().stream().map((f) -> f.name() + "," + f.faceId()).collect(Collectors.joining(";")));

                ts = System.currentTimeMillis();
                //final List<Face> similars = getSimilars(cluster, face, faces, similarThreshold, cache);
                //expandSimilars(cluster, similars, faces, similarThreshold, cache);
                //context.logger().info("Similarity check took " + (System.currentTimeMillis() - ts));
            } else {
                face.status(PointStatus.NOISE);
            }
        }

        return diff;
    }

    public List<Cluster> clusterize(
        final List<Cluster> clusters,
        final Collection<Face> faces,
        final DpCache cache)
    {
        List<Cluster> diff = new ArrayList<>();
        AtomicInteger i = new AtomicInteger(0);
        int k = 0;
        for (Face face: faces) {
            face.index = k++;
        }

        Map<String, Double> th = new LinkedHashMap<>();
        th.put("Эрик", 0.75);
        th.put("Рубен", 0.75);
        th.put("Арам", 0.75);
        th.put("Гарри Худавердян", 0.9);
        Map<String, String> map = new LinkedHashMap<>();
        map.put("10761:4ed74329398e86dac438ae6daa6427c182131ec94382268705a5641bc1f9ae2f_0", "Эрик");
        map.put("10761:7b559dfa01af367e11784f1f7e88bfa3d6eaa82aa2430149fb5774c31a126f54_0", "Эрик");
        map.put("10761:9fdc41276ef1cd60f2e49bd972b3fab9772559bf723235dee13b1c6f8809df4c_0", "Эрик");
        map.put("10761:5925c79aaa9f1dcd5d830cab305d22915140fcd832ec8c16b04617bd021fa89f_0", "Эрик");
        map.put("10761:da2c10345b79dd5d95af975e8ce9454285a68a72a5b07c9dfb549ed949b5b907_0", "Эрик");
        map.put("10761:07f30b348e50a7a49af9fdee5f22848f74f59599e8d6d0352ab921ff71fa8da6_0", "Арам");
        map.put("10761:62dc2dec9ec802fbe1f7063f791e7bb01ea98549b944c96ca9c812cd967c60f6_0", "Арам");
        map.put("10761:a2db1bc2ae1331bf887396836c64c9832972b587340b9b522e2269c5313f2907_0", "Арам");
        map.put("10761:cdb73a1eee15971570eeeeeed75587eb26ba805df76b4d592f44b55530a1bb20_0", "Арам");
        map.put("10761:bd18f234978d12047c3b59f60f65cbfd6ba5d1953c864c5f258a98862a169443_0", "Рубен");
        map.put("10761:bc42a3bda66f25a58ad15f23ff87699ba33322157d52d2763aa32f19d32b452a_0", "Рубен");
        map.put("10761:caf8a45de757ae87765d0bf1c8e3154e0e79a2806f20c38aa12b64a7d65acc9a_0", "Рубен");
        map.put("10761:110ff7adbf29c12ca4c94e3fde770cc51f3a1cea69a7f373860364b9afbe315f_0", "Рубен");
        map.put("10761:3b706685f84830f2d939d08ca0018e4c408c5137540e9a75b0812c45b1375ae9_0", "Рубен");
        map.put("10761:61fde405f3987873a39c5101e636d06b3ec082a4eb270c0ebda846d653ef2ec4_0", "Рубен");
        map.put("10761:7fa908dc6638d345379de9d1c5cefde0185e71772f0fea07101341d84284de36_0", "Рубен");
        map.put("10761:901f416cc564d4240ae114299853f1aa9a2b6dd41e71d545285ae1423f5066d8_0", "Рубен");
        map.put("10761:0fca4f3f57ebd0d5cbdd5be4bbfcffee43c733c71998c6faec2baec0ad0d0cd2_0", "Гарри Худавердян");
        map.put("10761:ee769b4f9f5a0a2011d4850ed673dca4c3ca4c593fb23ae7a849a3e6cc71e1f4_0", "Гарри Худавердян");
        map.put("10761:7236155705e52185098ae03095ccc8aa580b9dae6765fe074d6f2345b7ae828b_0", "Гарри Худавердян");
        map.put("10761:eee6955a4fc63ff3248106ad53372b9a78d9705c19a839d2d319a4fe2e2ec760_0", "Гарри Худавердян");
        map.put("10761:acfc4babfa8330a48cb24d91027c62be5eaa71c42bfd1a5344d1a20a9ea07c64_0", "Гарри Худавердян");


        List<Face> facesWithNames = new ArrayList<>();
        for (Face face: faces) {
            if (face.name() != null && !face.name().isEmpty()) {
                facesWithNames.add(face);
            }
        }

        Map<String, List<Face>> hardcode = new LinkedHashMap<>();
        for (Face face: faces) {
            String name = map.get(face.faceId());
            if (name != null) {
                hardcode.computeIfAbsent(name, x -> new ArrayList<>()).add(face);
            }
        }

        for (String name: th.keySet()) {
            List<Face> thFaces = hardcode.get(name);
            Face face1 = thFaces.get(0);
            final Cluster cluster =
                new Cluster(baseId + '_' + i.incrementAndGet(), face1.faceId(), version);
            cluster.addFace(face1);
            clusters.add(cluster);
            diff.add(cluster);

            for (Face face2: facesWithNames) {
                for (Face face: thFaces) {
                    if (face2 == face || face2.cluster() != null || map.containsKey(face2.faceId())) {
                        continue;
                    }

                    if (face2.name().equalsIgnoreCase(face.name()) && face.dotProduct(face2) > th.get(face.name())) {
                        cluster.addFace(face2);
                        break;
                    }
                }
            }

            context.logger().info(
                "Prepared, Cluster " + cluster.id() + " faces "
                    + cluster.faces().size() + " " + cluster.names());
        }
        if (clusters.size() == 0) {
            throw new RuntimeException("Null hardcoded clusters");
        }
        for (Cluster cluster: clusters) {
            expandCluster1(cluster, cluster.faces().get(0), cluster.faces(), faces, cache);

            context.logger().info(
                "Cluster " + cluster.id() + " faces "
                    + cluster.faces().size() + " " + cluster.names());
        }

        //diff = clusterizeInternal(clusters, facesWithNames, cache, diff, i);
        diff = clusterizeInternal(clusters, faces, cache, diff, i);

        return diff;
    }

    private void expandSimilars(
        final Cluster cluster,
        final List<Face> neighbors,
        final Collection<Face> faces,
        final DpThreshold threshold,
        final DpCache cache)
    {
        List<Face> seeds = new ArrayList<Face>(neighbors);
        int index = 0;
        HashSet<Face> dedupSet = new HashSet(seeds);
        while (index < seeds.size()) {
            final Face current = seeds.get(index);
            final List<Face> currentSimilars =
                getSimilars(cluster, current, faces, threshold, cache);
            if (currentSimilars.size() >= minClusterSize) {
                seeds = merge(seeds, dedupSet, currentSimilars);
            }

            if (current.cluster() != cluster) {
                current.similarCluster(cluster);
            }

            index++;
        }

        context.logger().info("Similars check cnt " + index);
    }

    private Cluster expandCluster1(
        final Cluster cluster,
        final Face face,
        final Collection<Face> neighbors,
        final Collection<Face> faces,
        final DpCache cache)
    {
        cluster.addFace(face);
        face.status(PointStatus.PART_OF_CLUSTER);
        List<Face> seeds = new ArrayList<Face>(neighbors);
        int index = 0;
        HashSet<Face> dedupSet = new HashSet(seeds);
        while (index < seeds.size()) {
            final Face current = seeds.get(index);
            PointStatus pStatus = current.status();
            if (pStatus == null) {
                final Collection<Face> currentNeighbors =
                    getNeighbors(cluster, current, faces, cache);
                if (currentNeighbors.size() >= minClusterSize) {
                    seeds = merge(seeds, dedupSet, currentNeighbors);
                }
            }

            // current.resourceId().equalsIgnoreCase(face.resourceId()) two faces from same resource
            // can not be in the same cluster (mirror problem)
            if (pStatus != PointStatus.PART_OF_CLUSTER) {
                current.status(PointStatus.PART_OF_CLUSTER);
                if (context.debug()) {
                    session.logger().info("Adding on expand " + cluster.id() + " " + current.faceId());
                }

                cluster.addFace(current);
            } else if (context.debug()) {
                session.logger().info("Neighbour skipped " + current.status() + " " + current.resourceId() + " " + current.faceId());
            }
            index++;
        }

        return cluster;
    }

    private List<Face> getSimilars(
        final Cluster cluster,
        final Face face,
        final Collection<Face> faces,
        final DpThreshold threshold,
        final DpCache cache)
    {

        List<Face> simlars = new ArrayList<>();
        for (final Face neighbor: faces) {
            if (face == neighbor) {
                continue;
            }

            if (face.cluster() != null && face.cluster() != cluster) {
                continue;
            }

            if (cache.neigbour(face, neighbor, threshold)) {
                //if ((children && dp > 0.9) || (!children && dp > threshold)) {
                simlars.add(neighbor);
            }
        }

        return simlars;
    }

    private void printIfNamesConincide(final Face face, final Face neighbor) {
        if (face.dotProduct(neighbor) >= 0.7) {
            return;
        }

        StringBuilder sb = new StringBuilder();
        sb.append("NameCoincide, but th check bad");
        sb.append(face.name());
        sb.append(" ");
        sb.append(face.faceId());
        sb.append(" ");
        sb.append(neighbor.faceId());
        sb.append(" ");

        for (DpThreshold th: neigbourThresholds) {
            sb.append(th.name());
            sb.append(" exp ");
            sb.append(th.threshold());
            sb.append(" val ");
            sb.append(face.dotProduct(neighbor));
            sb.append(" ; ");
        }

        session.logger().info(sb.toString());
    }

    private boolean checkNeighbour(final Face face, final Face neighbor, final DpCache cache) {
        boolean nameCoincide = face.name() != null && face.name().equalsIgnoreCase(neighbor.name());
        for (DpThreshold th: neigbourThresholds) {
            ThresholdVerdict verdict = th.apply(face, neighbor, cache);
            if (verdict == ThresholdVerdict.CONTINUE) {
                continue;
            }

            if (verdict == ThresholdVerdict.NO) {
                if (nameCoincide){
                    //printIfNamesConincide(face, neighbor);
                }
                return false;
            }

            return true;
        }

//        if (nameCoincide){
//            printIfNamesConincide(face, neighbor);
//        }

        return false;
        //double dp = cache.dp(face, neighbor);
//        boolean child = neighbor.age() <= childAgeLimit || face.age() <= childAgeLimit;
//        boolean adolescant = neighbor.age() <= adolescantLimit || face.age() <= adolescantLimit;
//        //if (neighbor.dotProduct(face) > threshold(face, neighbor)) {
//
//        if (child) {
//            return cache.neigbour(face, neighbor, toddlerAgeTh);
//        } else if (adolescant) {
//            return cache.neigbour(face, neighbor, adolescantThreshold);
//        } else {
//            return cache.neigbour(face, neighbor, neighbourThreshold);
//        }
        //if (cache.neigbour(face, neighbor, neighbourThreshold))
    }

    public List<Cluster> expand(
        final List<Face> newFaces,
        final Collection<Face> faces)
    {
        DpCache cache;
        if (cachetype.equalsIgnoreCase("array")) {
            cache = new ArrayCache(faces.size());
        } else {
            cache = new BitCache(faces.size(), thresholds);
        }

        int index = 0;
        for (Face face: faces) {
            face.index = index++;
        }

        List<Cluster> result = new ArrayList<>();

        for (Face face: newFaces) {
            context.logger().info("Processing face " + face.faceId());
            int maxNeighborsInOneCluster = 0;
            Cluster maxCluster = null;
            List<Face> nonClusterNeighbors = new ArrayList<>();
            Map<Cluster, List<Face>> clusteredNeighbors = new LinkedHashMap<>();

            if (face.cluster() != null) {
                continue;
            }

            Map<String, Integer> maxMap = new LinkedHashMap<>();

            int neighbours = 0;
            // TODO should we pick up first cluster that matches?
            for (final Face neighbor: faces) {
                if (face == neighbor) {
                    continue;
                }

                if (checkNeighbour(face, neighbor, cache)) {
                    neighbours += 1;
                    if (context.debug()) {
                        context.logger().info("Face " + neighbor.faceId() + " is neighbor");
                    }

                    if (neighbor.cluster() != null) {
                        Set<Face> neigborNeigbors =
                            getNeighbors(neighbor.cluster(), neighbor, faces, cache);
                        if (neigborNeigbors.size() >= minClusterSize) {
                            List<Face> clusterFaces =
                                clusteredNeighbors.computeIfAbsent(
                                    neighbor.cluster(),
                                    (p) -> new ArrayList<>());

                            clusterFaces.add(neighbor);
                            if (clusterFaces.size() > maxNeighborsInOneCluster) {
                                maxNeighborsInOneCluster = clusterFaces.size();
                                maxCluster = neighbor.cluster();
                            }
                        }

                        int value = maxMap.getOrDefault(neighbor.cluster().id(), 0);
                        if (neigborNeigbors.size() > value) {
                            maxMap.put(neighbor.cluster().id(), value);
                        }
                    } else {
                        if (!neighbor.deleted()) {
                            nonClusterNeighbors.add(neighbor);
                        }
                    }
                }
            }

            context.logger().info("MaxMap " + maxMap);
            if (maxCluster != null) {
                context.logger().info(
                    "Face matched clusters " + maxMap.size()
                        + " max is " + maxCluster.id()
                        + " with neighbours in this cluster " + maxNeighborsInOneCluster);
                maxCluster.addFace(face);
            } else if (nonClusterNeighbors.size() - countSides(nonClusterNeighbors) >= minClusterSize) {
                // TODO check for similars and etc
                final Cluster cluster = new Cluster(baseId + '_' + result.size(), face.faceId(), version);
                expandCluster1(cluster, face, nonClusterNeighbors, faces, cache);
                result.add(cluster);
                context.logger().info(
                    "New cluster created " + cluster.id() + " size " + cluster.faces().size());
            } else {
                // now trying start from neigbours https://st.yandex-team.ru/PS-3660
                Cluster cluster = null;
                if (nonClusterNeighbors.size() - countSides(nonClusterNeighbors) >= minClusterSize) {
                    nonClusterNeighbors.add(face);

                    for (Face neighbour: nonClusterNeighbors) {
                        Set<Face> neigbours = getNeighbors(null, neighbour, nonClusterNeighbors, cache);
                        if (neigbours.size() - countSides(neigbours) >= minClusterSize) {
                            cluster = new Cluster(baseId + '_' + result.size(), neighbour.faceId(), version);
                            expandCluster1(cluster, neighbour, nonClusterNeighbors, faces, cache);
                            result.add(cluster);
                            context.logger().info(
                                "New cluster created on other neighbor "
                                    + cluster.id() + " size " + cluster.faces().size());
                            break;
                        }
                    }
                }

                context.logger().info(
                    "No clusters for face " + face.faceId() + " free neighbours: " + nonClusterNeighbors.size()
                        + " cluster neighbours " + clusteredNeighbors.size());
            }

            context.logger().info(face.faceId() + " have neighbours size: " + neighbours);
        }

        context.logger().info(
            "Cache hit " + cache.cacheHit()
                + " total " + cache.total()
                + " miss " + cache.cacheMiss());

        if (cache.total() > 0) {
            context.logger().info(
                "Cache hit " + 100 * cache.cacheHit() / cache.total()
                    + " miss " + 100 * cache.cacheMiss() / cache.total());
        }

        return result;
    }

    private Set<Face> getNeighbors(
        final Cluster cluster,
        final Face face,
        final Collection<Face> faces,
        final DpCache cache)
    {
        Set<Face> neighbors = new LinkedHashSet<>();
        for (final Face neighbor: faces) {
            if (face == neighbor) {
                continue;
            }

            if (neighbor.status() == PointStatus.PART_OF_CLUSTER
                && neighbor.cluster() != cluster)
            {
                continue;
            }

//            //double dp = cache.dp(face, neighbor);
//            boolean toddler = neighbor.age() <= 2 || face.age() <= 2;
//            boolean child = neighbor.age() <= 5 || face.age() <= 5;
//            //if (neighbor.dotProduct(face) > threshold(face, neighbor)) {

            //if (cache.neigbour(face, neighbor, neighbourThreshold))
            boolean leftHasName = face.name() != null && !face.name().isEmpty();
            boolean rightHasName = neighbor.name() != null && !neighbor.name().isEmpty();
            boolean haveNamesAndNotEquals =
                leftHasName && rightHasName && !face.name().equalsIgnoreCase(neighbor.name());
            if (cluster != null && rightHasName) {
                String clusterName = cluster.name();
                if (clusterName != null) {
                    haveNamesAndNotEquals |= !clusterName.equalsIgnoreCase(neighbor.name());
                }
            }
//            boolean nameCoincide = face.name() != null && face.name().equalsIgnoreCase(neighbor.name());
//            if (nameCoincide && ) {
//
//            }
//            if (nameCoincide) {
//                if (face.dotProduct(neighbor) >= 0.65) {
//                    neighbors.add(neighbor);
//                    checkNeighbour(face, neighbor, cache);
//                } else {
//                    checkNeighbour(face, neighbor, cache);
//                }
//            } else
            if (!haveNamesAndNotEquals && checkNeighbour(face, neighbor, cache)) {
                neighbors.add(neighbor);
            }
        }

        return neighbors;
    }

    private static List<Face> merge(
        final List<Face> one,
        final Set<Face> oneSet,
        final Collection<Face> two)
    {
        for (Face item : two) {
            if (!oneSet.contains(item)) {
                one.add(item);
                oneSet.add(item);
            }
        }
        return one;
    }

    private class ArrayCache implements DpCache {
        private final float[][] cache;
        private final int size;
        protected long cacheHit = 0;
        protected long total = 0;

        public ArrayCache(final int size) {
            this.size = Math.min(size, 20000);
            this.cache = new float[this.size][this.size];
            for (int i = 0; i < this.size; i++) {
                Arrays.fill(cache[i], -1);
            }
        }

        @Override
        public boolean neigbour(final Face f1, final Face f2, final DpThreshold threshold) {
            total += 1;
            if (f1.index >= size || f2.index >= size) {
                return f1.dotProduct(f2) >= threshold.threshold();
            }

            float dp = cache[f1.index][f2.index];
            if (dp < 0) {
                dp = cache[f1.index][f2.index];
            }

            if (dp < 0) {
                dp = (float) f1.dotProduct(f2);
                cache[f1.index][f2.index] = dp;
                cache[f2.index][f1.index] = dp;
            } else {
                cacheHit += 1;
            }

            if (context.debug()) {
                context.logger().info("Face " + f1.faceId() + " with " + f2.faceId() + " dp " + dp + " th " + threshold.threshold());
            }

            return dp >= threshold.threshold();
        }

        @Override
        public long cacheHit() {
            return cacheHit;
        }

        @Override
        public long cacheMiss() {
            return total - cacheHit;
        }

        @Override
        public long total() {
            return total;
        }
    }

    private class BitCache implements DpCache {
        private final int maxSize ;
        private final long[][] arrayCache;
        private final long[] cached;
        private final DpThreshold[] thresholds;
        protected long cacheHit = 0;
        protected long total = 0;
        protected int cacheSize;

        public BitCache(final long size, final List<DpThreshold> thresholds) {
            this.maxSize = (1024 * 1024 * 1024 / (thresholds.size() + 1));
            long requestCacheSize = 1 + size * size / 2;
            long cacheSize = 1 + requestCacheSize / 64;
            if (cacheSize > maxSize) {
                cacheSize = maxSize;
            }
            this.cacheSize = (int) cacheSize;

            this.cached = new long[this.cacheSize];
            this.arrayCache = new long[thresholds.size()][];
            this.thresholds = new DpThreshold[thresholds.size()];

            for (int i = 0; i < thresholds.size(); i++) {
                arrayCache[thresholds.get(i).index()] = new long[this.cacheSize];
                this.thresholds[i] = thresholds.get(i);
            }

            context.logger().info("Cache size real " + cacheSize * 64 + " requested " + requestCacheSize);
            // for 2 thresholds for 100k elements cache is 3.5 gb
        }

        private boolean get(final long[] bits, final long index) {
            int i = (int)(index >> 6);             // div 64
            if (i>=bits.length) {
                return false;
            }
            int bit = (int)index & 0x3f;           // mod 64
            long bitmask = 1L << bit;
            return (bits[i] & bitmask) != 0;
        }

        private boolean set(final long[] bits, final long index) {
            int wordNum = (int)(index >> 6);
            if (wordNum >= bits.length) {
                return false;
            }
            int bit = (int)index & 0x3f;
            long bitmask = 1L << bit;
            bits[wordNum] |= bitmask;
            return true;
        }

        @Override
        public boolean neigbour(final Face f1, final Face f2, final DpThreshold threshold) {
            if (f1.index == f2.index) {
                return true;
            }

            total += 1;
            long index;
            if (f1.index > f2.index) {
                index = (long) f1.index * (f1.index - 1) / 2 + f2.index;
            } else {
                index = (long) f2.index * (f2.index - 1) / 2 + f1.index;
            }

            if (context.debug()) {
                context.logger().info("f1 " + f1.index + " f2 " + f2.index + " finalindex " + index + ";");
            }

//            if (lindex >= maxSize) {
//                //context.logger().warning("Cache overflow size " + this.cacheSize + " index " + lindex);
//                return f1.dotProduct(f2) >= threshold.threshold();
//            }

            if (get(cached, index)) {
                cacheHit += 1;
                return get(arrayCache[threshold.index()], index);
            }


            double dp = f1.dotProduct(f2);
            boolean result = dp >= threshold.threshold();
            if (!set(cached, index)) {
                return result;
            }

            for (int i = 0; i < thresholds.length; i++) {
                if (dp >= thresholds[i].threshold()) {
                    set(arrayCache[thresholds[i].index()], index);
                }
            }

            return result;
        }

        @Override
        public long cacheHit() {
            return cacheHit;
        }

        @Override
        public long cacheMiss() {
            return total - cacheHit;
        }

        @Override
        public long total() {
            return total;
        }
    }

}
