package ru.yandex.ps.disk.search;

import java.io.IOException;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.collection.Pattern;
import ru.yandex.concurrent.TimeFrameQueue;
import ru.yandex.disk.search.AvatarSrwKeyConverter;
import ru.yandex.disk.search.DiskParams;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.FilterFutureCallback;
import ru.yandex.http.util.HttpStatusPredicates;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumer;
import ru.yandex.http.util.nio.StatusCheckAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumer;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.StringCollectorsFactory;
import ru.yandex.ocr.proxy.CvStat;
import ru.yandex.ocr.proxy.CvStater;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.ps.disk.search.config.ImmutableDifaceConfig;
import ru.yandex.ps.disk.search.reindex.ReindexHandler;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.proxy.universal.UniversalSearchProxy;
import ru.yandex.search.proxy.universal.UniversalSearchProxyRequestContext;
import ru.yandex.util.string.StringUtils;

public class Diface
    extends UniversalSearchProxy<ImmutableDifaceConfig>
    implements HttpAsyncRequestHandler<JsonObject>
{
    private final AsyncClient imageparserClient;
    private final AsyncClient cokemulatorClient;
    private final AsyncClient djfsClient;
    private final FaceCache faceCache;
    private final TimeFrameQueue<CvStat> faceParserStats;
    private final List<Face> sideFaces;
    private final List<Face> bannedFaces;
    //private final ExifInfoExtractor exifExtractor;

    public Diface(final ImmutableDifaceConfig config) throws IOException {
        super(config);

        this.djfsClient = client("DjfsClient", config.djfsConfig());
        this.imageparserClient = client("ImageparserClient", config.imageparserConfig());
        this.cokemulatorClient = client("CokemulatorClient", config.cokemulatorConfig());
        faceParserStats = new TimeFrameQueue<>(config.metricsTimeFrame());
        registerStater(new CvStater(faceParserStats));

        this.sideFaces = FaceExclusionsLoader.loadSides(logger(), config.sideConfig());
        this.bannedFaces = FaceExclusionsLoader.loadSides(logger(), config.bannedFacesConfig());
        this.faceCache = new FaceCache(this);
        //this.exifExtractor = new ExifInfoExtractor(this);
        this.register(new Pattern<>("/reindex", true), new ReindexHandler(this));
        this.register(
            new Pattern<>("/face", true),
            new FaceInHandler(this));
        this.register(
            new Pattern<>("/compare", true),
            new CompareFacesHandler(this));
        this.register(
            new Pattern<>("/extract", true),
            new ExtractOneHandler(this));
    }

    public AsyncClient imageparserClient() {
        return imageparserClient;
    }

    public AsyncClient cokemulatorClient() {
        return cokemulatorClient;
    }

    public List<Face> sideFaces() {
        return sideFaces;
    }

    public AsyncClient djfsClient() {
        return djfsClient;
    }

    public void stat(final CvStat stat) {
        faceParserStats.accept(stat);
    }
    @Override
    public void handle(
        final JsonObject payload,
        final HttpAsyncExchange exchange,
        final HttpContext httpContext)
        throws HttpException
    {
        ProxySession session = new BasicProxySession(this, exchange, httpContext);
        DifaceRequestContext context =
            new DifaceRequestContext(this, session);
        StringBuilder sb = new StringBuilder("");
        sb.append("id:");
        sb.append(context.id());
        sb.append("+OR+type:face+OR+type:cluster");
        QueryConstructor qc = new QueryConstructor("/search-diface?");
        qc.append("prefix", context.prefix());
        qc.append("service", context.user().service());
        qc.append("text", sb.toString());
        qc.append("get", "*");

        sequentialRequest(
            session,
            context,
            new BasicAsyncRequestProducerGenerator(qc.toString()),
            1000L,
            false,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            session.listener().createContextGeneratorFor(context.searchClient()),
            new DiskBackendDataCallback(context));
    }

    @Override
    public HttpAsyncRequestConsumer<JsonObject> processRequest(
        final HttpRequest request,
        final HttpContext context)
        throws HttpException
    {
        if (request instanceof HttpEntityEnclosingRequest) {
            HttpEntity entity =
                ((HttpEntityEnclosingRequest) request).getEntity();
            if (entity.getContentLength() != 0) {
                return new JsonAsyncTypesafeDomConsumer(
                    entity,
                    StringCollectorsFactory.INSTANCE,
                    BasicContainerFactory.INSTANCE);
            }
        }
        return new EmptyAsyncConsumer<>(JsonNull.INSTANCE);
    }


    public void getIndexedFacesAndClusters(
        final LongPrefix prefix,
        final ProxySession session,
        final FutureCallback<UserFacesAndClusters> cb)
    {
        UserFacesAndClusters result = faceCache.get(prefix.prefix());
        if (result != null) {
            session.logger().info("From cache " + result);
            cb.completed(result);
            return;
        }

        GetIndexedFacesContext context = new GetIndexedFacesContext(session, prefix);
        QueryConstructor qc = new QueryConstructor("/search?faces&IO_PRIO=1000");
        try {
            qc.append("text", "type:(face face_cluster)");
            qc.append("service", config.faceIndexQueue());
            qc.append("prefix", prefix.toStringFast());
            qc.append("get", "*");
        } catch (BadRequestException bre) {
            cb.failed(bre);
            return;
        }

        this.sequentialRequest(
            session,
            context,
            new BasicAsyncRequestProducerGenerator(qc.toString()),
            TimeUnit.MINUTES.toMillis(2),
            false,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            session.listener().createContextGeneratorFor(context.client()),
            new IndexedFacesSearchCallback(session, prefix.prefix(), cb));
    }

    public void updateCache(
        final Long prefix,
        final UserFacesAndClusters data)
    {
        faceCache.put(prefix, data);
    }

    private class IndexedFacesSearchCallback
        extends AbstractFilterFutureCallback<JsonObject, UserFacesAndClusters>
    {
        private final Long prefix;
        private final ProxySession session;

        public IndexedFacesSearchCallback(
            final ProxySession session,
            final Long prefix,
            final FutureCallback<? super UserFacesAndClusters> callback)
        {
            super(callback);

            this.session = session;
            this.prefix = prefix;
        }

        @Override
        public void completed(final JsonObject resultObj) {
            try {
                JsonList list = resultObj.asMap().getList("hitsArray");

                //session.logger().info("Current faces " + JsonType.NORMAL.toString(resultObj));
                Map<String, Cluster> clusterMap = new LinkedHashMap<>(list.size());
                Map<String, String> faceClusterMap = new LinkedHashMap<>(list.size());

                List<Face> faces = new ArrayList<>(list.size());

                for (JsonObject itemObj: list) {
                    JsonMap item = itemObj.asMap();
                    String typeStr = item.getString("type");
                    if ("face".equalsIgnoreCase(typeStr)) {
                        Face face = Face.parseFromBackend(item);
                        String clusterId =
                            item.getString(FaceBackendFields.FACE_CLUSTER_ID.stored(), null);
                        if (clusterId != null) {
                            faceClusterMap.put(face.faceId(), clusterId);
                        }

                        faces.add(face);
                    } else if ("face_cluster".equalsIgnoreCase(typeStr)) {
                        Cluster cluster = Cluster.parseFromBackend(item);
                        clusterMap.put(cluster.id(), cluster);
                    }
                }

                for (Face face: faces) {
                    String clusterId = faceClusterMap.get(face.faceId());
                    if (clusterId != null) {
                        Cluster cluster = clusterMap.get(clusterId);
                        if (cluster == null) {
                            session.logger().warning("Cluster id in index, but cluster not found " + clusterId);
                            continue;
                        }

                        cluster.addFace(face);
                        face.cluster(cluster);
                        face.status(PointStatus.PART_OF_CLUSTER);
                        face.resetChanged();
                    }
                }

                List<Cluster> clusters = new ArrayList<>(clusterMap.values());
                UserFacesAndClusters cached = new UserFacesAndClusters(faces, clusters);
                updateCache(prefix, cached);

                callback.completed(cached);
            } catch (JsonException je) {
                failed(je);
            }
        }
    }


    private class GetIndexedFacesContext implements UniversalSearchProxyRequestContext {
        private final User user;
        private final ProxySession session;
        private final AsyncClient client;

        public GetIndexedFacesContext(final ProxySession session, final LongPrefix prefix) {
            this.session = session;
            this.user = new User(config.faceIndexQueue(), prefix);
            this.client = searchClient.adjust(session.context());
        }

        @Override
        public User user() {
            return user;
        }

        @Override
        public Long minPos() {
            return null;
        }

        @Override
        public AsyncClient client() {
            return client;
        }

        @Override
        public Logger logger() {
            return session.logger();
        }

        @Override
        public long lagTolerance() {
            return 0L;
        }
    }

    protected BasicAsyncRequestProducerGenerator createRequest(
        final String stid)
        throws BadRequestException
    {
        boolean avaKey = AvatarSrwKeyConverter.INSTANCE.isAvatarStid(stid);

        String unistorageKey;
        String processStid;
        if (avaKey) {
            processStid = AvatarSrwKeyConverter.INSTANCE.parse(stid);
            unistorageKey = StringUtils.concat(
                DiskParams.AVATAR_X_SRW_NAMESPACE.getValue(),
                '/',
                processStid,
                '/',
                "1280_nocrop/webp");
        } else {
            unistorageKey = stid;
            processStid = stid;
        }

        BasicAsyncRequestProducerGenerator generator =
            new BasicAsyncRequestProducerGenerator(
                "/process?extract-faces=true&old-cv=false&fail-on-empty=false&stid="
                    + unistorageKey);
        if (avaKey) {
            generator.addHeader(DiskParams.AVATAR_X_SRW_KEY_TYPE);
            generator.addHeader(DiskParams.AVATAR_X_SRW_NAMESPACE);
        } else {
            generator.addHeader(DiskParams.DISK_X_SRW_KEY_TYPE);
            generator.addHeader(DiskParams.DISK_X_SRW_NAMESPACE);
        }

        generator.addHeader(YandexHeaders.X_SRW_KEY, processStid);

        return generator;
    }

    public Future<List<Face>> extractFaces(
        final DifaceContext context,
        final String stid,
        final DiskDoc doc,
        final FutureCallback<? super List<Face>> callback)
        throws BadRequestException
    {
        ProxySession session = context.session();
        boolean cokemulator = context.cokemulator();

        AsyncClient client = imageparserClient;
        HttpHost host = config.imageparserConfig().host();
        if (cokemulator) {
            client = cokemulatorClient;
            host = config.cokemulatorConfig().host();
        }

        client = client.adjust(session.context());

        if (context.listener()) {
            return client.execute(
                host,
                createRequest(stid),
                new StatusCheckAsyncResponseConsumerFactory<>(
                    HttpStatusPredicates.OK,
                    new FaceConsumerFactory(doc, context.stat())),
                session.listener().createContextGeneratorFor(client),
                callback);
        } else {
            return client.execute(
                host,
                createRequest(stid),
                new StatusCheckAsyncResponseConsumerFactory<>(
                    HttpStatusPredicates.OK,
                    new FaceConsumerFactory(doc, context.stat())),
                callback);
        }
    }

//    public void extractExif(
//        final DifaceContext context,
//        final String stid,
//        final FutureCallback<Map<String, JsonObject>> callback)
//        throws BadRequestException
//    {
//        exifExtractor.extract(context.session(), stid, callback);
//    }

    public Future<List<Face>> extractFaces(
        final AbstractDifaceContext context,
        final DiskDoc diskDoc,
        final FutureCallback<List<Face>> callback)
        throws BadRequestException
    {
        if (diskDoc.previewStid() != null && !diskDoc.stid().equalsIgnoreCase(diskDoc.previewStid())) {
            FutureCallback<List<Face>> previewCallback = new SideFacesFilterCallback(callback, context);
            if (context.tryOriginalOnPreviewFailed()) {
                previewCallback = new OriginalImageFallback(previewCallback, context, diskDoc.stid(), diskDoc);
            }
            return extractFaces(
                context,
                diskDoc.previewStid(),
                diskDoc,
                previewCallback);
        } else {
            return extractFaces(
                context,
                diskDoc.stid(),
                diskDoc,
                new SideFacesFilterCallback(callback, context));
        }
    }

    public boolean bannedFace(final Face face) {
        for (Face side: bannedFaces) {
            double dp = side.dotProduct(face);

            if (dp > 0.75) {
                return true;
            }
        }
        return false;
    }

    public boolean sideFace(final Face face) {
        for (Face side: sideFaces) {
            double dp = side.dotProduct(face);

            if (dp > 0.75) {
                return true;
            }
        }
        return false;
    }

    public Face postprocessFace(final Face face, final AbstractDifaceContext context) {
        if (bannedFace(face)) {
            context.stat().bannedFace(1);
            if (context.debug()) {
                context.logger().info("banned face " + face.faceId());
            }
            return null;
        }

        if (sideFace(face)) {
            context.stat().sideFaces(1);
            face.side(true);
            if (context.debug()) {
                context.logger().info("Filtered side face " + face.faceId());
            }
        }

        if (context.checkDimensions()) {
            // now dimensions fix
            double originalRatio = -1.0;
            if (face.resourceHeight() > 0) {
                originalRatio = ((double) face.resourceWidth()) / face.resourceHeight();
            }

            double exImageRatio = -1.0;
            if (face.imageHeight() > 0) {
                exImageRatio = face.imageWidth() / face.imageHeight();
            }

            double diff = Math.abs(exImageRatio - originalRatio);

            if (originalRatio < 0 ||  diff > context.dimensionsRatioEps()) {
                // try rotate
                if (face.imageWidth() > 0) {
                    exImageRatio = face.imageHeight() / face.imageWidth();

                    if (Math.abs(exImageRatio - originalRatio) <= context.dimensionsRatioEps()) {
                        context.stat().invertedDimensions(1);
                        Face fixedFace = new Face(
                            face.faceId(),
                            face.resourceId(),
                            face.resourceHeight(),
                            face.resourceWidth(),
                            face.imageWidth(),
                            face.imageHeight(),
                            face.x(),
                            face.y(),
                            face.w(),
                            face.h(),
                            face.confidence(),
                            face.vector(),
                            face.age(),
                            face.femaleProb());
                        context.logger().warning("Inverted dimensions for " + face.resourceId());
                        return fixedFace;
                    }
                }

                context.logger().warning(
                    "Broken dimensions, skipping " + face.resourceId()
                        + " Original ratio " + originalRatio + " extract ratio " + exImageRatio + " " + diff);
                context.stat().brokenDimensions(1);
                return null;
            }
        }

        return face;
    }

    private class SideFacesFilterCallback
        extends FilterFutureCallback<List<Face>>
    {
        private final AbstractDifaceContext context;

        public SideFacesFilterCallback(
            final FutureCallback<? super List<Face>> callback,
            final AbstractDifaceContext context)
        {
            super(callback);
            this.context = context;
        }

        @Override
        public void completed(final List<Face> result) {
            List<Face> keep = new ArrayList<>(result.size());
            for (Face face: result) {
                Face filteredFace = postprocessFace(face, context);
                if (filteredFace != null) {
                    keep.add(filteredFace);
                }
            }

            callback.completed(keep);
        }
    }

    private class OriginalImageFallback
        extends FilterFutureCallback<List<Face>>
    {
        private final String stid;
        private final DiskDoc doc;
        private final AbstractDifaceContext context;

        public OriginalImageFallback(
            final FutureCallback<? super List<Face>> callback,
            final AbstractDifaceContext context,
            final String stid,
            final DiskDoc doc)
        {
            super(callback);
            this.stid = stid;
            this.doc = doc;
            this.context = context;
        }

        @Override
        public void failed(final Exception e) {
            if (context.debug()) {
                logger.log(Level.WARNING, "Preview stid failed, fallback to original " + stid, e);
            }

            try {
                extractFaces(context, stid, doc, callback);
            } catch (BadRequestException bre) {
                super.failed(bre);
            }
        }
    }
}
