package ru.yandex.ps.disk.search;

import java.util.logging.Logger;

import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.client.AbstractAsyncClient;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.string.NonNegativeLongValidator;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.search.disk.kali.KaliActionType;
import ru.yandex.search.disk.kali.KaliCleanupType;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.proxy.universal.UniversalSearchProxyRequestContext;

public class DifaceRequestContext
    implements UniversalSearchProxyRequestContext
{
    private final Diface diface;
    private final ProxySession session;
    private final long prefix;
    private final KaliActionType actionType;
    private final KaliCleanupType cleanupType;
    private final Long timestamp;
    private final String zooQueue;
    private final long zooShardId;
    private final Long minPos;
    private final long cgiZooQueueId;
    private final AsyncClient searchClient;
    private final AsyncClient indexerClient;
    private final User user;

    private final String resourceId;
    private final String id;
    private final long owner;
    private final long version;

    public DifaceRequestContext(
        final Diface diface,
        final ProxySession session)
        throws BadRequestException
    {
        this.diface = diface;
        this.session = session;
        CgiParams params = session.params();
        prefix = params.get("prefix", NonNegativeLongValidator.INSTANCE);
        user = new User( "disk_queue", new LongPrefix(prefix));
        switch (params.get("action", NonEmptyValidator.INSTANCE)) {
            case "rm":
            case "kicked_by_unshare":
            case "leave_folder":
            case "trash_drop_all":
            case "trash_drop_element":
                actionType = KaliActionType.REMOVE;
                break;
            case "reindex":
                actionType = KaliActionType.REINDEX;
                break;
            default:
                actionType = KaliActionType.UPDATE;
                break;
        }
        cleanupType = params.getEnum(
            KaliCleanupType.class,
            "cleanup-type",
            KaliCleanupType.NONE);
        timestamp = params.get(
            "timestamp",
            null,
            NonNegativeLongValidator.INSTANCE);
        zooQueue = session.headers().get(
            YandexHeaders.ZOO_QUEUE,
            NonEmptyValidator.INSTANCE);
        zooShardId = session.headers().get(
            YandexHeaders.ZOO_SHARD_ID,
            NonNegativeLongValidator.INSTANCE);
        minPos = session.headers().getLong(YandexHeaders.ZOO_QUEUE_ID, 0L);
        // zooQueueId could absent for parallel requests, while this
        // value is taken from zoo-queue-id CGI-param
        if (minPos == null) {
            cgiZooQueueId = session.params().get(
                "zoo-queue-id",
                NonNegativeLongValidator.INSTANCE);
        } else {
            cgiZooQueueId = minPos.longValue();
        }

        searchClient =
            diface.searchClient().adjust(session.context());
        indexerClient =
            diface.indexerClient().adjust(session.context());

        id = params.get("id", NonEmptyValidator.INSTANCE);
        resourceId = params.get("resource_id", NonEmptyValidator.INSTANCE);
        try {
            owner = Long.parseLong(
                resourceId.substring(0, resourceId.indexOf(':')));
        } catch (RuntimeException e) {
            throw params.parseFailedException("resource_id", resourceId, e);
        }
        version = params.get("version", NonNegativeLongValidator.INSTANCE);
    }

    public AsyncClient searchClient() {
        return searchClient;
    }

    public AsyncClient indexerClient() {
        return indexerClient;
    }

    public String resourceId() {
        return resourceId;
    }

    public String id() {
        return id;
    }

    public long owner() {
        return owner;
    }

    public long version() {
        return version;
    }

    public Diface proxy() {
        return diface;
    }

    public ProxySession session() {
        return session;
    }

    public long prefix() {
        return prefix;
    }

    public KaliActionType actionType() {
        return actionType;
    }

    public KaliCleanupType cleanupType() {
        return cleanupType;
    }

    public String zooQueue() {
        return zooQueue;
    }

    public long zooShardId() {
        return zooShardId;
    }

    public long cgiZooQueueId() {
        return cgiZooQueueId;
    }

    @Override
    public User user() {
        return user;
    }

    @Override
    public Long minPos() {
        return minPos;
    }

    @Override
    public AbstractAsyncClient<?> client() {
        return searchClient;
    }

    @Override
    public Logger logger() {
        return session.logger();
    }

    @Override
    public long lagTolerance() {
        return 0L;
    }

    public Long timestamp() {
        return timestamp;
    }
}
