package ru.yandex.ps.disk.search;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.WritableByteChannel;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;

import org.apache.http.HttpEntity;
import org.apache.http.HttpException;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpResponseException;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.ContentDecoder;
import org.apache.http.nio.IOControl;
import org.apache.http.nio.protocol.AbstractAsyncResponseConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestProducer;
import org.apache.http.protocol.HttpContext;

import ru.yandex.charset.Decoder;
import ru.yandex.collection.Pattern;
import ru.yandex.function.GenericSupplier;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.HttpStatusPredicates;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.HttpAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.io.IOStreamUtils;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.ps.disk.search.config.ImmutableDifaceConfig;
import ru.yandex.search.proxy.universal.UniversalSearchProxy;

public class ExifInfoExtractor
    extends UniversalSearchProxy<ImmutableDifaceConfig>
{
    private final Timer timer;
    private final AsyncClient cokemulatorClient;

    public ExifInfoExtractor(final ImmutableDifaceConfig config) throws IOException {
        super(config);
        this.cokemulatorClient = client("CokemulatorClient", config.cokemulatorConfig());
        this.timer = new Timer(true);
        this.register(
            new Pattern<>("/exif", true),
            new ExtractExifHandler(this));
    }

    public ExifInfoExtractor(final Diface diface) throws IOException {
        super(diface.config());

        this.cokemulatorClient = diface.cokemulatorClient();
        this.timer = new Timer(true);
    }

    public void extract(
        final ProxySession session,
        final String stid,
        final FutureCallback<Map<String, JsonObject>> callback)
    {
        BasicAsyncRequestProducerGenerator generator
            = new BasicAsyncRequestProducerGenerator(
                config.cokemulatorConfig().host()
                    + "/get-text?stid=" + stid);

        DataCallback dataCallback = new DataCallback(session, callback);
        cokemulatorClient.execute(
            config.cokemulatorConfig().host(),
            generator,
            new SubprocessPipeConsumerFactory(dataCallback),
            dataCallback);
    }

    private class ProcessWaiter extends TimerTask {
        private final Process process;
        private final long startTs;
        private final long timeout;

        public ProcessWaiter(final Process process, final long timeout) {
            this.process = process;
            this.startTs = System.currentTimeMillis();
            this.timeout = timeout;
        }

        @Override
        public void run() {
            long left = timeout - (System.currentTimeMillis() - startTs);
            if (left <= 0) {
                process.destroyForcibly();
                this.cancel();
                return;
            }

            if (!process.isAlive()) {
                this.cancel();
                return;
            }

            timer.schedule(this, left + 100);
        }
    }

    private class DataCallback
        extends AbstractFilterFutureCallback<Void, Map<String, JsonObject>>
        implements Function<Process, Void>, GenericSupplier<OutputStream, IOException>
    {
        private final ProxySession session;
        private volatile ProcessWaiter waiter;
        private volatile boolean done = false;

        public DataCallback(
            final ProxySession session,
            final FutureCallback<? super Map<String, JsonObject>> callback)
        {
            super(callback);
            this.session = session;
        }

        @Override
        public OutputStream get() throws IOException{
            synchronized (this) {
                if (done) {
                    throw new IOException("Request already done");
                }

                if (this.waiter != null) {
                    return waiter.process.getOutputStream();
                }

                ProcessBuilder pb = new ProcessBuilder("exiftool", "-", "-json");
                Process process = pb.start();
                process.onExit().thenApply(this);
                this.waiter = new ProcessWaiter(process, TimeUnit.MINUTES.toMillis(3));
                timer.schedule(waiter, waiter.timeout);
                return waiter.process.getOutputStream();
            }
        }

        @Override
        public void completed(final Void process) {
            //session.logger().info("Mulcagate request completed, process started " + process.pid());
//            try {
//                process.waitFor();
//                //session.logger().info("Process completed " + process.exitValue());
//            } catch (Exception e) {
//                failed(e);
//                return;
//            }

        }

        @Override
        public void cancelled() {
            synchronized (this) {
                if (!done && waiter != null) {
                    waiter.process.destroy();
                    done = true;
                }
            }

            callback.cancelled();
        }

        @Override
        public void failed(final Exception e) {
            synchronized (this) {
                if (!done && waiter != null) {
                    waiter.process.destroy();
                    done = true;
                }
            }
            super.failed(e);
        }

        @Override
        public Void apply(final Process process) {
            if (this.waiter != null) {
                this.waiter.cancel();
            }
            //session.logger().info("Process finished " + process.exitValue());
            if (process.exitValue() != 0) {
                session.logger().info("Exiftool code " + process.exitValue());
                try {
                    Decoder errorDecoder = new Decoder(StandardCharsets.UTF_8);
                    IOStreamUtils.consume(process.getErrorStream()).processWith(errorDecoder);
                    Decoder outputDecoder = new Decoder(StandardCharsets.UTF_8);
                    IOStreamUtils.consume(process.getInputStream()).processWith(outputDecoder);
                    callback.failed(
                        new Exception(
                            "ExifTool returned " + process.exitValue()
                                + " error: " + errorDecoder.toString() + " output " + outputDecoder.toString()));
                } catch (IOException ioe) {
                    callback.failed(new Exception("ExifTool returned " + process.exitValue(), ioe));
                }
            } else {
                try {
                    JsonList list =
                        TypesafeValueContentHandler.parse(
                            new InputStreamReader(process.getInputStream(), StandardCharsets.UTF_8)).asList();
                    if (list.size() == 0) {
                        callback.completed(Collections.emptyMap());
                        return null;
                    }

                    JsonMap map = list.get(0).asMap();
                    Map<String, JsonObject> result = new LinkedHashMap<>(map.size() << 1);
                    for (Map.Entry<String, JsonObject> entry: map.entrySet()) {
                        result.put(entry.getKey(), entry.getValue());
                    }

                    callback.completed(result);
                } catch (JsonException | IOException e) {
                    callback.failed(new Exception("ExifTool returned " + process.exitValue(), e));
                }
            }

            return null;
        }
    }

    private static class SubprocessPipeConsumerFactory
        implements HttpAsyncResponseConsumerFactory<Void> {
        private final GenericSupplier<OutputStream, IOException> pb;

        public SubprocessPipeConsumerFactory(
            final GenericSupplier<OutputStream, IOException> pb) {
            this.pb = pb;
        }

        @Override
        public SubprocessPipeConsumer create(
            final HttpAsyncRequestProducer producer,
            final HttpResponse response)
            throws HttpException, FileNotFoundException
        {
            return new SubprocessPipeConsumer(pb);
        }
    }

    private static class SubprocessPipeConsumer extends AbstractAsyncResponseConsumer<Void> {
        private WritableByteChannel outputChannel;
        private HttpResponse response;
        private final ByteBuffer buffer;
        private final GenericSupplier<OutputStream, IOException> outputStreamSupplier;

        public SubprocessPipeConsumer(final GenericSupplier<OutputStream, IOException> outputStreamSupplier) {
            buffer = ByteBuffer.allocate(4096 * 4);
            this.outputStreamSupplier = outputStreamSupplier;
        }

        @Override
        protected void onResponseReceived(final HttpResponse response) throws HttpException, IOException {
            this.response = response;
        }

        @Override
        protected void onEntityEnclosed(HttpEntity entity, ContentType contentType) throws IOException {
            outputChannel = Channels.newChannel(outputStreamSupplier.get());
        }

        @Override
        protected void onContentReceived(
            final ContentDecoder decoder,
            final IOControl ioControl)
            throws IOException
        {
            if (!HttpStatusPredicates.OK.test(response.getStatusLine().getStatusCode())) {
                while (decoder.read(buffer) > 0) {
                    buffer.clear();
                }
            } else {
                while (decoder.read(buffer) > 0) {
                    buffer.flip();
                    outputChannel.write(buffer);
                    buffer.clear();
                }
            }
        }

        @Override
        protected Void buildResult(final HttpContext context) throws Exception {
            if (HttpStatusPredicates.OK.test(response.getStatusLine().getStatusCode())) {
                outputChannel.close();
                return null;
            } else {
                throw new HttpResponseException(
                    response.getStatusLine().getStatusCode(),
                    "Failed to get image data: " + response.getStatusLine());
            }
        }

        @Override
        protected void releaseResources() {
        }
    }
}
