package ru.yandex.ps.disk.search;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.FilterFutureCallback;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyNHttpEntity;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.string.PositiveIntegerValidator;
import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.proxy.universal.BasicUniversalSearchProxyRequestContext;
import ru.yandex.search.proxy.universal.UniversalSearchProxy;

public class ExtractExifHandler implements ProxyRequestHandler {
    private static final CollectionParser<String, Set<String>, Exception>
        SET_PARSER = new CollectionParser<>(NonEmptyValidator.TRIMMED, LinkedHashSet::new);
    private final ExifInfoExtractor extractor;
    private final UniversalSearchProxy server;

    public ExtractExifHandler(final ExifInfoExtractor extractor) {
        this.extractor = extractor;
        this.server = extractor;
    }

    @Override
    public void handle(final ProxySession session) throws HttpException, IOException {
        CgiParams params = session.params();
        List<String> stids = params.getAll("stid");
        if (!stids.isEmpty()) {
            MultiFutureCallback<Map<String, JsonObject>> mfcb =
                new MultiFutureCallback<>(
                    new PicasaPrinter(session, stids));

            for (String stid: stids) {
                extractor.extract(session, stid, mfcb.newCallback());
            }

            mfcb.done();
        } else {
            long uid = params.getLong("uid");
            QueryConstructor qc = new QueryConstructor("/search-diface&exif?");
            StringBuilder querySb = new StringBuilder();
            querySb.append(
                params.getString(
                    "query",
                    "mimetype:image/*"));

            qc.append("text", querySb.toString());
            //qc.append("dp", "fallback(etime,ctime,mtime date)");
            //qc.append("sort", "date");
            qc.copyIfPresent(params, "length");
            qc.copyIfPresent(params, "dp");
            qc.copyIfPresent(params, "postfilter");
            qc.copyIfPresent(params, "sort");
            qc.append("prefix", uid);
            qc.append("get", "stid");
            session.params().putIfAbsent("service", Collections.singletonList("disk_queue"));

            Set<String> fields = session.params().get("field", Collections.emptySet(), SET_PARSER);
            server.sequentialRequest(
                session,
                new BasicUniversalSearchProxyRequestContext(
                    server,
                    session,
                    EmptyNHttpEntity.INSTANCE,
                    false),
                new BasicAsyncRequestProducerGenerator(qc.toString()),
                60000L,
                false,
                JsonAsyncTypesafeDomConsumerFactory.OK,
                session.listener().createContextGeneratorFor(server.searchClient().adjust(session.context())),
                new IndexCallback(session, fields));
        }
    }

    private class IndexCallback extends AbstractProxySessionCallback<JsonObject> {
        private final Set<String> fields;
        public IndexCallback(final ProxySession session, final Set<String> fields) {
            super(session);

            this.fields = fields;
        }

        @Override
        public void completed(final JsonObject result) {
            try {
                JsonList hits = result.asMap().getList("hitsArray");
                List<String> stids = new ArrayList<>(hits.size());
                for (JsonObject obj: hits) {
                    stids.add(obj.asMap().getString("stid"));
                }

                session.logger().info("Total stids: " + stids.size());
                PicasaPrinter printer = new PicasaPrinter(session, stids);

                if (stids.size() <= 0) {
                    printer.completed(Collections.emptyList());
                } else {
                    BatchCallback batchCallback = new BatchCallback(printer, session, stids, fields);
                    batchCallback.nextBatch();
                }
            } catch (JsonException | BadRequestException je) {
                failed(je);
            }
        }
    }

    private class BatchCallback extends FilterFutureCallback<List<Map<String, JsonObject>>> {
        private final List<String> stids;
        private final ProxySession session;
        private final Set<String> fields;
        private volatile int index = 0;
        private final int batchSize;
        private final List<Map<String, JsonObject>> result = new ArrayList<>();

        public BatchCallback(
            final FutureCallback<? super List<Map<String, JsonObject>>> callback,
            final ProxySession session,
            final List<String> stids,
            final Set<String> fields)
            throws BadRequestException
        {
            super(callback);
            this.stids = stids;
            this.session = session;
            this.fields = fields;
            this.batchSize = session.params().get("batch", 50, PositiveIntegerValidator.INSTANCE);
        }

        public void nextBatch() {
            MultiFutureCallback<Map<String, JsonObject>> mfcb
                = new MultiFutureCallback<>(this);

            synchronized (this) {
                int max = Math.min(stids.size(), batchSize + index);
                for (;index < max; index++) {
                    extractor.extract(
                        session,
                        stids.get(index),
                        new ExifSuppressErrorCallback(
                            mfcb.newCallback(),
                            session.logger(),
                            fields));
                }
            }

            mfcb.done();
        }

        @Override
        public void completed(final List<Map<String, JsonObject>> result) {
            boolean done = false;
            synchronized (this) {
                this.result.addAll(result);
                if (index >= stids.size()) {
                    done = true;
                }
            }

            session.logger().info("Completed " + this.result.size() + "/" + stids.size());
            if (!done) {
                nextBatch();
            } else {
                callback.completed(this.result);
            }
        }
    }

    private static class ExifSuppressErrorCallback extends FilterFutureCallback<Map<String, JsonObject>> {
        private final PrefixedLogger logger;
        private final Set<String> fields;

        public ExifSuppressErrorCallback(
            final FutureCallback<? super Map<String, JsonObject>> callback,
            final PrefixedLogger logger,
            final Set<String> fields)
        {
            super(callback);
            this.logger = logger;
            this.fields = fields;
        }

        @Override
        public void completed(final Map<String, JsonObject> map) {
            map.keySet().retainAll(fields);
            if (map.size() > 0) {
                callback.completed(map);
            } else {
                callback.completed(Collections.emptyMap());
            }
        }

        @Override
        public void failed(final Exception e) {
            logger.log(Level.WARNING, "Exif failed", e);
            callback.completed(Collections.emptyMap());
        }

        @Override
        public void cancelled() {
            callback.completed(Collections.emptyMap());
        }
    }

    private static final class PicasaPrinter
        extends AbstractProxySessionCallback<List<Map<String, JsonObject>>>
    {
        private final List<String> stids;
        private final JsonType jsonType;

        public PicasaPrinter(final ProxySession session, final List<String> stids) throws BadRequestException {
            super(session);
            this.stids = stids;
            this.jsonType = JsonTypeExtractor.NORMAL.extract(session.params());
        }

        @Override
        public void completed(final List<Map<String, JsonObject>> maps) {
            StringBuilderWriter sbw = new StringBuilderWriter();
            try (JsonWriter writer = jsonType.create(sbw)) {
                writer.startArray();
                for (int i = 0; i < stids.size(); i++) {
                    Map<String, JsonObject> map = maps.get(i);
                    if (map.size() > 0) {
                        writer.startObject();
                        writer.key("stid");
                        writer.value(stids.get(i));
                        for (Map.Entry<String, JsonObject> entry: map.entrySet()) {
                            writer.key(entry.getKey());
                            writer.value(entry.getValue());
                        }
                        writer.endObject();
                    }
                }

                writer.endArray();
            } catch (IOException ioe) {
                failed(ioe);
                return;
            }

            session.response(HttpStatus.SC_OK, sbw.toString());
        }
    }
}
