package ru.yandex.ps.disk.search;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.Arrays;
import java.util.LinkedHashSet;
import java.util.Set;

import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.util.string.HexStrings;
import ru.yandex.util.string.StringUtils;
import ru.yandex.util.string.UnhexStrings;

public class Face implements JsonValue {
    private static final double DIVISOR = 127 * 127;
    private final String faceId;
    private final String resourceId;
    private byte[] vector;
    private double[] dvector;
    private final double x, y, w, h;
    private final double age;
    private final double femaleProb;
    private final double confidence;
    private final int resourceWidth;
    private final int resourceHeight;
    private final double imageHeight;
    private final double imageWidth;
    private final Set<String> similarToClusters = new LinkedHashSet<>();
    private volatile Cluster cluster = null;
    private volatile Cluster oldCluster = null;
    private volatile boolean side = false;
    private volatile boolean deleted = false;
    private volatile PointStatus status;
    private volatile String name;

    public int index = -1;

    public Face(
        final String faceId,
        final String resourceId,
        final int resourceWidth,
        final int resourceHeight,
        final double x,
        final double y,
        final double w,
        final double h,
        final double confidence,
        final byte[] vector,
        final double[] dvector,
        final double age,
        final double femaleProb)
    {
        this.vector = vector;
        this.dvector = dvector;
        this.faceId = faceId;
        this.confidence = confidence;
        this.resourceId = resourceId;
        this.x = x;
        this.y = y;
        this.w = w;
        this.h = h;
        this.age = age;
        this.femaleProb = femaleProb;
        this.resourceWidth = resourceWidth;
        this.resourceHeight = resourceHeight;
        this.imageHeight = resourceHeight;
        this.imageWidth = resourceWidth;
    }

    public Face(
        final String faceId,
        final String resourceId,
        final int resourceWidth,
        final int resourceHeight,
        final double imageWidth,
        final double imageHeight,
        final double x,
        final double y,
        final double w,
        final double h,
        final double confidence,
        final byte[] vector,
        final double age,
        final double femaleProb)
    {
        this.vector = vector;
        this.dvector = bytesToDoubleArray(vector);
        this.faceId = faceId;
        this.confidence = confidence;
        this.resourceId = resourceId;
        this.x = x;
        this.y = y;
        this.w = w;
        this.h = h;
        this.age = age;
        this.femaleProb = femaleProb;
        this.resourceWidth = resourceWidth;
        this.resourceHeight = resourceHeight;
        this.imageWidth = imageWidth;
        this.imageHeight = imageHeight;
    }

    public Face(
        final String faceId,
        final String resourceId,
        final int resourceWidth,
        final int resourceHeight,
        final double x,
        final double y,
        final double w,
        final double h,
        final double confidence,
        final byte[] vector,
        final double age,
        final double femaleProb)
    {
        this(
            faceId,
            resourceId,
            resourceWidth,
            resourceHeight,
            resourceWidth,
            resourceHeight,
            x,
            y,
            w,
            h,
            confidence,
            vector,
            age,
            femaleProb);
    }

    public String name() {
        return name;
    }

    public Face name(final String name) {
        this.name = name;
        return this;
    }

    public String resourceId() {
        return resourceId;
    }

    public void resetChanged() {
        oldCluster = cluster;
    }

    public boolean changed() {
        return oldCluster != cluster;
    }

    public Cluster oldCluster() {
        return oldCluster;
    }

    public Set<String> clustersIds() {
        return similarToClusters;
    }

    public int resourceWidth() {
        return resourceWidth;
    }

    public int resourceHeight() {
        return resourceHeight;
    }

    public double imageHeight() {
        return imageHeight;
    }

    public double imageWidth() {
        return imageWidth;
    }

    public static byte[] doubleToByteArray(final double[] doubleArray){
        int times = Float.SIZE / Byte.SIZE;
        byte[] bytes = new byte[doubleArray.length * times];
        for(int i=0;i<doubleArray.length;i++){
            ByteBuffer.wrap(bytes, i*times, times).order(ByteOrder.LITTLE_ENDIAN).putFloat((float)doubleArray[i]);
        }
        return bytes;
    }

    public static double[] bytesToDoubleArray(final byte[] byteArray) {
        int times = Float.SIZE / Byte.SIZE;
        double[] doubles = new double[byteArray.length / times];
        ByteBuffer buffer =
            ByteBuffer.wrap(byteArray, 0, byteArray.length).order(ByteOrder.LITTLE_ENDIAN);
        int i = 0;
        while (buffer.position() < byteArray.length) {
            doubles[i] = buffer.getFloat();
            i += 1;
        }

        return doubles;
    }

    public double dotProduct(final Face other) {
//        if (side || other.side) {
//            return 0;
//        }

        double dp = dotProduct(other.dvector);

        return dp;
    }

    public double dotProduct(final byte[] other) {
        return dotProduct(vector, other);
    }

    public double dotProduct(final double[] other) {
        return dotProduct(dvector, other);
    }

    public static double dotProduct(final byte[] a, final byte[] b) {
        final int len = a.length;
        int idp = 0;
        for (int i = 0; i < len; i++) {
            idp += a[i] * b[i];
        }
        return idp / DIVISOR;
    }

    public static double dotProduct(final double[] a, final double[] b) {
        final int len = a.length;
        double dp = 0;
        for (int i = 0; i < len; i++) {
            dp += a[i] * b[i];
        }
        return dp;
    }

    @Override
    public String toString() {
        return faceId;
    }

    public String faceId() {
        return faceId;
    }

    public byte[] vector() {
        return vector;
    }

    public double[] dvector() {
        return dvector;
    }

    public double x() {
        return x;
    }

    public double y() {
        return y;
    }

    public double w() {
        return w;
    }

    public double h() {
        return h;
    }

    public PointStatus status() {
        return status;
    }

    public double age() {
        return age;
    }

    public Cluster cluster() {
        return cluster;
    }

    public synchronized boolean side() {
        return side;
    }

    public Face vector(byte[] vector) {
        this.vector = vector;
        return this;
    }

    public Face dvector(double[] dvector) {
        this.dvector = dvector;
        return this;
    }

    public Face status(PointStatus status) {
        this.status = status;
        return this;
    }

    public synchronized Face similarCluster(final Cluster cluster) {
        similarToClusters.add(cluster.id());
        return this;
    }

    public Set<String> similarToClusters() {
        return similarToClusters;
    }

    public Face cluster(final Cluster cluster) {
        this.cluster = cluster;
        return this;
    }

    public synchronized Face side(boolean side) {
        this.side = side;
        return this;
    }

    public Face deleted(final boolean deleted) {
        this.deleted = deleted;
        return this;
    }

    public boolean deleted() {
        return deleted;
    }

    public double femaleProb() {
        return femaleProb;
    }

    public double confidence() {
        return confidence;
    }

    public static Face parseFromBackend(
        final JsonObject jsonObj)
        throws JsonException
    {
        JsonMap map = jsonObj.asMap();
        JsonObject vectorObj = map.get(FaceBackendFields.FACE_VECTOR.stored());
        byte[] bvector;
        if (vectorObj.type() == JsonObject.Type.LIST) {
            JsonList list = vectorObj.asList();
            bvector = new byte[list.size()];
            for (int i = 0; i < bvector.length; i++) {
                bvector[i] = (byte) list.get(i).asLong();
            }
        } else {
            bvector = UnhexStrings.unhex(vectorObj.asString());
        }
//        = UnhexStrings.unhex(
//            );

        Face face = new Face(
            map.getString(FaceBackendFields.FACE_ID.stored()),
            map.getString(FaceBackendFields.FACE_RESOURCE_ID.stored()),
            map.getInt(FaceBackendFields.FACE_RESOURCE_WIDTH.stored(), -1),
            map.getInt(FaceBackendFields.FACE_RESOURCE_HEIGHT.stored(), -1),
            map.getDouble(FaceBackendFields.FACE_COORD_X.stored()),
            map.getDouble(FaceBackendFields.FACE_COORD_Y.stored()),
            map.getDouble(FaceBackendFields.FACE_WIDTH.stored()),
            map.getDouble(FaceBackendFields.FACE_HEIGHT.stored()),
            map.getDouble(FaceBackendFields.FACE_CONFIDENCE.stored()),
            bvector,
            map.getDouble(FaceBackendFields.FACE_AGE.stored()),
            0.0);
        Boolean side = map.getBoolean(FaceBackendFields.FACE_IS_SIDE.stored(), false);
        if (Boolean.TRUE.equals(side)) {
            System.err.println("Side true " + face.resourceId());
            face.side(true);
        }

        return face;
    }

    public static Face parseFromExclusions(final JsonObject jsonObject) throws JsonException {
        JsonMap map = jsonObject.asMap();
        String resId = map.getString(FaceBackendFields.FACE_RESOURCE_ID.stored());
        String faceId = map.getString(FaceBackendFields.FACE_ID.stored());
        JsonList varr = map.getList(FaceBackendFields.FACE_VECTOR.stored());
        double[] dvector = new double[varr.size()];
        for (int i = 0; i < varr.size(); i++) {
            dvector[i] = varr.get(i).asDouble();
        }

        byte[] bvector = doubleToByteArray(dvector);
        return new Face(
            faceId,
            resId,
            1000,
            1000,
            0,
            0,
            100,
            100,
            1.0,
            bvector,
            dvector,
            30.0,
            0);
    }

    public static Face parseFromCoke(
        final DiskDoc doc,
        final int sequence,
        final JsonObject jsonObj)
        throws JsonException
    {
        JsonMap map = jsonObj.asMap();
        double x = map.getDouble("Left");
        double y = map.getDouble("Top");
        double w = map.getDouble("Width");
        double h = map.getDouble("Height");
        double age = map.getDouble("Age");
        double femaleProb = map.getDouble("FemaleProb");
        double conf = map.getDouble("Confidence");
        if (conf < 0.5) {
            return null;
        }

        JsonList varr = map.getList("Signature");
        double[] dvector = new double[varr.size()];
        for (int i = 0; i < varr.size(); i++) {
            dvector[i] = varr.get(i).asDouble();
        }

        byte[] bvector = doubleToByteArray(dvector);

        String faceId = doc.resourceId() + '_' + sequence;

        return new Face(
            faceId,
            doc.resourceId(),
            doc.width(),
            doc.height(),
            x,
            y,
            w,
            h,
            conf,
            bvector,
            dvector,
            age,
            femaleProb);
    }

    public static Face parseFromModel(
        final DiskDoc doc,
        final FaceStat stat,
        final int sequence,
        final JsonObject jsonObj)
        throws JsonException
    {
        JsonMap map = jsonObj.asMap();
        byte[] bvector =
            UnhexStrings.unhex(map.getString("signature"));
        double x = map.getDouble("x");
        double y = map.getDouble("y");
        double w = map.getDouble("width");
        double h = map.getDouble("height");
        double age = map.getDouble("age");
        double femaleProb = map.getDouble("female_probability");
        double conf = map.getDouble("confidence");
        if (conf < 0.5) {
            return null;
        }

        int originalWidth = doc.width();
        int originalHeight = doc.height();

        double exImageWidth = map.getDouble("image_width", (double) originalWidth);
        double exImageHeight = map.getDouble("image_height", (double) originalHeight);

//        JsonList varr = map.getList("Signature");
//        double[] dvector = new double[varr.size()];
//        for (int i = 0; i < varr.size(); i++) {
//            dvector[i] = varr.get(i).asDouble();
//            clamp(Math.round(dvector[i] * 127.0));
//        }


        //byte[] bvector = doubleToByteArray(dvector);

        String faceId = doc.resourceId() + '_' + sequence;

        return new Face(
            faceId,
            doc.resourceId(),
            originalWidth,
            originalHeight,
            exImageWidth,
            exImageHeight,
            x,
            y,
            w,
            h,
            conf,
            bvector,
            age,
            femaleProb);
    }

    public static byte clamp(final long i) {
        return (byte) Math.max(-127, Math.min(127, i));
    }

    public void writeFields(
        final JsonWriterBase writer,
        final boolean hexifyVector)
        throws IOException
    {
        writer.key("id");
        writer.value(Face.searchBackendId(this));
        writer.key(DiskBackendFields.TYPE.stored());
        writer.value("face");
        writer.key(FaceBackendFields.FACE_ID.stored());
        writer.value(faceId);
        writer.key(FaceBackendFields.FACE_RESOURCE_ID.stored());
        writer.value(resourceId);
        writer.key(FaceBackendFields.FACE_RESOURCE_WIDTH.stored());
        writer.value(resourceWidth);
        writer.key(FaceBackendFields.FACE_RESOURCE_HEIGHT.stored());
        writer.value(resourceHeight);
        writer.key(FaceBackendFields.FACE_COORD_X.stored());
        writer.value(x);
        writer.key(FaceBackendFields.FACE_COORD_Y.stored());
        writer.value(y);
        writer.key(FaceBackendFields.FACE_WIDTH.stored());
        writer.value(w);
        writer.key(FaceBackendFields.FACE_HEIGHT.stored());
        writer.value(h);
        writer.key(FaceBackendFields.FACE_AGE.stored());
        writer.value(age);
        writer.key(FaceBackendFields.FACE_CONFIDENCE.stored());
        writer.value(confidence);
        writer.key(FaceBackendFields.FACE_GENDER.stored());
        writer.value(femaleProb);
        writer.key(FaceBackendFields.FACE_VECTOR.stored());
        if (hexifyVector) {
            writer.value(HexStrings.LOWER.toString(vector));
        } else {
            writer.value(Arrays.toString(vector));
        }

        writer.key("face_name");
        writer.value(name);

        if (cluster != null) {
            writer.key(FaceBackendFields.FACE_CLUSTER_ID.stored());
            writer.value(cluster.id());
        }

        writer.key(FaceBackendFields.FACE_IS_SIDE.stored());
        if (side) {
            writer.value(1);
        } else {
            writer.value(JsonNull.INSTANCE);
        }

        if (!similarToClusters.isEmpty()) {
            writer.key(FaceBackendFields.FACE_SIMILAR_CLUSTERS.stored());
            writer.value(StringUtils.join(similarToClusters, '\n'));
        }
    }

    @Override
    public void writeValue(final JsonWriterBase writer) throws IOException {
        writer.startObject();
        writeFields(writer, true);
        writer.endObject();
    }

    public static String searchBackendId(final Face face) {
        return "face_" + face.faceId;
    }
}
