package ru.yandex.ps.disk.search;

import java.io.IOException;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.PathMatcher;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.logging.Level;

import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.ps.disk.search.config.ImmutableStaticFacesConfig;

public class FaceExclusionsLoader {
    private static List<Face> loadFacesFromFile(
        final Path path)
        throws IOException, JsonException
    {
        JsonObject obj = TypesafeValueContentHandler.parse(Files.readString(path));
        List<Face> faces = new ArrayList<>();
        for (JsonObject item : obj.asList()) {
            faces.add(Face.parseFromExclusions(item));
        }

        return faces;
    }

    public static List<Face> loadSides(
        final PrefixedLogger logger,
        final ImmutableStaticFacesConfig config)
        throws IOException
    {
        if (!config.enabled()) {
            return Collections.emptyList();
        }

        List<Path> files;
        List<Face> sideFaces = new ArrayList<>();
        String path = config.path();

        if (path.contains("*")) {
            String[] split = path.split("/");
            StringBuilder baseLocation = new StringBuilder();
            for (String s: split) {
                if (s.contains("*")) {
                   break;
                }

                baseLocation.append(s);
                baseLocation.append('/');
            }

            if (baseLocation.length() > 1) {
                baseLocation.setLength(baseLocation.length() - 1);
            }

            logger.info("Looking for glob " + path + " in location " + baseLocation);
            Visitor visitor = new Visitor(path);
            Files.walkFileTree(Paths.get(baseLocation.toString()), visitor);
            files = visitor.files();
        } else {
            files = Collections.singletonList(Paths.get(path));
        }

        logger.info("Files to parse for face exclusions " + files);

        for (Path file: files) {
            try {
                sideFaces.addAll(loadFacesFromFile(file));
            } catch (IOException | JsonException e) {
                logger.log(Level.WARNING, "Failed to load exclusions from " + path, e);
            }
        }

        logger.info("Loaded exclusions " + sideFaces.size() + " from files " + files.size());
        return sideFaces;
    }

    private static class Visitor extends SimpleFileVisitor<Path> {
        private final PathMatcher pathMatcher;
        private final List<Path> files;

        public Visitor(final String glob) {
            pathMatcher = FileSystems.getDefault().getPathMatcher("glob:" + glob);
            files = new ArrayList<>();
        }

        public List<Path> files() {
            return files;
        }

        @Override
        public FileVisitResult visitFile(
            final Path file,
            final BasicFileAttributes attrs)
            throws IOException
        {
            if (pathMatcher.matches(file)) {
                files.add(file);
            }

            return FileVisitResult.CONTINUE;
        }

        @Override
        public FileVisitResult visitFileFailed(
            final Path file,
            final IOException exc)
            throws IOException
        {
            return FileVisitResult.CONTINUE;
        }
    }
}
