package ru.yandex.ps.disk.search;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;

import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.BadResponseException;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.YandexHttpStatus;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.ocr.proxy.OcrProxyBasicStat;
import ru.yandex.ps.disk.search.delta.DifaceActionHandler;
import ru.yandex.ps.disk.search.delta.RemoveHandler;
import ru.yandex.ps.disk.search.delta.RestoreHandler;
import ru.yandex.ps.disk.search.delta.SkipHandler;
import ru.yandex.ps.disk.search.delta.UpdateHandler;
import ru.yandex.search.disk.DiskFaceActionType;
import ru.yandex.search.disk.DiskFaceActionTypeParser;
import ru.yandex.util.string.HexStrings;

public class FaceInHandler implements ProxyRequestHandler {
    public static final String HASH_PREFIX = HexStrings.LOWER.toString(
        "face_index".getBytes(StandardCharsets.UTF_8));

    private final Diface server;
    private final Map<DiskFaceActionType, DifaceActionHandler> subHandlers;
    private final SkipHandler skipHandler;

    public FaceInHandler(final Diface server) {
        this.server = server;

        this.skipHandler = new SkipHandler();
        Map<DiskFaceActionType, DifaceActionHandler> subHandlers = new LinkedHashMap<>();
        subHandlers.put(DiskFaceActionType.IGNORE, skipHandler);
        subHandlers.put(DiskFaceActionType.UPDATE, new UpdateHandler(server));
        subHandlers.put(DiskFaceActionType.REMOVE, new RemoveHandler(server));
        subHandlers.put(DiskFaceActionType.RESTORE, new RestoreHandler(server));
        this.subHandlers = Collections.unmodifiableMap(subHandlers);
    }

    @Override
    public void handle(
        final ProxySession session)
        throws HttpException, IOException
    {
        HttpRequest request = session.request();
        if (!(request instanceof HttpEntityEnclosingRequest)) {
            throw new BadRequestException("Payload expected");
        }

        PostIndexContext context;
        String data = CharsetUtils.toString(((HttpEntityEnclosingRequest) request).getEntity());
        try {
            List<JsonObject> list;
            JsonObject dataObj = TypesafeValueContentHandler.parse(data);
            if (dataObj.type() == JsonObject.Type.MAP) {
                list = Collections.singletonList(dataObj);
            } else {
                list = dataObj.asList();
            }

            if (list.size() <= 0) {
                throw new BadRequestException("No items in post");
            }

            // TODO need to extract all resources
            context = new PostIndexContext(
                server,
                session,
                list);
        } catch (JsonException | BadRequestException e) {
            DiskFaceActionType actionType = session.params().get("action", DiskFaceActionTypeParser.INSTANCE);
            if (actionType == DiskFaceActionType.IGNORE) {
                session.logger().info("Ignoring action, skip");
                session.response(HttpStatus.SC_OK);
                return;
            }
            throw new BadRequestException("Bad body", e);
        }

        subHandlers.get(context.action()).handle(context);
    }

    public static boolean criticalNonRetriable(
        final Exception e,
        final OcrProxyBasicStat stat)
    {
        boolean result = true;
        if (e instanceof BadResponseException) {
            switch (((BadResponseException) e).statusCode()) {
                case YandexHttpStatus.SC_NOT_FOUND:
                    stat.notFound(true);
                    result = false;
                    break;
                case YandexHttpStatus.SC_LOCKED:
                    stat.locked(true);
                    result = false;
                    break;
                case YandexHttpStatus.SC_UNSUPPORTED_MEDIA_TYPE:
                    stat.unsupported(true);
                    result = false;
                    break;
                case YandexHttpStatus.SC_REQUEST_TOO_LONG:
                    stat.tooLarge(true);
                    result = false;
                    break;
                default:
                    break;
            }
        }
        return result;
    }
}
