package ru.yandex.ps.disk.search.delta;

import java.io.IOException;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.FormBodyPartBuilder;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.entity.mime.content.StringBody;
import org.apache.http.nio.entity.NStringEntity;

import ru.yandex.client.producer.ProducerClient;
import ru.yandex.disk.search.face.DeltaChangeType;
import ru.yandex.disk.search.face.FaceModification;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.StatusCodeAsyncConsumerFactory;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.parser.searchmap.SearchMap;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.ps.disk.search.Diface;
import ru.yandex.ps.disk.search.DiskDoc;
import ru.yandex.ps.disk.search.Face;
import ru.yandex.ps.disk.search.FaceBackendFields;
import ru.yandex.ps.disk.search.FaceInHandler;
import ru.yandex.ps.disk.search.PostIndexContext;
import ru.yandex.ps.disk.search.UserFacesAndClusters;

public class RemoveHandler implements DifaceActionHandler {
    private final Diface server;

    public RemoveHandler(final Diface server) {
        this.server = server;
    }

    @Override
    public void handle(final PostIndexContext context) throws HttpException, IOException {
        context.logger().info("Handling remove");
        server.getIndexedFacesAndClusters(
            context.prefix(),
            context.session(),
            new RemoveGatherDataCallback(context));
    }

    private static class DeleteIndexCallback extends AbstractProxySessionCallback<Object> {
        private final PostIndexContext context;

        public DeleteIndexCallback(final PostIndexContext context) {
            super(context.session());
            this.context = context;
        }

        @Override
        public void completed(final Object o) {
            context.session().response(HttpStatus.SC_OK);
        }
    }

    private static class RemoveGatherDataCallback
        extends AbstractProxySessionCallback<UserFacesAndClusters> {
        private final PostIndexContext context;

        public RemoveGatherDataCallback(final PostIndexContext context) {
            super(context.session());
            this.context = context;
        }

        @Override
        public void completed(final UserFacesAndClusters data) {
            context.session().logger().info("Handle removing");
            DeleteIndexCallback callback = new DeleteIndexCallback(context);
            try {
                List<Face> deleted = new ArrayList<>();
                Map<String, ClusterDelta> deltas = new LinkedHashMap<>();
                for (DiskDoc doc : context.docs()) {
                    List<Face> faces = data.facesByResourceId(doc.resourceId());
                    if (faces == null) {
                        context.logger().info("No faces found for " + doc.resourceId());
                        continue;
                    }

                    context.logger().info("Removing faces: " + JsonType.NORMAL.toString(faces));

                    for (Face face : faces) {
                        if (!face.deleted()) {
                            deleted.add(face);
                            face.deleted(true);
                        } else {
                            continue;
                        }

                        if (face.cluster() != null) {
                            ClusterDelta delta = deltas.computeIfAbsent(
                                face.cluster().id(), (k) ->
                                    ClusterDelta.create(
                                        face.cluster(),
                                        context,
                                        context.queueId()));
                            delta.add(
                                new FaceModification(
                                    DeltaChangeType.ITEM_DELETED,
                                    face.faceId(),
                                    doc.resourceId()));
                        }
                    }
                }

                if (deleted.isEmpty() && deltas.isEmpty()) {
                    context.logger().info("No resource matched faces, skipping");
                    callback.completed(null);
                    return;
                }

                StringBuilderWriter sbw = new StringBuilderWriter();
                JsonWriter writer = JsonType.HUMAN_READABLE.create(sbw);
                writer.startObject();
                writer.key("prefix");
                writer.value(context.prefix());
                writer.key("docs");
                writer.startArray();
                // sending new ones
                for (Face face : deleted) {
                    writer.startObject();
                    writer.key("id");
                    writer.value(Face.searchBackendId(face));
                    writer.key(FaceBackendFields.FACE_DELETED.stored());
                    writer.value(1);
                    writer.endObject();
                }
                writer.endArray();
                writer.endObject();

                QueryConstructor markDeletedUri =
                    new QueryConstructor("/update?faces_index&face_delete");

                markDeletedUri.append("prefix", context.prefix().toStringFast());
                markDeletedUri.append("ids", context.idsRange());
                markDeletedUri.append("face_queue_id", context.queueId());
                markDeletedUri.append("ref_queue_id", context.refQueueId());
                markDeletedUri.append("ref_queue", context.refQueue());
                markDeletedUri.append("service", context.server().config().faceIndexQueue());

                context.logger().info(
                    "Delete Delta clusters " + deltas.size() + " faces " + deleted.size());

                BasicAsyncRequestProducerGenerator generator;
                if (deltas.size() > 0) {
                    MultipartEntityBuilder builder = MultipartEntityBuilder.create();
                    builder.setMimeSubtype("mixed");
                    builder.addPart(
                        FormBodyPartBuilder
                            .create()
                            .addField(
                                YandexHeaders.ZOO_SHARD_ID,
                                String.valueOf(context.prefix().prefix() % SearchMap.SHARDS_COUNT))
                            .addField(YandexHeaders.URI, markDeletedUri.toString())
                            .setBody(new StringBody(sbw.toString(), ContentType.APPLICATION_JSON))
                            .setName("envelope.json")
                            .build());

                    sbw = new StringBuilderWriter();

                    QueryConstructor deltaUri =
                        new QueryConstructor("/modify?faces_index&faces_delta&faces_delete");

                    deltaUri.append("prefix", context.prefix().toStringFast());
                    deltaUri.append("face_queue_id", context.queueId());
                    deltaUri.append("ids", context.idsRange());
                    deltaUri.append("ref_queue_id", context.refQueueId());
                    deltaUri.append("ref_queue", context.refQueue());
                    deltaUri.append("service", context.server().config().faceIndexQueue());

                    String callbackUri = context.server().config().faceDeltaCallback().toString();

                    StringBuilder callbackSb = new StringBuilder(callbackUri.length() + 40);
                    callbackSb.append(callbackUri);
                    callbackSb.append("&face_version=");
                    callbackSb.append(context.queueId());
                    callbackSb.append("&uid=");
                    callbackSb.append(context.prefix().prefix());
                    deltaUri.append("callback", callbackSb.toString());

                    StringBuilder zooHash =
                        new StringBuilder(FaceInHandler.HASH_PREFIX);
                    zooHash.append(Long.toHexString(context.prefix().prefix()));
                    zooHash.append("00000");
                    zooHash.append(Long.toHexString(context.queueId()));

                    deltaUri.append("zoo_hash", zooHash.toString());

                    writer = JsonType.HUMAN_READABLE.create(sbw);
                    writer.startObject();
                    writer.key("prefix");
                    writer.value(context.prefix());
                    writer.key("docs");
                    writer.startArray();
                    for (Map.Entry<String, ClusterDelta> delta: deltas.entrySet()) {
                        writer.value(delta.getValue());
                    }
                    writer.endArray();
                    writer.endObject();

                    builder.addPart(
                        FormBodyPartBuilder
                            .create()
                            .addField(
                                YandexHeaders.ZOO_SHARD_ID,
                                String.valueOf(context.prefix().prefix() % SearchMap.SHARDS_COUNT))
                            .addField(YandexHeaders.URI, deltaUri.toString())
                            .setBody(new StringBody(sbw.toString(), ContentType.APPLICATION_JSON))
                            .setName("envelope.json")
                            .build());

                    QueryConstructor producerUri =
                        new QueryConstructor("/notify?faces_index&face_delete");

                    producerUri.append("uid", context.prefix().toStringFast());
                    producerUri.append("face_queue_id", context.queueId());
                    producerUri.append("ref_queue_id", context.refQueueId());
                    producerUri.append("ref_queue", context.refQueue());
                    producerUri.append("service", context.server().config().faceIndexQueue());

                    generator = new BasicAsyncRequestProducerGenerator(
                        producerUri.toString(),
                        builder.build());
                    generator.addHeader(YandexHeaders.SERVICE, context.server().config().faceIndexQueue());
                } else {
                    NStringEntity entity = new NStringEntity(
                        sbw.toString(),
                        ContentType.APPLICATION_JSON
                            .withCharset(context.session().acceptedCharset()));

                    generator =
                        new BasicAsyncRequestProducerGenerator(
                            markDeletedUri.toString(),
                            entity);
                }

                context.session().logger().info("Index request uri " + markDeletedUri.toString());
                ProducerClient client =
                    context.server().producerClient().adjust(context.session().context());
                client.execute(
                    context.server().config().producerClientConfig().host(),
                    generator,
                    StatusCodeAsyncConsumerFactory.ANY_GOOD,
                    session.listener().createContextGeneratorFor(client),
                    callback);
            } catch (BadRequestException | IOException e) {
                failed(e);
                return;
            }
        }
    }
}
