package ru.yandex.ps.disk.search.delta;

import java.io.IOException;
import java.net.URI;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;

import ru.yandex.client.producer.ProducerClient;
import ru.yandex.collection.IntPair;
import ru.yandex.disk.search.face.ClusterModification;
import ru.yandex.disk.search.face.DeltaChangeType;
import ru.yandex.disk.search.face.FaceModification;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.DoubleFutureCallback;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.RequestErrorType;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.StatusCodeAsyncConsumerFactory;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.ocr.proxy.CvStat;
import ru.yandex.ocr.proxy.OcrProxy;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.ps.disk.search.Cluster;
import ru.yandex.ps.disk.search.DbScanClusterizer;
import ru.yandex.ps.disk.search.Diface;
import ru.yandex.ps.disk.search.DiskDoc;
import ru.yandex.ps.disk.search.Face;
import ru.yandex.ps.disk.search.FaceInHandler;
import ru.yandex.ps.disk.search.PostIndexContext;
import ru.yandex.ps.disk.search.UserFacesAndClusters;

public class UpdateHandler implements DifaceActionHandler {
    private final Diface server;

    public UpdateHandler(final Diface server) {
        this.server = server;
    }

    @Override
    public void handle(final PostIndexContext context) throws HttpException, IOException {
        context.logger().info("Handle update");

        DoubleFutureCallback<UserFacesAndClusters, List<Map.Entry<DiskDoc,List<Face>>>> dfcb =
            new DoubleFutureCallback<>(
                new GatherFacesCallback(context));

        server.getIndexedFacesAndClusters(context.prefix(), context.session(), dfcb.first());
        MultiFutureCallback<Map.Entry<DiskDoc, List<Face>>> mfcb =
            new MultiFutureCallback<>(dfcb.second());
        for (DiskDoc doc: context.docs()) {
            if (doc.width() < 0 || doc.height() < 0) {
                context.logger().info("Skipping " + doc.resourceId() + " no wdth or height");
                continue;
            }

            server.extractFaces(
                context,
                doc,
                new FaceExtractCallback(
                    doc,
                    mfcb.newCallback(),
                    context));
        }
        mfcb.done();
    }

    private static class GatherFacesCallback
        extends AbstractProxySessionCallback<Map.Entry<UserFacesAndClusters, List<Map.Entry<DiskDoc, List<Face>>>>>
    {
        private final PostIndexContext context;

        public GatherFacesCallback(final PostIndexContext context) {
            super(context.session());
            this.context = context;
        }

        @Override
        public void completed(
            final Map.Entry<UserFacesAndClusters, List<Map.Entry<DiskDoc, List<Face>>>> entry)
        {
            UserFacesAndClusters indexed = entry.getKey();

            long version = context.queueId();

            List<Face> faces = new ArrayList<>(entry.getKey().faces().size() + entry.getValue().size());
            context.session().logger().info("From resource " + entry.getValue());


            int skippedExisting = 0;
            int facesFromRes = 0;
            List<Face> newFaces = new ArrayList<>(entry.getValue().size() * 2);
            for (Map.Entry<DiskDoc, List<Face>> resFaces: entry.getValue()) {
                if (indexed.facesByResourceId(resFaces.getKey().resourceId()) != null) {
                    skippedExisting += 1;
                    continue;
                }
                newFaces.addAll(resFaces.getValue());
                faces.addAll(resFaces.getValue());
                facesFromRes += resFaces.getValue().size();
            }


            context.session().logger().info("New faces " + facesFromRes + " skipped existing " + skippedExisting);
            if (newFaces.size() == 0) {
                context.logger().info("No new faces, skipping");
                session.response(HttpStatus.SC_OK);
                return;
            }

            //TODO need to deduplicate if updatng resource id
            // also need to cleanup old if present ?
            for (Map.Entry<String, List<Face>> faceEntry: entry.getKey().faces().entrySet()) {
                for (Face face: faceEntry.getValue()) {
                    face.resetChanged();
                    faces.add(face);
                }
            }

            context.session().logger().info(
                "From index, clusters: " + entry.getKey().clusters().size()
                    + " faces " + faces.size());

            DbScanClusterizer clusterizer = new DbScanClusterizer(
                context,
                context.prefix().toStringFast() + '_' + context.queueId(),
                context.queueId());
            List<Cluster> added = clusterizer.expand(newFaces, faces);

            context.session().logger().info("Clusters added " + added);

            Map<String, ClusterDelta> deltas = new LinkedHashMap<>(added.size() << 1);
            for (Face face: faces) {
                if (face.changed()) {
                    if (face.oldCluster() != null) {
                        ClusterDelta old = deltas.computeIfAbsent(
                            face.oldCluster().id(),
                            (k) -> ClusterDelta.create(face.oldCluster(), context, version));
                        old.add(
                            new FaceModification(
                                DeltaChangeType.ITEM_DELETED,
                                face.faceId(),
                                face.resourceId()));
                    }

                    if (face.cluster() != null) {
                        ClusterDelta newOne = deltas.computeIfAbsent(
                            face.cluster().id(),
                            (k) -> ClusterDelta.create(face.cluster(), context, version));
                        newOne.add(
                            new FaceModification(
                                DeltaChangeType.ITEM_ADDED,
                                face.faceId(),
                                face.resourceId()));
                    }
                }
            }

            for (Cluster cluster: added) {
                deltas.computeIfAbsent(
                    cluster.id(),
                    (k) -> ClusterDelta.create(cluster, context, version)).add(
                    new ClusterModification(
                        DeltaChangeType.CLUSTER_CREATED,
                        cluster.id()));
            }
            List<Cluster> clusters = entry.getKey().clusters();
            clusters.addAll(added);

            try {
                StringBuilderWriter sbw = new StringBuilderWriter();
                try (JsonWriter writer = JsonType.HUMAN_READABLE.create(sbw)) {
                    writer.startObject();
                    writer.key("prefix");
                    writer.value(context.prefix());
//                writer.key("AddIfNotExists");
//                writer.value(true);
                    writer.key("docs");
                    writer.startArray();
                    Set<String> dedup = new LinkedHashSet<>(newFaces.size() << 1);

                    // sending new ones
                    for (Face face: newFaces) {
                        if (dedup.add(face.faceId())) {
                            writer.value(face);
                        }
                    }

                    // updated
                    for (Face face: faces) {
                        if (face.changed()) {
                            if (dedup.add(face.faceId())) {
                                writer.value(face);
                            }
                        }
                    }

                    // clusters
                    for (Cluster cluster: added) {
                        writer.value(cluster);
                    }

                    // now deltas
                    for (Map.Entry<String, ClusterDelta> delta: deltas.entrySet()) {
                        writer.value(delta.getValue());
                    }

                    writer.endArray();
                    writer.endObject();
                }

                context.session().logger().info("Index request body " + sbw.toString());
                NStringEntity entity = new NStringEntity(
                    sbw.toString(),
                    ContentType.APPLICATION_JSON
                        .withCharset(context.session().acceptedCharset()));

                QueryConstructor qc = new QueryConstructor("/modify?faces_index");
                qc.append("prefix", context.prefix().toStringFast());
                qc.append("ids", context.idsRange());
                qc.append("face_queue_id", context.queueId());
                qc.append("ref_queue_id", context.refQueueId());
                qc.append("ref_queue", context.refQueue());
                qc.append("service", context.server().config().faceIndexQueue());

                URI callback = context.server().config().faceDeltaCallback();

                if (deltas.size() > 0 && callback != null) {
                    String callbackUri = callback.toString();
                    StringBuilder callbackSb = new StringBuilder(callbackUri.length() + 40);
                    callbackSb.append(callbackUri);
                    callbackSb.append("&face_version=");
                    callbackSb.append(context.queueId());
                    callbackSb.append("&uid=");
                    callbackSb.append(context.prefix().prefix());
                    qc.append("callback", callbackSb.toString());
                }
                StringBuilder zooHash =
                    new StringBuilder(FaceInHandler.HASH_PREFIX);
                zooHash.append(Long.toHexString(context.prefix().prefix()));
                zooHash.append("00000");
                zooHash.append(Long.toHexString(context.queueId()));

                qc.append("zoo_hash", zooHash.toString());

                context.session().logger().info("Index request uri " + qc.toString());
                ProducerClient client =
                    context.server().producerClient().adjust(context.session().context());
                client.execute(
                    context.server().config().producerClientConfig().host(),
                    new BasicAsyncRequestProducerGenerator(qc.toString(), entity),
                    StatusCodeAsyncConsumerFactory.ANY_GOOD,
                    session.listener().createContextGeneratorFor(client),
                    new IndexCallback(context, clusters, faces));
            } catch (BadRequestException | IOException bre) {
                failed(bre);
            }
        }
    }

    private static class IndexCallback
        extends AbstractProxySessionCallback<IntPair<Void>>
    {
        private final PostIndexContext context;
        private final List<Cluster> clusters;
        private final List<Face> faces;

        public IndexCallback(
            final PostIndexContext context,
            final List<Cluster> clusters,
            final List<Face> faces)
        {
            super(context.session());
            this.context = context;
            this.clusters = clusters;
            this.faces = faces;
        }

        @Override
        public void completed(final IntPair<Void> pair) {
            context.server().updateCache(
                context.prefix().prefix(),
                new UserFacesAndClusters(faces, clusters));
            session.response(pair.first());
        }
    }

    private static class FaceExtractCallback
        extends AbstractFilterFutureCallback<List<Face>, Map.Entry<DiskDoc, List<Face>>>
    {
        private final PostIndexContext context;
        private final DiskDoc doc;

        public FaceExtractCallback(
            final DiskDoc doc,
            final FutureCallback<? super Map.Entry<DiskDoc, List<Face>>> callback,
            final PostIndexContext context)
        {
            super(callback);

            this.doc = doc;
            this.context = context;
        }

        @Override
        public void completed(final List<Face> faces) {
            callback.completed(
                new AbstractMap.SimpleEntry<>(doc, faces));
        }

        @Override
        public void failed(final Exception e) {
            CvStat stat = new CvStat();
            RequestErrorType errorType =
                RequestErrorType.ERROR_CLASSIFIER.apply(e);
            if (errorType == RequestErrorType.NON_RETRIABLE
                || errorType == RequestErrorType.HOST_NON_RETRIABLE)
            {
                if (OcrProxy.criticalNonRetriable(e, stat)) {
                    // Something very bad happended here. Rate limit or
                    // authorization error, let the caller decide what to do.
                    super.failed(e);
                } else {
                    // This file is unprocessable, error type already accounted
                    // in stat, so save stat and reply 200 OK
                    context.server().stat(stat);
                    //session.response(HttpStatus.SC_OK);
                    completed(Collections.emptyList());
                }
            } else {
                // Imageparser can't process image after all retries, skip it
                stat.error(true);
                context.server().stat(stat);
                completed(Collections.emptyList());
            }
        }
    }
}
