package ru.yandex.ps.disk.search.reindex;

import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.AsyncGetURIRequestProducerSupplier;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.ps.disk.search.Diface;
import ru.yandex.ps.disk.search.DiskDoc;

public class DjfsFilter {
    private final AsyncClient djfsClient;

    public DjfsFilter(final Diface server) {
        this.djfsClient = server.djfsClient();
    }

    public void filter(
        final ReindexContext context,
        final List<DiskDoc> items,
        final FutureCallback<List<DiskDoc>> callback)
        throws BadRequestException
    {
        StringBuilder sb =
            new StringBuilder(context.config().djfsConfig().uri().toASCIIString());
        sb.append(context.config().djfsConfig().firstCgiSeparator());
        sb.append("uid=");
        sb.append(context.prefix().prefix());

        AsyncClient client =
            djfsClient.adjust(context.session().context());

        if (items.size() > context.djfsBatch()) {
            MultiFutureCallback<List<DiskDoc>> mfcb =
                new MultiFutureCallback<>(new ConcatCallback(callback, context));
            for (int i = 0; i < items.size(); i+= context.djfsBatch()) {
                filter(
                    client,
                    context,
                    sb.toString(),
                    items.subList(i, Math.min(i + context.djfsBatch(), items.size())),
                    mfcb.newCallback());
            }
            mfcb.done();
        } else {
            filter(
                client,
                context,
                sb.toString(),
                items,
                callback);
        }
    }

    private void filter(
        final AsyncClient client,
        final ReindexContext context,
        final String baseUri,
        final List<DiskDoc> items,
        final FutureCallback<List<DiskDoc>> callback)
        throws BadRequestException
    {
        QueryConstructor query = new QueryConstructor(baseUri);
        Map<String, DiskDoc> docs = new LinkedHashMap<>(items.size() << 1);
        for (DiskDoc item: items) {
            query.append("resourceId", item.resourceId());
            docs.put(item.resourceId(), item);
        }

        try {
            if (context.listener()) {
                client.execute(
                    new AsyncGetURIRequestProducerSupplier(query.toString()),
                    JsonAsyncTypesafeDomConsumerFactory.OK,
                    context.session().listener()
                        .createContextGeneratorFor(client),
                    new DjfsCallback(callback, docs, context));
            } else {
                client.execute(
                    new AsyncGetURIRequestProducerSupplier(query.toString()),
                    JsonAsyncTypesafeDomConsumerFactory.OK,
                    new DjfsCallback(callback, docs, context));
            }

        } catch (URISyntaxException e) {
            throw new BadRequestException(e);
        }
    }

    private static final class ConcatCallback
        extends AbstractFilterFutureCallback<List<List<DiskDoc>>,List<DiskDoc>>
    {
        private final ReindexContext context;

        public ConcatCallback(
            final FutureCallback<? super List<DiskDoc>> callback,
            final ReindexContext context)
        {
            super(callback);
            this.context = context;
        }

        @Override
        public void completed(final List<List<DiskDoc>> lists) {
            List<DiskDoc> result = new ArrayList<>(lists.size() * context.djfsBatch());
            for (List<DiskDoc> docs: lists) {
                result.addAll(docs);
            }

            callback.completed(result);
        }
    }

    private static final class DjfsCallback
        extends AbstractFilterFutureCallback<JsonObject, List<DiskDoc>>
    {
        private final ReindexContext context;
        private final Map<String, DiskDoc> docs;

        public DjfsCallback(
            final FutureCallback<? super List<DiskDoc>> callback,
            final Map<String, DiskDoc> docs,
            final ReindexContext context)
        {
            super(callback);

            this.context = context;
            this.docs = docs;
        }

        @Override
        public void completed(final JsonObject result) {
            List<DiskDoc> items;
            try {
                JsonList list = result.get("items").asList();
                items = new ArrayList<>(list.size());
                for (JsonObject itemObj: list) {
                    JsonMap item = itemObj.asMap();
                    String resourceId = item.getString("resource_id");
                    DiskDoc doc = docs.get(resourceId);
                    if (doc == null) {
                        context.logger().warning("Unexpected resource in djfs " + resourceId);
                        continue;
                    }

                    items.add(doc);
                }
            } catch (JsonException je) {
                failed(je);
                return;
            }

            if (context.debug()) {
                context.logger().info("Before djfs " + docs.size() + " after " + items.size());
            }

            context.stat().djfsFiltered(docs.size() - items.size());
            callback.completed(items);
        }
    }
}
