package ru.yandex.ps.disk.search.reindex;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.ps.disk.search.DifaceContext;
import ru.yandex.ps.disk.search.DiskDoc;
import ru.yandex.ps.disk.search.Face;

public class FaceExifMergeCallback
    extends AbstractFilterFutureCallback<Map.Entry<List<Face>, Map<String, JsonObject>>, List<Face>>
{
    private final DifaceContext context;
    private final DiskDoc doc;

    public FaceExifMergeCallback(
        final DifaceContext context,
        final DiskDoc doc,
        final FutureCallback<? super List<Face>> callback)
    {
        super(callback);

        this.context = context;
        this.doc = doc;
    }

    private List<String> stringOrStringList(final JsonObject jo) throws JsonException {
        List<String> result = new ArrayList<>();;
        if (jo.type() == JsonObject.Type.STRING) {
            result.add(jo.asString());
        } else {
            for (JsonObject joi: jo.asList()) {
                result.add(joi.asString());
            }
        }

        return result;
    }

    private List<Double> doubleOrDoubleList(
        final JsonObject jo)
        throws JsonException
    {
        List<Double> result = new ArrayList<>();
        if (jo.type() == JsonObject.Type.DOUBLE || jo.type() == JsonObject.Type.LONG) {
            result.add(jo.asDouble());
        } else {
            for (JsonObject joi: jo.asList()) {
                result.add(joi.asDouble());
            }
        }

        return result;
    }

    private void merge(final List<Face> faces, final Map<String, JsonObject> exif) throws JsonException {
        if (!exif.containsKey("RegionName")) {
            return;
        }

        String orientation = exif.getOrDefault("Orientation", JsonNull.INSTANCE).asStringOrNull();
        boolean rotate180 = "rotate 180".equalsIgnoreCase(orientation);
        List<String> regionNames = stringOrStringList(exif.get("RegionName"));
        List<String> regionTypes = stringOrStringList(exif.get("RegionType"));
        List<Double> regionAreaX = doubleOrDoubleList(exif.get("RegionAreaX"));
        List<Double> regionAreaY = doubleOrDoubleList(exif.get("RegionAreaY"));
        List<Double> regionAreaW = doubleOrDoubleList(exif.get("RegionAreaW"));
        List<Double> regionAreaH = doubleOrDoubleList(exif.get("RegionAreaH"));

        Set<String> uniqueNames = new LinkedHashSet<>();
        Set<String> toDelete = new LinkedHashSet<>();
        for (String cname: regionNames) {
            if (!uniqueNames.add(cname)) {
                toDelete.add(cname);
            }
        }

        if (toDelete.size() > 0) {
            int index = 0;
            int deleted = 0;
            while (index < regionNames.size()) {
                if (toDelete.contains(regionNames.get(index))) {
                    regionNames.remove(index);
                    regionTypes.remove(index);
                    regionAreaX.remove(index);
                    regionAreaY.remove(index);
                    regionAreaW.remove(index);
                    regionAreaH.remove(index);
                    deleted += 1;
                } else {
                    index += 1;
                }
            }

            context.session().logger().info(
                "Deleted dup region name " +  doc.stid() + " "
                    + doc.resourceId()  + " deleted " + deleted + " " + toDelete);
        }

        if (regionNames.size() <= 0) {
            return;
        }

        if (rotate180) {
            for (int i = 0; i < regionAreaX.size(); i++) {
                regionAreaX.set(i, 1 - regionAreaX.get(i));
            }

            for (int i = 0; i < regionAreaY.size(); i++) {
                regionAreaY.set(i, 1 - regionAreaY.get(i));
            }
        }


        context.session().logger().info(
            "RegionName found " + doc.stid() + " " + doc.resourceId() + " ornt " + orientation
                + " " + regionNames
                + " " + regionTypes + " " + regionAreaX + " "
                + regionAreaY + " " + regionAreaW + " " + regionAreaH);

        boolean faceTypes = regionTypes.stream().allMatch("Face"::equalsIgnoreCase);
        if (!faceTypes) {
            context.logger().info("Not face regions " + regionTypes + " " + doc.stid());
            return;
        }

        if (regionTypes.size() != regionAreaX.size()
            || regionTypes.size() != regionAreaY.size()
            || regionTypes.size() != regionAreaH.size()
            || regionTypes.size() != regionAreaW.size())
        {
            context.logger().info(
                "Invalid exif face meta " + doc.stid()
                    + " " + JsonType.NORMAL.toString(exif));
            return;
        }

        if (faces.size() != regionTypes.size()) {
            context.logger().warning("Faces extracted  " + faces.size() + " from exif " + regionNames.size());
        }

        if (faces.size() == 1 && regionNames.size() == 1 && regionTypes.size() == 1) {
            faces.get(0).name(regionNames.get(0));
            return;
        }


        double[][] diffs = new double[regionNames.size()][];
        for (int i = 0; i < regionNames.size(); i++) {
            diffs[i] = new double[faces.size()];
        }
        for (int i = 0; i < regionNames.size(); i++) {
            for (int j = 0; j < faces.size(); j++) {
                Face face = faces.get(j);
                double c1x = face.x() + 0.5 * face.w();
                double c1y = face.y() + 0.5 * face.h();
                double c2x = regionAreaX.get(i);
                double c2y = regionAreaY.get(i);
                double diff = Math.sqrt((c1x - c2x) * (c1x - c2x) + (c1y - c2y) * (c1y - c2y));
                diffs[i][j] = diff;
            }
        }

        int[] exifs = new int[regionNames.size()];
        Arrays.fill(exifs, -1);
        int[] models = new int[faces.size()];
        Arrays.fill(models, -1);
        while (true) {
            double min = Double.MAX_VALUE;
            int minI = -1;
            int minJ = -1;
            for (int i = 0; i < regionNames.size(); i++) {
                if (exifs[i] >= 0) {
                    continue;
                }
                for (int j = 0; j < faces.size(); j++) {
                    if (models[j] >= 0) {
                        continue;
                    }

                    double th1 = 0.5 * Math.sqrt(regionAreaW.get(i) * regionAreaW.get(i) + regionAreaH.get(i) * regionAreaH.get(i));
                    double th2 = 0.5 * Math.sqrt(faces.get(j).w() * faces.get(j).w() + faces.get(j).h() * faces.get(j).h());
                    if (diffs[i][j] < min
                        && diffs[i][j] < th1
                        && diffs[i][j] < th2)
                    {
                        min = diffs[i][j];
                        minI = i;
                        minJ = j;
                    }
                }
            }

            if (minI < 0 || minJ < 0) {
                break;
            }

            exifs[minI] = minJ;
            models[minJ] = minI;
            faces.get(minJ).name(regionNames.get(minI));

            Face face = faces.get(minJ);

            context.logger().warning(
                "RegionName match  " + doc.stid() + " " + doc.resourceId()
                    + " face " + face.name() + " " + face.x() + "," + face.y() + "," + face.w() + "," + face.h()
                    + " exif " + regionAreaX.get(minI) + "," +  regionAreaY.get(minI) + "," +  regionAreaW.get(minI) + "," +  regionAreaH.get(minI));
        }

        boolean miss = false;
        for (int i = 0; i < regionNames.size(); i++) {
            if (exifs[i] < 0) {
                miss = true;
                context.logger().info("not found match for exif face " + i + " " + regionNames.get(i));
                context.logger().info(Arrays.toString(diffs[i]));
                context.stat().exifWithoutMatch(1);
            }
        }

        if (miss) {
            context.logger().info(Arrays.toString(models));
            context.logger().info(Arrays.toString(exifs));
            for (int j = 0; j < faces.size(); j++) {
                Face face = faces.get(j);
                context.logger().info("Face " + j + " " + faces.get(j)
                    + " " + face.w() + " " +face.h() + " " + face.x() + " " + face.y());
            }
        }
    }

    @Override
    public void completed(final Map.Entry<List<Face>, Map<String, JsonObject>> entry) {
        try {
            merge(entry.getKey(), entry.getValue());
        } catch (Exception e) {
            context.logger().log(Level.WARNING, "Exif - face merge error " + doc.stid(), e);
        }

        callback.completed(entry.getKey());
    }
}
