package ru.yandex.ps.disk.search.reindex;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.AbstractMap;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;

import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.client.producer.ProducerClient;
import ru.yandex.concurrent.NamedThreadFactory;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.BadResponseException;
import ru.yandex.http.util.DoubleFutureCallback;
import ru.yandex.http.util.ErrorSuppressingFutureCallback;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.PassPayloadThroughFutureCallback;
import ru.yandex.http.util.RequestErrorType;
import ru.yandex.http.util.YandexHttpStatus;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumer;
import ru.yandex.http.util.nio.StatusCodeAsyncConsumerFactory;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumer;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.StringCollectorsFactory;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.ocr.proxy.CvStat;
import ru.yandex.ocr.proxy.OcrProxy;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.ps.disk.search.Cluster;
import ru.yandex.ps.disk.search.DbScanClusterizer;
import ru.yandex.ps.disk.search.Diface;
import ru.yandex.ps.disk.search.DiskDoc;
import ru.yandex.ps.disk.search.ExifInfoExtractor;
import ru.yandex.ps.disk.search.Face;
import ru.yandex.ps.disk.search.UserFacesAndClusters;
import ru.yandex.util.string.HexStrings;

public class ReindexHandler implements HttpAsyncRequestHandler<JsonObject> {
    //private static final String DEFAULT_BACKEND_QUERY = "mediatype:9";
    private static final String HASH_PREFIX = HexStrings.LOWER.toString(
        "face_reindex".getBytes(StandardCharsets.UTF_8));
    private static final int SELF_CLUSTERIZE_LIMIT = 13000;

    private final Diface server;
    private final DjfsFilter djfsFilter;
    private final Timer timer;
    private final ThreadPoolExecutor executor;
    private final ArrayBlockingQueue<Runnable> queue
        = new ArrayBlockingQueue<>(100);

    public ReindexHandler(final Diface server) {
        this.server = server;
        this.djfsFilter = new DjfsFilter(server);
        this.timer = new Timer("Reindex-RetryTimer", true);
        this.executor = new ThreadPoolExecutor(
            server.config().workers() -1,
            server.config().workers() -1,
            1,
            TimeUnit.HOURS,
            queue,
            new NamedThreadFactory(
                server.getThreadGroup(),
                "ReindexClusterizer",
                true));
        server.logger().info("Launching clusterizer threads " + executor.getMaximumPoolSize());
    }

    @Override
    public HttpAsyncRequestConsumer<JsonObject> processRequest(
        final HttpRequest request,
        final HttpContext context)
        throws HttpException
    {
        if (request instanceof HttpEntityEnclosingRequest) {
            HttpEntity entity =
                ((HttpEntityEnclosingRequest) request).getEntity();
            if (entity.getContentLength() != 0) {
                return new JsonAsyncTypesafeDomConsumer(
                    entity,
                    StringCollectorsFactory.INSTANCE,
                    BasicContainerFactory.INSTANCE);
            }
        }
        return new EmptyAsyncConsumer<>(JsonNull.INSTANCE);
    }

    protected void handleWithResource(
        final ReindexContext context,
        final JsonObject payload)
        throws HttpException, IOException
    {
        try {
            JsonList list = payload.asList();
            MergeCallback mergeCallback = new MergeCallback(context);
            List<DiskDoc> items = new ArrayList<>(list.size());
            int skippedNulResourceId = 0;
            int skippedBadSize = 0;
            for (JsonObject jo: list) {
                JsonMap item = jo.asMap();
                int width = item.getInt("width", -1);
                int height = item.getInt("height", -1);
                String stid = item.getString("stid", null);
                String previewStid = item.getString("preview_stid", stid);
                String resourceId = item.getString("resource_id", null);
                if (resourceId == null || previewStid == null) {
                    skippedNulResourceId += 1;
                    continue;
                }

                if (width < 0 || height < 0) {
                    skippedBadSize += 1;
                    continue;
                }

                String mimetype = item.getString("mimetype", null);
                int size = item.getInt("size", -1);
                //List<Face> stidFaces = Collections.emptyList();

                items.add(new DiskDoc(resourceId, stid, previewStid, width, height, mimetype, size));
                //context.logger().info(stid + " faces extracted " + stidFaces.size());
            }

            if (skippedBadSize > 0) {
                context.logger().info("Skipped bad size: " + skippedBadSize);
                context.stat().badCandidates(skippedBadSize);
            }

            if (skippedNulResourceId > 0) {
                context.logger().info("Skipped null resource Ids: " + skippedNulResourceId);
                context.stat().badCandidates(skippedNulResourceId);
            }

            if (context.batch() > 0) {
                context.logger().info("Launching batched " + context.batch());
                new BatchedCallback(context, items, mergeCallback).nextBatch();
            } else {
                context.logger().info("Launching non batched " + context.batch());
                MultiFutureCallback<List<Face>> mfcb = new MultiFutureCallback<>(mergeCallback);
                for (DiskDoc item: items) {
                    server.extractFaces(
                        context,
                        item,
                        new FaceExtractCallback(mfcb.newCallback(), context, item));
                }
                mfcb.done();
            }
        } catch (Exception e) {
            context.callback().failed(e);
        }
    }

    private class BatchedCallback
        extends AbstractProxySessionCallback<List<List<Face>>>
    {
        private final List<DiskDoc> items;
        private final List<List<Face>> faces;
        private final FutureCallback<List<List<Face>>> callback;
        private final ReindexContext context;
        private int position = 0;
        private int batchNum = 0;

        public BatchedCallback(
            final ReindexContext context,
            final List<DiskDoc> items,
            final FutureCallback<List<List<Face>>> callback)
        {
            super(context.session());
            this.items = items;
            this.callback = callback;
            this.context = context;
            this.faces = new ArrayList<>(items.size());
            if (items.size() == 0) {
                context.logger().info("Batches total 0");
            }
        }

        public void nextBatch() {
            if (position >= items.size()) {
                callback.completed(faces);
                return;
            }

            if (position == 0) {
                int batches = 1 + (items.size() - 1) / context.batch();
                context.logger().info("Batches total " + batches);
            }
            batchNum += 1;
            int nextPosition = position + Math.min(context.batch(), items.size() - position);
            List<DiskDoc> batch = items.subList(position, nextPosition);
            position = nextPosition;
            try {
                djfsFilter.filter(
                    context,
                    batch,
                    new DjfsFilterCallback(this, context));
            } catch (BadRequestException bre) {
                failed(bre);
            }
        }

        @Override
        public void completed(final List<List<Face>> lists) {
            synchronized (faces) {
                faces.addAll(lists);
            }

            context.logger().info("Batch finished " + batchNum);
            nextBatch();
        }
    }

    private class DjfsFilterCallback
        extends AbstractFilterFutureCallback<List<DiskDoc>, List<List<Face>>>
    {
        private final ReindexContext context;

        public DjfsFilterCallback(
            final FutureCallback<List<List<Face>>> callback,
            final ReindexContext context)
        {
            super(callback);
            this.context = context;
        }

        @Override
        public void completed(final List<DiskDoc> items) {
            if (items.size() == 0) {
                context.logger().warning("Djfs filtered all out");
            }
            MultiFutureCallback<List<Face>> mfcb = new MultiFutureCallback<>(callback);
            try {
                for (int i = 0; i < items.size(); i++) {
                    DiskDoc item = items.get(i);
                    if (!context.picasa() || (item.size() < 0) || !"image/jpeg".equalsIgnoreCase(item.mimetype())) {
                        server.extractFaces(
                            context,
                            item,
                            new FaceExtractCallback(mfcb.newCallback(), context, item));
                    } else {
                        DoubleFutureCallback<List<Face>, Map<String, JsonObject>> dfcb
                            = new DoubleFutureCallback<>(
                                new FaceExifMergeCallback(context, item, mfcb.newCallback()));

                        server.extractFaces(
                            context,
                            item,
                            new FaceExtractCallback(dfcb.first(), context, item));

                        HttpHost exifHost = new HttpHost("localhost", 90);
                        server.cokemulatorClient().execute(
                            exifHost,
                            new BasicAsyncRequestProducerGenerator(exifHost + "/exif?stid=" + item.stid()),
                            JsonAsyncTypesafeDomConsumerFactory.INSTANCE,
                            new AbstractFilterFutureCallback<>(
                                new ErrorSuppressingFutureCallback<>(
                                    dfcb.second(),
                                    Collections.emptyMap()))
                            {
                                @Override
                                public void completed(final JsonObject res) {
                                    try {
                                        callback.completed(res.asList().get(0).asMap());
                                    } catch (JsonException je) {
                                        failed(je);
                                    }
                                }
                            });
//                        server.extractExif(
//                            context,
//                            item.stid(),
//                            new ErrorSuppressingFutureCallback<>(
//                                dfcb.second(),
//                                Collections.emptyMap()));
                    }
                }
            } catch (BadRequestException bre) {
                failed(bre);
                return;
            }

            mfcb.done();
        }
    }

    private static class ClusterizeTask implements Runnable {
        private final ClusterizeCallback callback;
        private final List<Face> faces;
        private final ReindexContext context;

        public ClusterizeTask(
            final ClusterizeCallback callback,
            final List<Face> faces,
            final ReindexContext context)
        {
            this.callback = callback;
            this.faces = faces;
            this.context = context;
        }

        @Override
        public void run() {
            List<Cluster> clusters;
            try {
                context.logger().info("Starting clusterization");
                long version = context.version();
                long prefix = context.prefix().prefix();

                DbScanClusterizer clusterizer = new DbScanClusterizer(
                    context,
                    Long.toString(prefix) + '_' + Long.toString(version),
                    version);
                clusters = clusterizer.clusterize(faces);
                callback.completed(new AbstractMap.SimpleEntry<>(clusters, faces));
            } catch (Exception e) {
                callback.failed(e);
                return;
            }
        }
    }

    private class MergeCallback
        extends AbstractProxySessionCallback<List<List<Face>>>
    {
        private final ReindexContext context;

        public MergeCallback(final ReindexContext context) {
            super(context.session());
            this.context = context;
        }

        @Override
        public void completed(final List<List<Face>> facesResult) {

            //StringBuilder sb = new StringBuilder();
            List<Face> faces = new ArrayList<>(facesResult.size() * 2);
            for (List<Face> facesList: facesResult) {
                faces.addAll(facesList);
            }

            context.stat().faces(faces.size());
            context.logger().info("Total faces extracted " + faces.size() + " queue size " + queue.size());
            context.logger().info("Total extraction failed " + context.stat().extractFailures());

            if (context.length() > 0 && faces.size() > context.length()) {
                int newSize = Math.min(context.length(), faces.size());
                context.logger().info("Cutting down - to many faces " + (faces.size() - newSize));
                faces = new ArrayList<>(faces.subList(0, newSize));
            }

            ClusterizeTask task = new ClusterizeTask(new ClusterizeCallback(context), faces, context);

            if (queue.size() > 0 && faces.size() < SELF_CLUSTERIZE_LIMIT) {
                context.logger().info("Tasks in queue, we don't have much pohotos, try to clusterize self");
                task.run();
            } else {
                try {
                    executor.execute(task);
                } catch (RejectedExecutionException re) {
                    context.callback().failed(re);
                }
            }
        }
    }

    private class ClusterizeCallback
        extends AbstractProxySessionCallback<Map.Entry<List<Cluster>, List<Face>>>
    {
        private final ReindexContext context;

        public ClusterizeCallback(final ReindexContext context) {
            super(context.session());
            this.context = context;
        }

        @Override
        public void completed(final Map.Entry<List<Cluster>, List<Face>> pair) {
            List<Cluster> clusters = pair.getKey();
            List<Face> faces = pair.getValue();

            context.logger().info("Preclusters made " + clusters.size());
            try {
                Set<String> removedClusters = new LinkedHashSet<>();
                Iterator<Cluster> clusterIterator = clusters.iterator();
                while (clusterIterator.hasNext()) {
                    Cluster cluster = clusterIterator.next();
                    Set<String> resIds =
                        new LinkedHashSet<>(cluster.faces().size() << 1);
                    int sides = 0;
                    for (Face face : cluster.faces()) {
                        if (!resIds.add(face.resourceId())) {
                            face.cluster(null);
                            if (context.debug()) {
                                context.logger().info(
                                    "AfterDropping face: " + face.faceId());
                            }
                        }

                        if (face.side()) {
                            sides += 1;
                        }
                    }
                    if (cluster.name() == null && resIds.size() - sides < 10) {
                        //if (resIds.size() < 15) {
                        for (Face face : cluster.faces()) {
                            face.cluster(null);
                        }

                        context.logger().info("Dropping cluster: not enough faces " + resIds.size() + " " + cluster.id());
                        clusterIterator.remove();
                        removedClusters.add(cluster.id());
                        continue;
                    }
                }

                Map<String, Cluster> clustersMap =
                    new LinkedHashMap<>(clusters.size() << 1);
                for (Cluster cluster : clusters) {
                    clustersMap.put(cluster.id(), cluster);
                }

                int similarsDropped = 0;
                int similarsKept = 0;
                clusterIterator = clusters.iterator();
                while (clusterIterator.hasNext()) {
                    Cluster cluster = clusterIterator.next();
                    Map<String, AtomicInteger> map = new LinkedHashMap<>();
                    for (Face face : cluster.faces()) {
                        for (String clusterId : face.similarToClusters()) {
                            if (!clusterId.equals(cluster.id()) && !removedClusters.contains(clusterId)) {
                                map.computeIfAbsent(
                                    clusterId,
                                    (k) -> new AtomicInteger(0))
                                    .incrementAndGet();
                            }
                        }
                    }

                    if (!map.isEmpty()) {
                        context.logger().info(
                            "For cluster " + cluster.id()
                                + " Similars are " + map.toString());
                        for (Map.Entry<String, AtomicInteger> entry : map.entrySet()) {
                            Cluster otherCluster = clustersMap.get(entry.getKey());
                            if (entry.getValue().get() > 0.5 * cluster.faces().size()
                                && cluster.faces().size() < 150
                                && 5 * cluster.faces().size() < otherCluster.faces().size()) {
                                for (Face face : cluster.faces()) {
                                    face.cluster(null);
                                }
                                clusterIterator.remove();
                                context.logger().info(
                                    "Dropping cluster for similarity "
                                        + cluster.id()
                                        + " to " + otherCluster.id()
                                        + " size " + cluster.faces().size()
                                        + " " + cluster.faces().get(0).resourceId());
                                similarsDropped += 1;
                                break;
                            } else {
                                similarsKept += 1;
                            }
                        }
                    }
                }
                context.stat().similarsDropped(similarsDropped);
                context.stat().similarsDropped(similarsKept);
                context.stat().clustersCreated(clusters.size());
                UserFacesAndClusters fc = new UserFacesAndClusters(faces, clusters);
                if (context.action() == ReindexAction.PRINT) {
                    context.callback().completed(fc);
                } else {
                    BatchedIndexer indexer =
                        new BatchedIndexer(context, new PassPayloadThroughFutureCallback<>(fc, context.callback()),
                            clusters, faces);
                    indexer.sendNext();
                }
            } catch (Exception e) {
                failed(e);
            }
        }
    }


    private class BatchedIndexer
        extends AbstractProxySessionCallback<Object>
    {
        private final ReindexContext context;
        private final List<Cluster> clusters;
        private final List<Face> faces;
        private final ProducerClient client;
        private final FutureCallback<Object> reindexCallback;
        private final int batches;
        private int current = 0;
        private int batch = 0;

        public BatchedIndexer(
            final ReindexContext context,
            final FutureCallback<Object> reindexCallback,
            final List<Cluster> clusters,
            final List<Face> faces)
        {
            super(context.session());
            this.context = context;
            this.clusters = clusters;
            this.faces = faces;
            this.reindexCallback = reindexCallback;
            if (faces.size() > 0) {
                batches = 2 + (faces.size() - 1) / server.config().producerBatchSize();
            } else {
                batches = 1;
            }

            client =
                server.producerClient().adjust(
                    context.session().context());
        }

        @Override
        public void completed(final Object o) {
            sendNext();
        }

        public synchronized void sendNext() {
            try {
                if (batch == 0) {
                    sendCleanup();
                } else {
                    sendData();
                }
            } catch (IOException | BadRequestException e) {
                failed(e);
            }
        }

        public synchronized void sendCleanup() throws IOException, BadRequestException {
            QueryConstructor qc  = new QueryConstructor("/delete?face_reindex");
            qc.append("prefix", context.prefix().toStringFast());
            qc.append("service", server.config().faceIndexQueue());
            if (context.reextract()) {
                qc.append("text", "type:(face face_cluster face_delta)");
            } else {
                qc.append("text", "type:(face face_cluster face_delta)");
            }

            batch += 1;

            FutureCallback<Object> callback = this;
            if (faces.size() <= 0 && clusters.size() <= 0) {
                callback = reindexCallback;
                applyCallbacks(qc);
            }

            client.execute(
                server.config().producerClientConfig().host(),
                new BasicAsyncRequestProducerGenerator(qc.toString()),
                StatusCodeAsyncConsumerFactory.ANY_GOOD,
                session.listener().createContextGeneratorFor(client),
                callback);
        }

        private void applyCallbacks(final QueryConstructor qc) throws BadRequestException {
            if (context.config().faceReindexCallback() != null) {
                String callbackUri = context.config().faceReindexCallback().toString();
                StringBuilder callbackSb = new StringBuilder(callbackUri.length() + 40);
                callbackSb.append(callbackUri);
                callbackSb.append("&uid=");
                callbackSb.append(context.prefix().prefix());
                qc.append("callback", callbackSb.toString());
            }

            qc.copyIfPresent(context.session().params(), "callback");

            StringBuilder zooHash =
                new StringBuilder(HASH_PREFIX);
            zooHash.append(Long.toHexString(System.currentTimeMillis()));
            zooHash.append("00000");
            zooHash.append(Long.toHexString(context.prefix().prefix()));

            qc.append("zoo_hash", zooHash.toString());
        }

        public synchronized void sendData()
            throws IOException, BadRequestException
        {
            context.logger().info("Launching batch " + batch);
            QueryConstructor qc =
                new QueryConstructor("/modify?faces_reindex");
            StringBuilderWriter sbw = new StringBuilderWriter();
            FutureCallback<Object> callback = this;
            try (JsonWriter writer = JsonType.HUMAN_READABLE.create(sbw)) {
                writer.startObject();
                writer.key("prefix");
                writer.value(context.prefix());
                writer.key("docs");
                writer.startArray();

                int max = Math.min(
                    current + server.config().producerBatchSize(),
                    faces.size());
                for (; current < max; current++) {
                    writer.value(faces.get(current));
                }

                qc.append("prefix", context.prefix().toStringFast());
                qc.append("service", context.user().service());
                qc.append("batch", batch++);
                qc.append(
                    "batch_size",
                    server.config().producerBatchSize());
                qc.append("batches", batches);

                if (current >= faces.size()) {
                    for (Cluster cluster : clusters) {
                        writer.value(cluster);
                    }

                    // last batch
                    callback = reindexCallback;
                    applyCallbacks(qc);
                }

                writer.endArray();
                writer.endObject();
            }

            NStringEntity entity = new NStringEntity(
                sbw.toString(),
                ContentType.APPLICATION_JSON
                    .withCharset(context.session().acceptedCharset()));

            server.producerClient().execute(
                server.config().producerClientConfig().host(),
                new BasicAsyncRequestProducerGenerator(qc.toString(), entity),
                StatusCodeAsyncConsumerFactory.ANY_GOOD,
                context.session().listener().createContextGeneratorFor(client),
                callback);
        }
    }

    @Override
    public void handle(
        final JsonObject payload,
        final HttpAsyncExchange exchange,
        final HttpContext httpContext)
        throws HttpException, IOException
    {
        ProxySession session =
            new BasicProxySession(server, exchange, httpContext);
        ReindexContext context = new ReindexContext(server, session);

        if (payload == JsonNull.INSTANCE) {
            FutureCallback<JsonObject> callback;
            QueryConstructor qc = new QueryConstructor("/search-diface&reindex?");
            qc.append("prefix", session.params().getLong("prefix"));
            qc.append("service", "disk_queue");

            StringBuilder querySb = new StringBuilder();
            if (context.reextract()) {
                callback = new SearchCallback(context);
                querySb.append(
                    session.params().getString(
                        "query",
                        context.config().reindexQuery()));

                qc.append("text", querySb.toString());
                qc.append("dp", "fallback(etime,ctime,mtime date)");
                qc.append("sort", "date");
                if (context.length() > 0) {
                    qc.append("length", context.length());
                }

                qc.append("get", "preview_stid,stid,resource_id,id,name,width,height,mimetype,size");
            } else {
                callback = new IndexedFacesCallback(context, new MergeCallback(context));
                querySb.append("type:face");

                qc.append("text", querySb.toString());
                qc.append("get", "*,-face_cluster_id");
            }

            server.sequentialRequest(
                session,
                context,
                new BasicAsyncRequestProducerGenerator(qc.toString()),
                30000L,
                false,
                JsonAsyncTypesafeDomConsumerFactory.OK,
                session.listener().createContextGeneratorFor(context.searchClient()),
                callback);
        } else {
            handleWithResource(context, payload);
        }
    }

    private static final class IndexedFacesCallback
        extends AbstractFilterFutureCallback<JsonObject, List<List<Face>>>
    {
        private final ReindexContext context;

        public IndexedFacesCallback(
            final ReindexContext context,
            final FutureCallback<? super List<List<Face>>> callback)
        {
            super(callback);

            this.context = context;
        }

        @Override
        public void completed(final JsonObject dataObj) {
            try {
                JsonList facesObjList = dataObj.asMap().getList("hitsArray");
                List<List<Face>> result = new ArrayList<>(facesObjList.size());
                int skipped = 0;
                for (JsonObject jo: facesObjList) {
                    try {
                        Face face = Face.parseFromBackend(jo);
                        face = context.server().postprocessFace(face, context);
                        if (face != null) {
                            result.add(Collections.singletonList(face));
                        } else {
                            if (context.debug()) {
                                context.logger().info("Filter face on postprocess" + face.faceId());
                            }

                            skipped += 1;
                        }
                    } catch (JsonException je) {
                        context.logger().log(Level.WARNING, "Failed to parse " + JsonType.NORMAL.toString(jo), je);
                        failed(je);
                        return;
                    }
                }

                context.stat().candidates(result.size());
                context.logger().info("Faces from backend " + result.size() + " skipped " + skipped);

                callback.completed(result);
            } catch (JsonException je) {
                failed(je);
            }

        }
    }

    private final class SearchCallback
        extends AbstractFilterFutureCallback<JsonObject, UserFacesAndClusters>
    {
        private final ReindexContext context;

        public SearchCallback(final ReindexContext context) {
            super(context.callback());

            this.context = context;
        }

        @Override
        public void completed(final JsonObject resultObj) {
            try {
                JsonMap result = resultObj.asMap();
                JsonList list = result.getList("hitsArray");
                int hitsCount = result.getInt("hitsCount");
                context.logger().info("Got from backend " + list.size() + " total " + hitsCount);
                if (hitsCount > list.size()) {
                    context.logger().info("We cut down " + (hitsCount - list.size()) + " images");
                }
                context.stat().candidates(list.size());
                if (queue.size() > server.config().workers() / 2 && list.size() > SELF_CLUSTERIZE_LIMIT) {
                    context.session().response(
                        YandexHttpStatus.SC_TOO_MANY_REQUESTS,
                        "No resources for clusterization for " + list.size());
                    return;
                }
                handleWithResource(context, list);
            } catch (Exception je) {
                failed(je);
            }
        }
    }

    private final class FaceExtractCallback
        implements FutureCallback<List<Face>>
    {
        private final FutureCallback<List<Face>> callback;
        private final ReindexContext context;
        private final DiskDoc doc;
        private int retries = 0;

        public FaceExtractCallback(
            final FutureCallback<List<Face>> callback,
            final ReindexContext context,
            final DiskDoc doc)
        {
            this.callback = callback;
            this.context = context;
            this.doc = doc;
        }

        @Override
        public void cancelled() {
            callback.cancelled();
        }

        @Override
        public void completed(final List<Face> result) {
            callback.completed(result);
        }

        private void doRetry(final Exception e) {
            synchronized (this) {
                retries += 1;
            }

            long delay;
            if (retries > 10) {
                failedPermanent(e);
                return;
//                context.logger().log(
//                    Level.WARNING,
//                    "Retry " + retries + " on " + doc.stid() + " " + doc.resourceId(),
//                    e);
//                delay = (retries - 9) * context.rateLimitRetryDelay();
            } else {
                delay = (long) (new Random().nextFloat() * context.rateLimitRetryDelay());
            }

            timer.schedule(new RetryFaceTask(this), delay);
        }

        @Override
        public void failed(final Exception e) {
            CvStat stat = new CvStat();
            RequestErrorType errorType =
                RequestErrorType.ERROR_CLASSIFIER.apply(e);
            if (errorType == RequestErrorType.NON_RETRIABLE
                || errorType == RequestErrorType.HOST_NON_RETRIABLE)
            {
                if (OcrProxy.criticalNonRetriable(e, stat)) {
                    // Something very bad happended here. Rate limit or
                    // authorization error, let the caller decide what to do.
                    //callback.failed(e);
                    if (e instanceof BadResponseException
                        && ((BadResponseException) e).statusCode() == YandexHttpStatus.SC_TOO_MANY_REQUESTS)
                    {
                        doRetry(e);
                    } else {
                        failedPermanent(e);
                    }
                } else {
                    // This file is unprocessable, error type already accounted
                    // in stat, so save stat and reply 200 OK
                    context.server().stat(stat);
                    //session.response(HttpStatus.SC_OK);
                    context.logger().info(
                        "Stid extract failed, unprocessable, skipping "
                            + doc.stid() + " " + doc.resourceId() + " " + e.getMessage());
                    completed(Collections.emptyList());
                }
            } else {
                if (e instanceof BadResponseException) {
                    int statusCode = ((BadResponseException) e).statusCode();
                    if (statusCode == HttpStatus.SC_BAD_GATEWAY) {
                        doRetry(e);

                        return;
                    }

                    if (statusCode == HttpStatus.SC_INTERNAL_SERVER_ERROR && e.getMessage().contains("queue is full")) {
                        doRetry(e);
                        return;
                    }
                }
                context.logger().log(
                    Level.WARNING,
                    "Stid extract failed, skipping " + doc.stid() + " " + doc.resourceId(),
                    e);
                // Imageparser can't process image after all retries, skip it
                context.stat().extractFailures(1);
                context.server().stat(stat);
                completed(Collections.emptyList());
            }
            //completed(Collections.emptyList());
        }

        public FutureCallback<List<Face>> callback() {
            return callback;
        }

        public ReindexContext context() {
            return context;
        }

        public DiskDoc doc() {
            return doc;
        }

        public void failedPermanent(final Exception e) {
            context.logger().info("Failed permanently " + doc.resourceId() + " " + doc.stid());
            callback.failed(e);
        }
    }

    private class RetryFaceTask extends TimerTask {
        private final FaceExtractCallback callback;

        public RetryFaceTask(final FaceExtractCallback callback) {
            this.callback = callback;
        }

        @Override
        public void run() {
            try {
                server.extractFaces(
                    callback.context(),
                    callback.doc(),
                    callback);
            } catch (BadRequestException bre) {
                callback.failedPermanent(bre);
            }
        }
    }

}
