package ru.yandex.search.disk.indexer;

import java.io.IOException;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.StringEntity;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.DoubleFutureCallback;
import ru.yandex.http.util.FilterFutureCallback;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.StatusCodeAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.AsyncPostURIRequestProducerSupplier;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriterBase;
import ru.yandex.logger.PrefixedLogger;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.string.URIParser;
import ru.yandex.parser.uri.QueryConstructor;

public class CheckIndexHandler implements ProxyRequestHandler {
    private static final String MAIN_SERVICE = "disk_queue";
    private static final String REINDEX_SERVICE = "disk_queue_offline";
    private final DiskIndexer server;

    public CheckIndexHandler(final DiskIndexer server) throws ConfigException {
        this.server = server;
    }

    @Override
    public void handle(final ProxySession session) throws HttpException, IOException {
        CheckIndexContext context = new CheckIndexContext(session);
        FutureCallback<? super CheckResult> callback
            = new ResultPrinter(context);
        if (context.callback() != null) {
            callback = new CallbackLaunchCallback(callback, context);
        }

        DoubleFutureCallback<JsonObject, JsonObject> producerCb =
            new DoubleFutureCallback<>(
                new ProducerCallback(callback, context));
        AsyncClient producer = server.producerClient().adjust(session.context());
        producer.execute(
            server.config().producerConfig().host(),
            new BasicAsyncRequestProducerGenerator(
                server.config().producerConfig().host()
                    + "/_status?hr&service=" + MAIN_SERVICE + "&prefix=" + context.uid),
            JsonAsyncTypesafeDomConsumerFactory.OK,
            context.session().listener()
                .createContextGeneratorFor(producer),
            producerCb.first());

        producer.execute(
            server.config().producerConfig().host(),
            new BasicAsyncRequestProducerGenerator(
                server.config().producerConfig().host()
                    + "/_status?hr&service=" + REINDEX_SERVICE + "&prefix=" + context.uid),
            JsonAsyncTypesafeDomConsumerFactory.OK,
            context.session().listener()
                .createContextGeneratorFor(producer),
            producerCb.second());
    }

    private static class ProducerCheckResult implements JsonValue {
        private final Set<String> actualMain;
        private final Set<String> actualReindex;
        private final int actualMainNotReindexCnt;

        public ProducerCheckResult(
            final Set<String> actualMain,
            final Set<String> actualReindex) {
            this.actualMain = actualMain;
            this.actualReindex = actualReindex;
            Set<String> actualMainNotReindex = new LinkedHashSet<>(actualMain);
            actualMainNotReindex.removeAll(actualReindex);
            this.actualMainNotReindexCnt = actualMainNotReindex.size();
        }

        @Override
        public void writeValue(final JsonWriterBase writer)
            throws IOException
        {
            writer.startObject();
            writer.key("actual_main");
            writer.value(actualMain);
            writer.key("actual_reindex");
            writer.value(actualReindex);
            writer.key("actual_main_not_reindex");
            writer.value(actualMainNotReindexCnt);
            writer.endObject();
        }
    }

    private class ProducerCallback
        extends AbstractFilterFutureCallback<Map.Entry<JsonObject, JsonObject>, CheckResult>
    {
        private final CheckIndexContext context;

        public ProducerCallback(
            final FutureCallback<? super CheckResult> callback,
            final CheckIndexContext context)
        {
            super(callback);
            this.context = context;
        }

        private Map<String, Long> parseResponse(final JsonObject respObj) throws JsonException {
            Map<String, Long> result = new LinkedHashMap<>();
            for (JsonObject item: respObj.asList()) {
                JsonMap map = item.asMap();
                if (map.size() != 1) {
                    throw new JsonException("Invalid prodcuer response format " + JsonType.NORMAL.toString(respObj));
                }
                Map.Entry<String, JsonObject> entry = map.entrySet().iterator().next();
                String host = entry.getKey();
                Long value = entry.getValue().asLong();
                if (host.contains("gencfg")) {
                    continue;
                }
                result.put(host, value);
            }

            return result;
        }

        @Override
        public void completed(final Map.Entry<JsonObject, JsonObject> producerResp) {
            try {
                Map<String, Long> main = parseResponse(producerResp.getKey());
                Map<String, Long> reindex = parseResponse(producerResp.getValue());
                Long maxMain = main.values().stream().max(Long::compare).get();
                Long maxReindex = reindex.values().stream().max(Long::compare).get();
                Set<String> actualReindex = new LinkedHashSet<>();
                for (Map.Entry<String, Long> item: reindex.entrySet()) {
                    if (item.getValue() >= maxReindex) {
                        actualReindex.add(item.getKey());
                    }
                }
                Set<String> actualMain = new LinkedHashSet<>();
                for (Map.Entry<String, Long> item: main.entrySet()) {
                    if (item.getValue() >= maxMain) {
                        actualMain.add(item.getKey());
                    }
                }

                context.result().producer(new ProducerCheckResult(actualMain, actualReindex));
            } catch (JsonException je) {
                callback.failed(je);
                return;
            }

            context.session().logger().info(
                "Producer stat gathered " + JsonType.NORMAL.toString(context.result().producer()));
            IterableIndexChecker indexChecker =
                new IterableIndexChecker(
                    context,
                    new PhotosliceLaunchCallback(callback, context));
            indexChecker.next();
        }
    }

    private static class CheckResult implements JsonValue {
        private CheckIndexResult index;
        private PhotosliceCheckResult photoslice;
        private ProducerCheckResult producer;

        public synchronized CheckIndexResult index() {
            return index;
        }

        public synchronized void index(
            final CheckIndexResult index)
        {
            this.index = index;
        }

        public synchronized PhotosliceCheckResult photoslice() {
            return photoslice;
        }

        public synchronized void photoslice(
            final PhotosliceCheckResult photoslice)
        {
            this.photoslice = photoslice;
        }

        public synchronized ProducerCheckResult producer() {
            return producer;
        }

        public synchronized void producer(final ProducerCheckResult producer) {
            this.producer = producer;
        }

        @Override
        public synchronized void writeValue(final JsonWriterBase writer)
            throws IOException
        {
            writer.startObject();
            writer.key("index");
            writer.value(index);
            writer.key("photoslice");
            writer.value(photoslice);
            writer.key("producer");
            writer.value(producer);
            writer.endObject();
        }
    }

    public void checkPhotoslice(
        final CheckIndexContext context,
        final FutureCallback<? super CheckResult> fcb)
    {
        if (server.searchProxyClient() == null) {
            fcb.failed(new BadRequestException("No search client configured"));
            return;
        }

        context.session().logger().info("Checking photoslice for user");
        AsyncClient client = server.searchProxyClient().adjust(context.session().context());

        try {
            QueryConstructor qc = new QueryConstructor(
                "/clusterize?disk_indexer");
            qc.append("uid", context.uid());
            qc.append(
                "get",
                "id,key,date,mediatype");
            qc.append("online", "false");
            String nfmUri = qc.toString();
            qc.append("fast-moved", "true");
            PhotosliceMergeCallback mergeCallback = new PhotosliceMergeCallback(fcb, context);

            client.execute(
                server.config().searchProxy().host(),
                new BasicAsyncRequestProducerGenerator(qc.toString()),
                JsonAsyncTypesafeDomConsumerFactory.OK,
                context.session().listener()
                    .createContextGeneratorFor(client),
                new PhotosliceCallback(
                    mergeCallback, context, nfmUri));
        } catch (BadRequestException bre) {
            fcb.failed(bre);
            return;
        }
    }

    private static class PhotosliceStat implements JsonValue {
        private int total;
        private int images;
        private int videos;
        private int totalLastMonth;
        private int imageLastMonth;
        private int videoLastMonth;
        private final long lastMonth =
            System.currentTimeMillis() / 1000 - TimeUnit.DAYS.toSeconds(31);

        public PhotosliceStat() {
        }

        public void accept(final JsonMap jo) throws JsonException {
            int mediatype = jo.getInt("mediatype");
            boolean image = mediatype == 9;
            boolean video = mediatype == 12;
            long date = jo.getLong("date");
            total += 1;

            if (image) {
                images += 1;
            }
            if (video) {
                videos += 1;
            }

            if (lastMonth <= date) {
                if (image) {
                    imageLastMonth += 1;
                }
                if (video) {
                    videoLastMonth += 1;
                }

                totalLastMonth +=1;
            }
        }

        public int total() {
            return total;
        }

        public int images() {
            return images;
        }

        public int videos() {
            return videos;
        }

        public int totalLastMonth() {
            return totalLastMonth;
        }

        public int imageLastMonth() {
            return imageLastMonth;
        }

        public int videoLastMonth() {
            return videoLastMonth;
        }

        @Override
        public void writeValue(final JsonWriterBase writer)
            throws IOException
        {
            writer.startObject();
            writer.key("total");
            writer.value(total);
            writer.key("images");
            writer.value(images);
            writer.key("videos");
            writer.value(videos);
            writer.key("total_last_month");
            writer.value(totalLastMonth);
            writer.key("images_last_month");
            writer.value(imageLastMonth);
            writer.key("videos_last_month");
            writer.value(videoLastMonth);
            writer.endObject();
        }

        @Override
        public String toString() {
            return "PhotosliceStat{" +
                       "total=" + total +
                       ", images=" + images +
                       ", videos=" + videos +
                       ", totalLastMonth=" + totalLastMonth +
                       ", imageLastMonth=" + imageLastMonth +
                       ", videoLastMonth=" + videoLastMonth +
                       ", lastMonth=" + lastMonth +
                       '}';
        }
    }

    private class CallbackLaunchCallback extends FilterFutureCallback<CheckResult> {
        private final CheckIndexContext context;

        public CallbackLaunchCallback(
            final FutureCallback<? super CheckResult> callback,
            final CheckIndexContext context)
        {
            super(callback);
            this.context = context;
        }

        @Override
        public void completed(final CheckResult entry) {
            AsyncClient cbClient =
                server.callbacksClient().adjust(context.session().context());
            try {
                cbClient.execute(
                    new AsyncPostURIRequestProducerSupplier(
                        context.callback(),
                        new StringEntity(
                            JsonType.NORMAL.toString(entry),
                            StandardCharsets.UTF_8)),
                    StatusCodeAsyncConsumerFactory.ANY_GOOD,
                    context.session().listener()
                        .createContextGeneratorFor(cbClient),
                    new AbstractFilterFutureCallback<Object, CheckResult>(
                        callback)
                    {
                        @Override
                        public void completed(Object o) {
                            callback.completed(entry);
                        }
                    });
            } catch (IOException e) {
                callback.failed(e);
            }
        }
    }

    private class PhotosliceCallback extends AbstractFilterFutureCallback<JsonObject, List<PhotosliceStat>> {
        private final String nextUri;
        private final PhotosliceStat result;
        private final CheckIndexContext context;

        public PhotosliceCallback(
            final FutureCallback<? super List<PhotosliceStat>> callback,
            final CheckIndexContext context,
            final String nextUri)
        {
            super(callback);
            this.nextUri = nextUri;
            this.context = context;
            this.result = null;
        }

        public PhotosliceCallback(
            final FutureCallback<? super List<PhotosliceStat>> callback,
            final CheckIndexContext context,
            final PhotosliceStat result)
        {
            super(callback);
            this.result = result;
            this.context = context;
            this.nextUri = null;
        }

        @Override
        public void completed(final JsonObject resultObj) {
            PhotosliceStat stat = new PhotosliceStat();
            try {
                for (JsonObject hitItem: resultObj.asMap().getList("hitsArray")) {
                    for (JsonObject doc: hitItem.asMap().getList("merged_docs")) {
                        stat.accept(doc.asMap());
                    }
                }
            } catch (JsonException je) {
                failed(je);
                return;
            }

            if (nextUri != null) {
                context.session().logger().info("photoslice fetch finished, launching next " + JsonType.NORMAL.toString(stat));
                AsyncClient client = server.searchProxyClient().adjust(context.session().context());
                client.execute(
                    server.config().searchProxy().host(),
                    new BasicAsyncRequestProducerGenerator(nextUri),
                    JsonAsyncTypesafeDomConsumerFactory.OK,
                    context.session().listener()
                        .createContextGeneratorFor(client),
                    new PhotosliceCallback(callback, context, stat));
            } else {
                context.session().logger().info("photoslice finished, last stat" + JsonType.NORMAL.toString(stat));
                callback.completed(Arrays.asList(result, stat));
            }
        }
    }

    private static class PhotosliceMergeCallback
        extends AbstractFilterFutureCallback<List<PhotosliceStat>, CheckResult>
    {
        private final CheckIndexContext context;

        public PhotosliceMergeCallback(
            final FutureCallback<? super CheckResult> callback,
            final CheckIndexContext context)
        {
            super(callback);

            this.context = context;
        }

        @Override
        public void completed(final List<PhotosliceStat> stats) {
            context.result().photoslice(new PhotosliceCheckResult(stats.get(0), stats.get(1)));
            callback.completed(context.result());
        }
    }

    private static class PhotosliceCheckResult implements JsonValue {
        private final PhotosliceStat fmStat;
        private final PhotosliceStat nfmStat;

        public PhotosliceCheckResult(
            final PhotosliceStat fmStat,
            final PhotosliceStat nfmStat)
        {
            this.fmStat = fmStat;
            this.nfmStat = nfmStat;
        }

        @Override
        public void writeValue(final JsonWriterBase writer)
            throws IOException
        {
            writer.startObject();
            writer.key("fast_move_and_not_coincide");
            writer.value(fmStat.total() == nfmStat.total());
            writer.key("fast_moved");
            writer.value(fmStat);
            writer.key("not_fast_moved");
            writer.value(nfmStat);
            writer.endObject();
        }
    }

    private class PhotosliceLaunchCallback
        extends AbstractFilterFutureCallback<CheckIndexResult, CheckResult>
    {
        private final CheckIndexContext context;

        public PhotosliceLaunchCallback(
            final FutureCallback<? super CheckResult> callback,
            final CheckIndexContext context)
        {
            super(callback);
            this.context = context;
        }

        @Override
        public void completed(final CheckIndexResult checkResult) {
            context.result().index(checkResult);
            if (context.checkPhotoslice()) {
                checkPhotoslice(context, callback);
            } else {
                callback.completed(context.result());
            }
        }
    }
    private static class ResultPrinter extends AbstractProxySessionCallback<CheckResult> {
        public ResultPrinter(final CheckIndexContext context) {
            super(context.session());
        }

        @Override
        public void completed(final CheckResult entry) {
            session.response(HttpStatus.SC_OK, JsonType.NORMAL.toString(entry));
        }
    }

    private class IterableIndexChecker extends AbstractProxySessionCallback<DiskSnapshot> {
        private final CheckIndexContext context;
        //private final List<DiskDocumentMeta> folders = new ArrayList<>();
        private final Map<String, DiskDocumentMeta> resources = new LinkedHashMap<>();
        private final Map<String, Collection<DiskDocumentMeta>> duplicateFolders
            = new LinkedHashMap<>();
        private final List<DiskDocumentMeta> notFound = new ArrayList<>();
        private final FutureCallback<CheckIndexResult> callback;
        private final AtomicInteger filesNotTrash = new AtomicInteger(0);
        private final AtomicInteger foldersNotTrash = new AtomicInteger(0);
        private final AtomicInteger filesTrash = new AtomicInteger(0);
        private final AtomicInteger foldersTrash = new AtomicInteger(0);
        private final AtomicInteger imagesNotTrash = new AtomicInteger(0);
        private final AtomicInteger videosNotTrash = new AtomicInteger(0);

        private volatile String iterationKey = null;

        public IterableIndexChecker(
            final CheckIndexContext context,
            final FutureCallback<CheckIndexResult> callback)
        {
            super(context.session());
            this.context = context;
            this.callback = callback;
        }

        public void next() {
            context.session().logger()
                .info("Requesting data for iteration_key " + iterationKey);
            server.mpfsClient().execute(
                server.mpfsHost(),
                new BasicAsyncRequestProducerGenerator(
                    server.mpfsUri() + context.uid(),
                    JsonType.NORMAL.toString(
                        Collections.singletonMap(
                            "iteration_key",
                            iterationKey)),
                    server.mpfsContentType()),
                DiskSnapshotConsumerFactory.OK_WITH_PATH,
                context.session().listener()
                    .createContextGeneratorFor(server.mpfsClient()),
                this);
        }

        private void finish() {
            context.session().logger()
                .info("Finishing " + duplicateFolders);

            callback.completed(
                new CheckIndexResult(
                    context,
                    notFound,
                    duplicateFolders,
                    foldersNotTrash.get(),
                    filesNotTrash.get(),
                    imagesNotTrash.get(),
                    videosNotTrash.get(),
                    filesTrash.get(),
                    foldersTrash.get()));
        }

        private void statDoc(final DiskDocumentMeta meta) {
            if (!meta.path().startsWith("/") || meta.path().length() <= 1) {
                return;
            }

            if (meta.path().startsWith("/trash")) {
                if (meta.docType() == DocType.DIR) {
                    foldersTrash.incrementAndGet();
                } else {
                    filesTrash.incrementAndGet();
                }
            } else {
                if (meta.docType() == DocType.DIR) {
                    foldersNotTrash.incrementAndGet();
                } else {
                    filesNotTrash.incrementAndGet();
                    if ("image".equalsIgnoreCase(meta.mediatype())) {
                        imagesNotTrash.incrementAndGet();
                    } else if ("video".equalsIgnoreCase(meta.mediatype())) {
                        videosNotTrash.incrementAndGet();
                    }
                }
            }
        }
        @Override
        public void completed(final DiskSnapshot snapshot) {
            for (DiskDocumentMeta doc: snapshot.docs()) {
                statDoc(doc);
                if (context.dirOnly() && DocType.DIR != doc.docType()) {
                    continue;
                }
                DiskDocumentMeta old = resources.put(doc.resourceId(), doc);
                if (old != null) {
                    Collection<DiskDocumentMeta> col =
                        duplicateFolders.computeIfAbsent(
                            doc.resourceId(),
                            (k) -> new LinkedHashSet<>());

                    col.add(old);
                    col.add(doc);
                }
            }

            context.session().logger()
                .info("Processing finished for iteration_key "
                    + iterationKey + " next " + snapshot.iterationKey());
            iterationKey = snapshot.iterationKey();

            if (context.compare() && snapshot.docs().size() > 0) {
                AsyncClient client = server.searchProxyClient().adjust(session.context());

                Map<String, DiskDocumentMeta> requested = new LinkedHashMap<>();
                QueryConstructor qc = new QueryConstructor("/get?disk_indexer");
                try {
                    qc.append("uid", context.uid());
                    qc.append("build_path", "true");
                    qc.append("get", "key,resource_id");
                    for (DiskDocumentMeta doc: snapshot.docs()) {
                        if (context.dirOnly() && DocType.DIR != doc.docType()) {
                            continue;
                        }
                        qc.append("id", doc.fileId());
                        requested.put(doc.resourceId(), doc);
                    }
                } catch (BadRequestException bre) {
                    failed(bre);
                    return;
                }

                if (requested.size() > 0) {
                    client.execute(
                        server.config().searchProxy().host(),
                        new BasicAsyncRequestProducerGenerator(qc.toString()),
                        JsonAsyncTypesafeDomConsumerFactory.OK,
                        context.session().listener()
                            .createContextGeneratorFor(client),
                        new GetIndexedResourcesCallback(context, requested,this));
                    return;
                }
            }

            if (iterationKey == null) {
                finish();
            } else {
                next();
            }
        }

        private synchronized void indexCheckFinished(final Map<String, DiskDocumentMeta> notFound) {
            this.notFound.addAll(notFound.values());

            if (iterationKey == null) {
                finish();
            } else {
                next();
            }
        }
    }

    private static class GetIndexedResourcesCallback
        extends AbstractProxySessionCallback<JsonObject>
    {
        private final Map<String, DiskDocumentMeta> requested;
        private final IterableIndexChecker checker;

        public GetIndexedResourcesCallback(
            final CheckIndexContext context,
            final Map<String, DiskDocumentMeta> requested,
            final IterableIndexChecker checker)
        {
            super(context.session());
            this.requested = requested;
            this.checker = checker;
        }

        @Override
        public void completed(final JsonObject indexed) {
            try {
                JsonList list = indexed.asMap().getList("hitsArray");
                for (JsonObject item: list) {
                    JsonMap itemMap = item.asMap();
                    //String key = itemMap.getString("key");
                    String resId = itemMap.getString("resource_id");
                    requested.remove(resId);
                }

                Iterator<Map.Entry<String, DiskDocumentMeta>> iterator = requested.entrySet().iterator();
                while (iterator.hasNext()) {
                    Map.Entry<String, DiskDocumentMeta> entry = iterator.next();
                    String path = entry.getValue().path();
                    if (path.indexOf('/') == path.lastIndexOf('/')) {
                        session.logger().info("Skipping not founding " + path);
                        iterator.remove();
                    }
                }
                checker.indexCheckFinished(requested);
            } catch (JsonException je) {
                failed(je);
            }
        }
    }


    private static class CheckIndexResult implements JsonValue {
        private final CheckIndexContext context;
        private final Map<String, Collection<DiskDocumentMeta>> duplicated;
        private final List<DiskDocumentMeta> notFound;
        private final boolean statusOk;
        private final int foldersNotTrash;
        private final int filesNotTrash;
        private final int imagesNotTrash;
        private final int videosNotTrash;
        private final int filesTrash;
        private final int foldersTrash;

        public CheckIndexResult(
            final CheckIndexContext context,
            final List<DiskDocumentMeta> notFound,
            final Map<String, Collection<DiskDocumentMeta>> duplicated,
            final int foldersNotTrash,
            final int filesNotTrash,
            final int imagesNotTrash,
            final int videosNotTrash,
            final int filesTrash,
            final int foldersTrash)
        {
            this.context = context;
            this.duplicated = duplicated;
            this.notFound = notFound;
            this.statusOk = duplicated.isEmpty() && notFound.isEmpty();
            this.foldersNotTrash = foldersNotTrash;
            this.filesNotTrash = filesNotTrash;
            this.imagesNotTrash = imagesNotTrash;
            this.videosNotTrash = videosNotTrash;
            this.filesTrash = filesTrash;
            this.foldersTrash = foldersTrash;
        }

        @Override
        public void writeValue(final JsonWriterBase writer)
            throws IOException
        {
            writer.startObject();
            writer.key("uid");
            writer.value(context.uid());
            writer.key("ok");
            writer.value(statusOk);
            writer.key("folders_not_trash");
            writer.value(foldersNotTrash);
            writer.key("folders_trash");
            writer.value(foldersTrash);

            writer.key("files_not_trash");
            writer.value(filesNotTrash);
            writer.key("files_trash");
            writer.value(filesTrash);

            writer.key("images_not_trash");
            writer.value(imagesNotTrash);
            writer.key("videos_not_trash");
            writer.value(videosNotTrash);

            writer.key("notFound");
            writer.startArray();
            for (DiskDocumentMeta meta: notFound) {
                writer.startObject();
                writer.key("type");
                writer.value(meta.docType().name().toLowerCase(Locale.ENGLISH));
                writer.key("path");
                writer.value(meta.path());
                writer.key("resource_id");
                writer.value(meta.resourceId());
                writer.endObject();
            }
            writer.endArray();
            writer.key("duplicates");
            writer.startArray();
            for (Map.Entry<String, Collection<DiskDocumentMeta>> item: duplicated.entrySet()) {
                writer.startObject();
                writer.key("resource_id");
                writer.value(item.getKey());
                writer.key("docs");
                writer.startArray();
                for (DiskDocumentMeta meta: item.getValue()) {
                    writer.startObject();
                    writer.key("type");
                    writer.value(meta.docType().name().toLowerCase(Locale.ENGLISH));
                    writer.key("path");
                    writer.value(meta.path());
                    writer.endObject();
                }
                writer.endArray();
                writer.endObject();
            }

            writer.endArray();
            writer.endObject();
        }
    }


    private static class CheckIndexContext {
        private final Long uid;
        private final URI callback;
        private final String ticket;
        private final boolean compare;
        private final boolean dirOnly;
        private final boolean checkPhotoslice;
        private final ProxySession session;
        private final PrefixedLogger logger;
        private final CheckResult result;

        public CheckIndexContext(final ProxySession session) throws BadRequestException  {
            uid = session.params().getLong("uid");
            callback = session.params().get("callback", null, URIParser.FRAGMENT_STRIPPING);
            ticket = session.params().getString("ticket", null);
            compare = session.params().getBoolean("compare", false);
            dirOnly = session.params().getBoolean("dir_only", true);
            checkPhotoslice = session.params().getBoolean("check_photoslice", true);
            this.session = session;
            this.logger = session.logger();
            this.result = new CheckResult();
        }

        public boolean dirOnly() {
            return dirOnly;
        }

        public boolean checkPhotoslice() {
            return checkPhotoslice;
        }

        public String ticket() {
            return ticket;
        }

        public ProxySession session() {
            return session;
        }

        public PrefixedLogger logger() {
            return logger;
        }

        public Long uid() {
            return uid;
        }

        public URI callback() {
            return callback;
        }

        public boolean compare() {
            return compare;
        }

        public CheckResult result() {
            return result;
        }
    }
}
