package ru.yandex.search.disk.indexer;

import java.io.Serializable;

import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonType;

public class DiskDocumentMeta implements Serializable {
    private static final long serialVersionUID = 0L;

    private final String resourceId;
    private final String fileId;
    private final String path;
    private final long ownerUid;
    private final long version;
    private final String mediatype;
    private final DocType docType;

    public DiskDocumentMeta(final JsonObject doc)
        throws JsonException
    {
        this(doc, false);
    }

    public DiskDocumentMeta(final JsonObject doc, final boolean parsePath)
        throws JsonException
    {
        try {
            JsonMap docMap = doc.asMap();
            JsonMap meta = docMap.get("meta").asMap();
            resourceId = meta.getString("resource_id");
            int colon = resourceId.indexOf(':');
            if (colon == -1 || colon == resourceId.length() - 1) {
                throw new JsonException("Malformed resource_id " + resourceId);
            }
            fileId = resourceId.substring(colon + 1);
            try {
                ownerUid = Long.parseLong(resourceId.substring(0, colon));
            } catch (NumberFormatException e) {
                throw new JsonException(
                    "Malformed owner uid in resource_id " + resourceId,
                    e);
            }
            version = meta.getLong("revision");
            mediatype = meta.getString("media_type", null);
            docType = docMap.getEnum(DocType.class, "type");
            if (parsePath) {
                path = docMap.getString("path");
            } else {
                path = null;
            }
        } catch (JsonException e) {
            throw new JsonException(
                "Failed to parse document: "
                + JsonType.HUMAN_READABLE.toString(doc),
                e);
        }
    }

    public DocType docType() {
        return docType;
    }

    public String resourceId() {
        return resourceId;
    }

    public String fileId() {
        return fileId;
    }

    public long ownerUid() {
        return ownerUid;
    }

    public long version() {
        return version;
    }

    public String mediatype() {
        return mediatype;
    }

    public String path() {
        return path;
    }
}

