package ru.yandex.search.disk.indexer;

import java.io.IOException;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import org.apache.http.HttpHost;
import org.apache.http.HttpStatus;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;

import ru.yandex.collection.Pattern;
import ru.yandex.http.config.HttpTargetConfigBuilder;
import ru.yandex.http.config.ImmutableURIConfig;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.BasicProxySessionCallback;
import ru.yandex.http.proxy.HttpProxy;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.DoubleFutureCallback;
import ru.yandex.http.util.FilterFutureCallback;
import ru.yandex.http.util.MultiFutureCallback;
import ru.yandex.http.util.ServerException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.HttpAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.StatusCodeAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.AsyncPostURIRequestProducerSupplier;
import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.parser.uri.QueryPostProcessor;

public class DiskIndexer
    extends HttpProxy<ImmutableDiskIndexerConfig>
    implements ProxyRequestHandler
{
    private final AsyncClient mpfsClient;
    private final ContentType mpfsContentType;
    private final HttpHost mpfsHost;
    private final String mpfsUri;
    private final AsyncClient producerClient;
    private final AsyncClient callbacksClient;
    private final AsyncClient searchProxyClient;
    private final HttpHost producerHost;

    public DiskIndexer(final ImmutableDiskIndexerConfig config)
        throws IOException
    {
        super(config);

        ImmutableURIConfig mpfsConfig = config.mpfsConfig();
        mpfsClient = client("MPFS", mpfsConfig);
        mpfsContentType = ContentType.APPLICATION_JSON.withCharset(
            mpfsClient.requestCharset());
        mpfsHost = mpfsConfig.host();
        mpfsUri =
            mpfsConfig.request() + mpfsConfig.firstCgiSeparator() + "uid=";
        if (config.searchProxy() != null) {
            searchProxyClient = client("SearchProxyClient", config.searchProxy());
        } else {
            searchProxyClient = null;
        }
        producerClient = client("Producer", config.producerConfig());
        producerHost = config.producerConfig().host();
        register(
            new Pattern<>("/reindex", false),
            this,
            RequestHandlerMapper.GET);
        try {
            callbacksClient = client(
                "CallbackClient",
                new HttpTargetConfigBuilder().connections(10).timeout((int) TimeUnit.SECONDS.toMillis(60)).build());

            register(
                new Pattern<>("/check_index", false),
                new CheckIndexHandler(this),
                RequestHandlerMapper.GET);
        } catch (ConfigException ce) {
            throw new IOException(ce);
        }
    }

    public AsyncClient producerClient() {
        return producerClient;
    }

    public AsyncClient searchProxyClient() {
        return searchProxyClient;
    }

    public AsyncClient callbacksClient() {
        return callbacksClient;
    }

    public AsyncClient mpfsClient() {
        return mpfsClient;
    }

    public ContentType mpfsContentType() {
        return mpfsContentType;
    }

    public HttpHost mpfsHost() {
        return mpfsHost;
    }

    public String mpfsUri() {
        return mpfsUri;
    }

    @Override
    public void handle(final ProxySession session) throws BadRequestException {
        handle(new DiskSnapshotCallback(session), null);
    }

    private void handle(
        final DiskSnapshotCallback callback,
        final String iterationKey)
    {
        callback.context().session().logger()
            .info("Requesting data for iteration_key " + iterationKey);

        HttpAsyncResponseConsumerFactory<DiskSnapshot> consumerFactory
            = DiskSnapshotConsumerFactory.OK;

        if (callback.context().checkDuplicates()) {
            consumerFactory = DiskSnapshotConsumerFactory.OK_WITH_PATH;
        }
        mpfsClient.execute(
            mpfsHost,
            new BasicAsyncRequestProducerGenerator(
                mpfsUri + callback.context().uid(),
                JsonType.NORMAL.toString(
                    Collections.singletonMap(
                        "iteration_key",
                        iterationKey)),
                mpfsContentType),
            consumerFactory,
            callback.context().session().listener()
                .createContextGeneratorFor(mpfsClient),
            callback);
    }

    private class DiskSnapshotCallback
        extends AbstractProxySessionCallback<DiskSnapshot>
    {
        private final List<DiskDocumentMeta> docs = new ArrayList<>();
        private final DiskReindexContext context;
        private final AsyncClient producerClient;
        private final Supplier<? extends HttpClientContext> contextGenerator;
        private long revision = -1;

        DiskSnapshotCallback(final ProxySession session)
            throws BadRequestException
        {
            super(session);
            context = new DiskReindexContext(session);
            producerClient =
                DiskIndexer.this.producerClient.adjust(session.context());
            contextGenerator =
                session.listener().createContextGeneratorFor(producerClient);
        }

        @Override
        public void failed(final Exception e) {
            if (e instanceof ServerException
                && ((ServerException) e).statusCode()
                    == HttpStatus.SC_FORBIDDEN)
            {
                session.logger().warning(
                    "403 Forbidden received from MPFS, skipping user");
                super.failed(new BadRequestException(e));
            } else {
                super.failed(e);
            }
        }

        public DiskReindexContext context() {
            return context;
        }

        private void sendBatch(
            final DiskIndexerBatch batch,
            final FutureCallback<Object> callback)
            throws BadRequestException, IOException
        {
            BasicAsyncRequestProducerGenerator producerGenerator =
                batch.createRequest(context);
            producerGenerator.addHeader(
                YandexHeaders.SERVICE,
                context.serviceName());
            producerClient.execute(
                producerHost,
                producerGenerator,
                EmptyAsyncConsumerFactory.OK,
                contextGenerator,
                callback);
        }

        private void indexDocs() throws BadRequestException, IOException {
            FutureCallback<Object> callback =
                new BasicProxySessionCallback(session);

            if (context.checkDuplicates()) {
                Map<String, DiskDocumentMeta> resIds = new LinkedHashMap<>(docs.size());
                Map<String, Set<DiskDocumentMeta>> dups = new LinkedHashMap<>(docs.size());

                //boolean found = false;
                for (DiskDocumentMeta meta: docs) {
                    DiskDocumentMeta old = resIds.put(meta.resourceId(), meta);
                    if (old != null) {
                        Set<DiskDocumentMeta> set =
                            dups.computeIfAbsent(meta.resourceId(), (k) -> new LinkedHashSet<>());
                        set.add(old);
                        set.add(meta);
                        //found = true;
                    }
                }

                StringBuilderWriter sbw = new StringBuilderWriter();
                try (JsonWriter writer = JsonType.NORMAL.create(sbw)) {
                    writer.startObject();
                    writer.key("duplicates");
                    writer.startArray();
                    for (Map.Entry<String, Set<DiskDocumentMeta>> item: dups.entrySet()) {
                        writer.startObject();
                        writer.key("resource_id");
                        writer.value(item.getKey());
                        writer.key("docs");
                        writer.startArray();
                        for (DiskDocumentMeta meta: item.getValue()) {
                            writer.startObject();
                            writer.key("type");
                            writer.value(meta.docType().name().toLowerCase(Locale.ENGLISH));
                            writer.key("path");
                            writer.value(meta.path());
                            writer.endObject();
                        }
                        writer.endArray();
                        writer.endObject();
                    }

                    writer.endArray();
                    writer.endObject();
                } catch (IOException e) {
                    failed(e);
                    return;
                }

                DoubleFutureCallback<Object, Object> dfcb = new DoubleFutureCallback<>(callback);
                AsyncClient cbClient = callbacksClient.adjust(session.context());
                cbClient.execute(
                    new AsyncPostURIRequestProducerSupplier(
                        context.checkDuplicatesCallback(),
                        new StringEntity(
                            sbw.toString(),
                            StandardCharsets.UTF_8)),
                    StatusCodeAsyncConsumerFactory.ANY_GOOD,
                    context.session().listener()
                        .createContextGeneratorFor(cbClient),
                    dfcb.first());
                callback = dfcb.second();
            }
            int size = docs.size();
            if (size == 0) {
                callback.completed(null);
                return;
            }
            PrefixAppender prefixAppender =
                new PrefixAppender(context.uid(), revision);
            if (context.cleanup()) {
                docs.sort(
                    (lhs, rhs)
                        -> lhs.resourceId().compareTo(rhs.resourceId()));
                DiskDocumentMeta firstDoc = docs.get(0);
                DiskDocumentMeta lastDoc = docs.get(size - 1);
                QueryPostProcessor queryPostProcessor =
                    prefixAppender.andThen(new CleanupTypeAppender("outer"));
                URI callbackUri = context.callback();
                if (callbackUri != null) {
                    queryPostProcessor = queryPostProcessor.andThen(
                        new CallbackAppender(callbackUri));
                }
                DiskIndexerBatch batch = new DiskIndexerSingleRequestBatch(
                    new DiskIndexerMultiDocBatchEntryFactory(2),
                    queryPostProcessor);
                batch.add(firstDoc);
                batch.add(lastDoc);
                callback = new OuterCleanupCallback(callback, session, batch);
            }
            QueryPostProcessor queryPostProcessor;
            if (context.cleanup()) {
                queryPostProcessor =
                    prefixAppender.andThen(new CleanupTypeAppender("inner"));
            } else {
                queryPostProcessor = prefixAppender;
            }
            MultiFutureCallback<Object> producerCallback =
                new MultiFutureCallback<>(callback);
            DiskIndexerBatchFactory batchFactory =
                new DiskIndexerBatchFactory(queryPostProcessor, context);
            DiskIndexerBatch batch = batchFactory.createBatch();
            int batches = 0;
            for (int i = 0; i < size; ++i) {
                DiskDocumentMeta doc = docs.get(i);
                docs.set(i, null);
                if (batch.isFull()) {
                    sendBatch(batch, producerCallback.newCallback());
                    batch = batchFactory.createBatch();
                    ++batches;
                }
                batch.add(doc);
            }
            if (!batch.isEmpty()) {
                sendBatch(batch, producerCallback.newCallback());
                ++batches;
            }
            producerCallback.done();
            session.logger().info(
                batches + " batches with " + size + " docs sent to producer");
        }

        @Override
        public void completed(final DiskSnapshot snapshot) {
            for (DiskDocumentMeta doc: snapshot.docs()) {
                if (context.mediatypeFilter().test(doc.mediatype())) {
                    docs.add(doc);
                }
            }
            if (revision == -1) {
                revision = snapshot.revision();
            }
            String iterationKey = snapshot.iterationKey();
            session.logger().info(
                "Response received with iteration_key " + iterationKey
                + ", docs received: " + snapshot.docs().size()
                + ", total docs accumulated: " + docs.size());
            if (iterationKey == null) {
                try {
                    indexDocs();
                } catch (BadRequestException | IOException e) {
                    failed(e);
                }
            } else {
                handle(this, iterationKey);
            }
        }

        private class OuterCleanupCallback
            extends FilterFutureCallback<Object>
        {
            private final ProxySession session;
            private final DiskIndexerBatch batch;

            OuterCleanupCallback(
                final FutureCallback<Object> callback,
                final ProxySession session,
                final DiskIndexerBatch batch)
            {
                super(callback);
                this.session = session;
                this.batch = batch;
            }

            @Override
            public void completed(final Object result) {
                session.logger().info("Sending outer cleanup request");
                try {
                    sendBatch(batch, callback);
                } catch (BadRequestException | IOException e) {
                    super.failed(e);
                }
            }
        }
    }

    private static class PrefixAppender implements QueryPostProcessor {
        private static final long serialVersionUID = 0L;

        private final long uid;
        private final long version;

        PrefixAppender(final long uid, final long version) {
            this.uid = uid;
            this.version = version;
        }

        @Override
        public QueryConstructor apply(final QueryConstructor query) {
            query.append("prefix", uid);
            query.append("version", version);
            return query;
        }
    }

    private static class CleanupTypeAppender implements QueryPostProcessor {
        private static final long serialVersionUID = 0L;

        private final String cleanupType;

        CleanupTypeAppender(final String cleanupType) {
            this.cleanupType = cleanupType;
        }

        @Override
        public QueryConstructor apply(final QueryConstructor query)
            throws BadRequestException
        {
            query.append("cleanup-type", cleanupType);
            return query;
        }
    }

    private static class CallbackAppender implements QueryPostProcessor {
        private static final long serialVersionUID = 0L;

        private final URI callback;

        CallbackAppender(final URI callback) {
            this.callback = callback;
        }

        @Override
        public QueryConstructor apply(final QueryConstructor query)
            throws BadRequestException
        {
            query.append("callback", callback.toString());
            return query;
        }
    }
}

