package ru.yandex.search.disk.indexer;

import java.io.IOException;
import java.util.ArrayList;

import org.apache.http.entity.ContentType;
import org.apache.http.entity.mime.FormBodyPartBuilder;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.entity.mime.content.StringBody;

import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.parser.uri.QueryPostProcessor;
import ru.yandex.util.string.StringUtils;

public class DiskIndexerMultiRequestBatch
    extends ArrayList<DiskIndexerBatchEntry>
    implements DiskIndexerBatch
{
    private static final long serialVersionUID = 0L;

    private static final StringBody EMPTY_BODY =
        new StringBody("", ContentType.TEXT_PLAIN);

    private final DiskIndexerBatchEntryFactory batchEntryFactory;
    private final QueryPostProcessor queryPostProcessor;
    private final int batchSize;
    private DiskIndexerBatchEntry currentBatchEntry = null;
    private DiskDocumentMeta firstDoc = null;
    private DiskDocumentMeta lastDoc = null;

    public DiskIndexerMultiRequestBatch(
        final DiskIndexerBatchEntryFactory batchEntryFactory,
        final QueryPostProcessor queryPostProcessor,
        final int batchSize)
    {
        this.batchEntryFactory = batchEntryFactory;
        this.queryPostProcessor = queryPostProcessor;
        this.batchSize = batchSize;
    }

    private void initCurrentBatchEntry() {
        currentBatchEntry = batchEntryFactory.createBatchEntry();
        add(currentBatchEntry);
    }

    @Override
    public boolean add(final DiskDocumentMeta doc) {
        if (firstDoc == null) {
            firstDoc = doc;
            initCurrentBatchEntry();
        } else if (currentBatchEntry.isFull()) {
            initCurrentBatchEntry();
        }
        lastDoc = doc;
        return currentBatchEntry.add(doc);
    }

    @Override
    public boolean isFull() {
        return size() >= batchSize && currentBatchEntry.isFull();
    }

    @Override
    public BasicAsyncRequestProducerGenerator createRequest(
        final DiskReindexContext context)
        throws BadRequestException, IOException
    {
        if (size() == 1) {
            return DiskIndexerSingleRequestBatch.createRequest(
                context,
                get(0),
                queryPostProcessor);
        }
        QueryConstructor query = new QueryConstructor("/reindex?", false);
        query.append("batch-size", size());
        query.append("uid", context.uid());
        query.append(
            "resource-ids",
            StringUtils.concat(
                firstDoc.resourceId(),
                '-',
                lastDoc.resourceId()));
        MultipartEntityBuilder builder = MultipartEntityBuilder.create();
        builder.setMimeSubtype("mixed");
        for (DiskIndexerBatchEntry batchEntry: this) {
            String body = batchEntry.body();
            StringBody stringBody;
            if (body.isEmpty()) {
                stringBody = EMPTY_BODY;
            } else {
                stringBody =
                    new StringBody(body, ContentType.APPLICATION_JSON);
            }
            builder.addPart(
                FormBodyPartBuilder.create()
                    .addField(
                        YandexHeaders.ZOO_SHARD_ID,
                        context.shard())
                    .addField(
                        YandexHeaders.URI,
                        queryPostProcessor.apply(batchEntry.uri(context))
                            .toString())
                    .setBody(stringBody)
                    .setName("name")
                    .build());
        }
        return new BasicAsyncRequestProducerGenerator(
            query.toString(),
            builder.build());
    }
}

