package ru.yandex.search.disk.indexer;

import java.net.URI;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.function.Predicate;

import ru.yandex.function.TruePredicate;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.string.PositiveIntegerValidator;
import ru.yandex.parser.string.URIParser;
import ru.yandex.parser.uri.CgiParams;

public class DiskReindexContext {
    private static final long SHARDS = 65534L;

    private final ProxySession session;
    private final long uid;
    private final String uri;
    private final String serviceName;
    private final int batchSize;
    private final int docsPerRequest;
    private final Predicate<String> mediatypeFilter;
    private final String shard;
    private final URI callback;
    private final URI checkDuplicatesCallback;
    private final boolean cleanup;

    public DiskReindexContext(final ProxySession session)
        throws BadRequestException
    {
        this.session = session;
        CgiParams params = session.params();
        uid = params.getLong("uid");
        URI uri = params.get("uri", URIParser.FRAGMENT_STRIPPING);
        if (uri.getRawQuery() == null) {
            this.uri = uri.toASCIIString() + '?';
        } else {
            this.uri = uri.toASCIIString();
        }
        serviceName = params.get("service-name", NonEmptyValidator.INSTANCE);
        batchSize =
            params.get("batch-size", 1, PositiveIntegerValidator.INSTANCE);
        docsPerRequest = params.get(
            "docs-per-request",
            1,
            PositiveIntegerValidator.INSTANCE);
        List<String> mediatypes = params.getAll("mediatype");
        switch (mediatypes.size()) {
            case 0:
                mediatypeFilter = TruePredicate.instance();
                break;
            case 1:
                String mediatype = mediatypes.get(0);
                mediatypeFilter = x -> mediatype.equals(x);
                break;
            default:
                Set<String> types = new HashSet<>(mediatypes);
                mediatypeFilter = x -> types.contains(x);
                break;
        }
        shard = Long.toString(uid % SHARDS);
        callback = params.get("callback", null, URIParser.FRAGMENT_STRIPPING);
        // checking duplicates for res Id
        checkDuplicatesCallback = params.get("check-dup-callback", null, URIParser.FRAGMENT_STRIPPING);
        cleanup = params.getBoolean("cleanup", false);
        if (cleanup) {
            if (docsPerRequest < 2) {
                throw new BadRequestException(
                    "At least two docs per request required "
                    + "for cleanup requests");
            }
            if (!mediatypes.isEmpty()) {
                throw new BadRequestException(
                    "Mediatype filter not allowed for cleanup requests");
            }
        } else if (callback != null) {
            throw new BadRequestException(
                "Callback can be used only for cleanup requests");
        }
    }

    public URI checkDuplicatesCallback() {
        return checkDuplicatesCallback;
    }

    public boolean checkDuplicates() {
        return checkDuplicatesCallback != null;
    }

    public ProxySession session() {
        return session;
    }

    public long uid() {
        return uid;
    }

    public String uri() {
        return uri;
    }

    public String serviceName() {
        return serviceName;
    }

    public int batchSize() {
        return batchSize;
    }

    public int docsPerRequest() {
        return docsPerRequest;
    }

    public Predicate<String> mediatypeFilter() {
        return mediatypeFilter;
    }

    public String shard() {
        return shard;
    }

    public URI callback() {
        return callback;
    }

    public boolean cleanup() {
        return cleanup;
    }
}

