package ru.yandex.search.disk.indexer;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.apache.http.HttpEntity;
import org.apache.http.HttpException;
import org.apache.http.protocol.HttpContext;

import ru.yandex.function.BasicGenericConsumer;
import ru.yandex.function.GenericConsumer;
import ru.yandex.json.async.consumer.JsonAsyncConsumer;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.TypesafePopValueContentHandler;
import ru.yandex.json.parser.ContentHandler;
import ru.yandex.json.parser.ExceptionHandlingContentHandler;
import ru.yandex.json.parser.InterningStringCollector;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.SparseStringCollector;
import ru.yandex.json.parser.StackContentHandler;
import ru.yandex.json.parser.StringCollectors;
import ru.yandex.json.xpath.AnyIndexMatcher;
import ru.yandex.json.xpath.CompositeHandlerResolver;
import ru.yandex.json.xpath.ListHandlerResolver;
import ru.yandex.json.xpath.NameMatcher;
import ru.yandex.json.xpath.XPathStackContentHandler;

public class DiskSnapshotConsumer extends JsonAsyncConsumer<DiskSnapshot> {
    private final List<DiskDocumentMeta> docs;
    private final BasicGenericConsumer<JsonObject, JsonException> iterationKey;
    private final BasicGenericConsumer<JsonObject, JsonException> revision;

    public DiskSnapshotConsumer(final HttpEntity entity, final boolean parsePath)
        throws HttpException
    {
        this(
            entity,
            new ArrayList<>(),
            new BasicGenericConsumer<>(),
            new BasicGenericConsumer<>(),
            parsePath);
    }

    // CSOFF: ParameterNumber
    private DiskSnapshotConsumer(
        final HttpEntity entity,
        final List<DiskDocumentMeta> docs,
        final BasicGenericConsumer<JsonObject, JsonException> iterationKey,
        final BasicGenericConsumer<JsonObject, JsonException> revision,
        final boolean parsePath)
        throws HttpException
    {
        super(entity, createContentHandler(docs, parsePath, iterationKey, revision));
        this.docs = docs;
        this.iterationKey = iterationKey;
        this.revision = revision;
    }
    // CSON: ParameterNumber

    private static ContentHandler createContentHandler(
        final List<DiskDocumentMeta> docs,
        final boolean parsePath,
        final BasicGenericConsumer<JsonObject, JsonException> iterationKey,
        final BasicGenericConsumer<JsonObject, JsonException> revision)
    {
        StackContentHandler stackContentHandler = new StackContentHandler();
        InterningStringCollector keyCollector = new InterningStringCollector();
        SparseStringCollector valueCollector = new SparseStringCollector();
        StringCollectors stringCollectors =
            new StringCollectors(keyCollector, valueCollector);
        TypesafePopValueContentHandler docsHandler =
            new TypesafePopValueContentHandler(
                new DocConsumer(docs, parsePath),
                stringCollectors);
        docsHandler.stackContentHandler(stackContentHandler);
        TypesafePopValueContentHandler iterationKeyHandler =
            new TypesafePopValueContentHandler(iterationKey, stringCollectors);
        iterationKeyHandler.stackContentHandler(stackContentHandler);
        TypesafePopValueContentHandler revisionHandler =
            new TypesafePopValueContentHandler(revision, stringCollectors);
        revisionHandler.stackContentHandler(stackContentHandler);
        XPathStackContentHandler xpathContentHandler =
            new XPathStackContentHandler(
                new CompositeHandlerResolver<>(
                    Arrays.asList(
                        new ListHandlerResolver<>(
                            docsHandler,
                            new NameMatcher("items"),
                            AnyIndexMatcher.INSTANCE),
                        new ListHandlerResolver<>(
                            iterationKeyHandler,
                            new NameMatcher("iteration_key")),
                        new ListHandlerResolver<>(
                            revisionHandler,
                            new NameMatcher("revision")))),
                stackContentHandler,
                keyCollector);
        stackContentHandler.push(xpathContentHandler);
        return new ExceptionHandlingContentHandler(
            new ExceptionHandlingContentHandler(
                stackContentHandler,
                stackContentHandler),
            xpathContentHandler);
    }

    @Override
    protected DiskSnapshot doBuildResult(final HttpContext context)
        throws JsonException
    {
        return new DiskSnapshot(
            docs,
            JsonMap.maskNull(iterationKey.get()).asStringOrNull(),
            JsonMap.maskNull(revision.get()).asLong());
    }

    private static class DocConsumer
        implements GenericConsumer<JsonObject, JsonException>
    {
        private final List<DiskDocumentMeta> docs;
        private final boolean parsePath;

        DocConsumer(final List<DiskDocumentMeta> docs, final boolean parsePath) {
            this.docs = docs;
            this.parsePath = parsePath;
        }

        @Override
        public void accept(final JsonObject doc) throws JsonException {
            docs.add(new DiskDocumentMeta(doc, parsePath));
        }
    }
}

